proc(b,ctrl) = calibrLS(T,Sig,H,F,Q,R,e,N,eps,itmax,aus)
; -----------------------------------------------------------------
; Library        kalman
; -----------------------------------------------------------------
;  See_also      rICfil
; -----------------------------------------------------------------
;   Macro        calibrLS
; -----------------------------------------------------------------
;   Description  Auxiliary routine for rLSfil
;                Calibrates the robust LS- Filter for a given State Space model
;                to a given relative efficiency loss in terms of the MSE in the
;                ideal model.
;                The state-space model is assumed to be in the
;                following form:
;
;                y_t = H x_t + v_t
;
;                x_t = F x_t-1 + w_t
;
;                x_0 ~ (mu,Sig), v_t ~ (0,Q), w_t ~ (0,R)
;
;                All parameters are assumed known.
;
; -----------------------------------------------------------------
;   Keywords     robust Kalman filter, Kalman filter, Kalman smoother, filter
; -----------------------------------------------------------------
;   Usage         {b,ctrl}=calibrLS(T,Sig,H,F,Q,R,e,N,eps,itmax,aus)
;   Input
;     Parameter   T  
;     Definition      number of observations/states to be filtered by rIC
;     Parameter   Sig    
;     Definition      n x n covariance matrix of the initial state
;     Parameter   H    
;     Definition      m x n matrix
;     Parameter   F    
;     Definition      n x n matrix
;     Parameter   Q   
;     Definition      m x m variance-covariance matrix
;     Parameter   R
;     Definition      n x n variance-covariance matrix 
;     Parameter   e
;     Definition     numeric; efficiency loss to attain
;     Parameter    N
;     Definition      integer; MC-sample size / integration grid-points
;     Parameter    eps
;     Definition      numeric; exactitude
;     Parameter    itmax
;     Definition      integer; maximal number of Iterations
;     Parameter    aus
;     Definition      integer; 0: no output during execution, 1: some output, 2: more output, 3: a lot of output
;   Output                                                           
;     Parameter    b
;     Definition      T vector of clipping heights
;     Parameter    ctrl
;     Definition      T integer: tells if everything went well in step 1.. T (1 - 0 else)
; -----------------------------------------------------------------
;   Author    P.Ruckdeschel 991010 
; -----------------------------------------------------------------
;   Example   to be looked up in rlsfil
; -----------------------------------------------------------------
; Notes
;      Calles [within kalman.lib]  rlsbnorm, rlsbnorm1.
; 
;      The Output produced if aus>=1 is not identical with the output parameters.
;      (aus>=1) is for interactive use ;   success is controlled by "verbal" output.
;      (aus==0) is for other quantlets calling calibrLS;
;                       success is controlled by variable "ctrl".
; -----------------------------------------------------------------


  ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;preparations
  ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
  dimX   = rows(F)                                ;get dimension of state vector
  dimY = rows(Q)                              ;get dimension of observation vector
;;;;;;;;;;;;;;;;;;;;;;;;error checks;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
  Rs=svd(R)
  Qs=svd(Q)
  Sigs=svd(Sig)
  ;
  error((sum (Rs.l<0) > 0 )||(R<>R')||(cols(R) <> rows(R)), "Matrix R is not a Covariance.")
  error((sum (Sigs.l<0) > 0)||(Sig<>Sig')||(cols(Sig) <> rows(Sig)) , "Matrix Sig is not a Covariance.")
  error((sum (Qs.l<0) > 0)||(Q<>Q')||(cols(Q) <> rows(Q)) , "Matrix Q is not a Covariance..")
  error(cols(F) <> rows(F), "Matrix F is not square.")
  error(cols(R) <> rows(F), "Matrices F and R mismatch.")
  error(cols(R) <> cols(H), "Matrices R and H mismatch.")
  error(rows(Q) <> rows(H), "Matrices Q and H mismatch.")
  error(e<0, "wrong efficiency loss")
  error(((ceil(N)<>floor(N))|| (N<2) ), "wrong number of grid points / MC-Sample Size")
  error(((ceil(itmax)<>floor(itmax))|| (itmax<2)), "wrong maximal number of iterations")
  error(((ceil(aus)<>floor(aus))|| (aus<0) || (aus>5)), "wrong Output-Parameter (aus).")
  warning(e>0.5,"possibly IC does not exist.")

  b=matrix(T)*0
  ctrl=matrix(T)*0
;finding "good" starting values for b:
 b0=1.6*sqrt(max(vec(abs(Sig))))
 b0=b0+(b0<=0)
 

  ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
  ;start of the loop
  ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
    
  i=0
  j=1
  PreviousP = Sig                               ;set starting covariance matrix P_0|0 = Sigma
   
  del=eps+1
 
  while (((i<T)&&(del>eps^2))||(i<5))
      i=i+1
  ; Step of the Kalman recursion - the notation follows Brockwell&Davis
    
    pom = (F * PreviousP * F' + R)              ;temporary variable, P_(t|t-1) in fact
    
    if (aus>0)
          "Time point" i
    endif
       
    switch
          case (dimX==1)  ; numeric calibration
              erg=rlsbnorm1(e,pom,H,Q,N,eps,itmax)
          break 
          case (dimX>1)  ; stochastic calibration
              erg=rlsbnorm(e,pom,H,Q,N,eps,itmax)
          break 
          default
          break
    endsw
   ctrl[i]=erg.ctrl    
    
    if (erg.ctrl==0)
        if ((dimX>1)&& (del<eps))
; some smoothing to improove exactitude (MC-sim!)
           b[i]=(erg.b+j*b0)/(j+1)
           j=j+1
        else
          b[i]=erg.b
        endif
        b0=b[i]
    else
        b[i]=b0
    endif


    Delta = H * pom * H' + Q
    DeltaInv = ginv(Delta)
    CurrentP = pom - pom * H' * DeltaInv * H * pom  
    
; Step of the loop 
    del=max(abs(vec(CurrentP-PreviousP)))
    PreviousP = CurrentP 
    del=abs(b0-erg.b)
endo  
if (i<T)
     b[i+1:T]=b[i]
     ctrl[i+1:T]=ctrl[i]
endif
if (aus>0)
   b
   ctrl
endif
endp
