proc(hopt,ch)=denbwcrit(crit, x, h, K, d)
; -----------------------------------------------------------------
; Library        smoother
; -----------------------------------------------------------------
;  See_also      denbwsel denrot denest
; -----------------------------------------------------------------
;   Macro        denbwcrit
; -----------------------------------------------------------------
;   Keywords     kernel smoothing, kernel density estimation,
;                bandwidth selection
; -----------------------------------------------------------------
;   Description  determines the optimal from a range of bandwidths 
;                by one of the following bandwidth selection criteria:
;                Least Squares Cross Validation (lscv),
;                Biased Cross Validation (bcv),
;                Smoothed Cross Validation (scv),
;                Jones, Marron and Park Cross Validation (jmp),
;                Park and Marron Plug-in (pm),
;                Sheather and Jones Plug-in (sj), and
;                Silverman's rule of thumb.
; -----------------------------------------------------------------
;   Reference    Park and Turlach (1992): Practical performance of
;                several data driven bandwidth selectors
;
;                Haerdle (1991): Smoothing Techniques
;  
;                Haerdle, Mueller, Sperlich, Werwatz (1999): Non- and
;                Semiparametric Modelling
; -----------------------------------------------------------------
;   Usage        {hopt, ch} = denbwcrit(crit, x {,h {,K} {,d} })
;   Input
;     Parameter  crit  
;     Definition     string, criterion for bandwidth selection:
;                    "lscv", "bcv", "scv", "jmp", "pm", "sj".
;     Parameter  x  
;     Definition     n x 1 vector, the data.
;     Parameter  h  
;     Definition     m x 1 vector, vector of bandwidths.
;     Parameter  K  
;     Definition     string, kernel function on [-1,1] or Gaussian
;                    kernel "gau". If not given, "gau" is used.
;     Parameter  d  
;     Definition     scalar, discretization binwidth. d must be smaller
;                    than h. If not given, the minimum of min(h)/3 and 
;                    (max(x)-min(x))/200 is used.
;   Output
;     Parameter  hopt  
;     Definition     scalar, optimal bandwidth. (If negative in
;                    case of "pm" or "sj", denbwcrit needs to be run
;                    again with different h range.)
;     Parameter  ch  
;     Definition     m x 2 vector, the criterion function for h values.
; -----------------------------------------------------------------
;   Notes     Internally, the Gaussian kernel is used for all
;             computations. If a different kernel function K is
;             given as input, the asymptotical equivalent
;             bandwidth for K is calculated via the canonical 
;             kernel approach.
; -----------------------------------------------------------------
;   Example   library("smoother")
;             x=normal(500)
;             h=grid(0.05,0.1,10)
;             {hopt,ch}=denbwcrit("lscv",x,h)
;             hopt
;             library("plot")
;             ch=setmask(ch,"line","blue")
;             plot(ch)
;             setgopt(plotdisplay,1,1,"title",string("hopt=%1.6g",hopt))
; -----------------------------------------------------------------
;   Result    hopt is the LSCV optimal bandwidth for these data.
;             The resulting curve for the LSCV criterion is plotted.
; -----------------------------------------------------------------
;   Author    Marco Bianchi & Sigbert Klinke, 930722;
;             Lijian Yang, 961026; Marlene Mueller 990413
; ----------------------------------------------------------------- 
;
  error(cols(x)<>1, "denbwcrit: cols(x) <> 1")
  lambda = diff(quantile(x,0.25|0.75))
;
  if (exist(K)==0)
    K="gau"
  endif
  if (exist(K)==1) ; no K but d
    d=K
    K="gau"
  endif
;
  if (K!="gau")
    h=canker(h,K,"gau")
  endif
;
  if (exist(d)==0)
    d=(max(x)-min(x))./200
    d=min(d~(min(h)./3),2)
  endif
;
  {xb, yb} = bindata(x, d, 0) ; bin and sort the data
  xb = sort(xb~yb)
  yb = xb[,2]
  xb = xb[,1]
  n = sum(yb)
  m = rows(h)
  rng = max(xb)-min(xb)+1     ; binning is a function of the
  wx = aseq(0, rng, 1)        ; data set  rng=range
  wy = wx.*d
;
  switch 
; -----------------------------------------------------------
    case (crit=="lscv") 
; -----------------------------------------------------------
;     LSCV = Least Squares Cross Validation
; -----------------------------------------------------------
      wy1=gauder(wy, sqrt(2).*h', 0) ; standard normal rescaled by sqrt(2)*h 
      wy2=gauder(wy, h', 0)          ; standard normal rescaled by h 
      r1=matrix(rows(xb)) 
      r2=matrix(rows(xb)) 
;     
;     For any fixed bandwidth, calculate sum_i sum_j K_h*K_h(x_i-X_j) 
;     and sum_i sum_j K_h(x_i-X_j) 
;     
      i=1 
      while(i<=m) 
        {xc, yc, or} = conv(xb, yb, wx, wy1[,i]) 
        nrow = rows(paf(yc,or))   
        r1 = r1[1:nrow,]~paf(yc, or)  
        {xc, yc, or} = conv(xb, yb, wx, wy2[,i]) 
        nrow = rows(paf(yc,or))   
        r2 = r2[1:nrow,]~paf(yc, or)          
        i=i+1   
      endo   
;
      r1 = r1[,2:cols(r1)]   
      r2 = r2[,2:cols(r2)]  
      yb = paf(yb, yb) 
      r1 = (r1')*yb-n./(2*sqrt(pi)*h)   
      r2 = (r2')*yb-n./(sqrt(2.*pi)*h)   
;
      ch = ((r1-2.*r2)./(n-1) + 1./(2.*sqrt(pi).*h))./n 
      if (K!="gau")
        h=canker(h,"gau",K)
      endif
      ch = h~ch
      hopt = ch[minind(ch[,2]),1]
      break; LSCV
; -----------------------------------------------------------
    case (crit=="bcv") 
; -----------------------------------------------------------
;     LSCV = Biased Cross Validation
; -----------------------------------------------------------
      wy1=gauder(wy, sqrt(2).*h', 4)
      r=matrix(rows(xb))  
;
;     For any fixed bandwidth, calculate sum_i sum_j K''_h*K''_h(x_i-X_j)     
;
      i=1   
      while(i<=m)   
        {xc, yc, or} = conv(xb, yb, wx, wy1[,i])
        nrow = rows(paf(yc,or))   
        r = r[1:nrow,]~paf(yc, or)       
        i = i+1   
      endo   
      r  = r[,2:cols(r)]   
      diagterm=(3*n)./(8.*sqrt(pi).*h.^5) 
      yb = paf(yb, yb)  
      r  = (r')*yb-diagterm   
;
      ch = 1./(2.*sqrt(pi)*n.*h)+0.25*n^(-2)*r.*h^4   
      if (K!="gau")
        h=canker(h,"gau",K)
      endif
      ch = h~ch
      hopt = ch[minind(ch[,2]),1]
      break; BCV
; -----------------------------------------------------------
    case (crit=="scv") 
; -----------------------------------------------------------
;     SCV = Smoothed Cross Validation
; -----------------------------------------------------------
      g=sqrt(var(x))*(21/(40*sqrt(2)))^(1/13)*n^(-2/13)
      a=sqrt(2)*(h.^2+g.^2)^0.5    ; a,b,c are the rescale parameters 
      b=(h.^2+2*g.^2)^0.5
      c=sqrt(2)*g
;    
      wy1=gauder(wy, a', 0)
      wy2=gauder(wy, b', 0)
      wy3=gauder(wy, c', 0)
;    
      r1=matrix(rows(xb))
      r2=matrix(rows(xb))
      r3=matrix(rows(xb))
;    
      i=1
      while(i<=m)
        {xc, yc, or} = conv(xb, yb, wx, wy1[,i])
        nrow = rows(paf(yc,or))   
        r1 = r1[1:nrow,]~paf(yc, or) 
        {xc, yc, or} = conv(xb, yb, wx, wy2[,i])
        nrow = rows(paf(yc,or))   
        r2 = r2[1:nrow,]~paf(yc, or) 
        i=i+1
      endo
;    
      {xc, yc, or} = conv(xb, yb, wx, wy3)
      r3 = paf(yc, or)
      yb = paf(yb, yb)
;
      r1 = r1[,2:cols(r1)]
      r2 = r2[,2:cols(r2)]
      r1 = (r1')*yb-n./(a*sqrt(2.*pi))
      r2 = (r2')*yb-n./(b*sqrt(2.*pi))
      r3 = (r3')*yb-n./(c*sqrt(2.*pi)) 
;
      ch = 1./(2*sqrt(pi)*n.*h) + 1/(n*(n-1))*(r1-2*r2+r3) 
      if (K!="gau")
        h=canker(h,"gau",K)
      endif
      ch = h~ch
      hopt = ch[minind(ch[,2]),1]
      break; SCV
; -----------------------------------------------------------
    case (crit=="jmp") 
; -----------------------------------------------------------
;     JMP = Jones, Marron and Park Cross Validation
; -----------------------------------------------------------
;     Here, g is a function of h. So we need a little bit of computing. 
;     We have to estimate R(f'''') and R(f'') through convolutions. 
; 
      sd=sqrt(var(x)) 
      aa=sd*(128/(9*sqrt(2)*n))^(1/11) 
      bb=sd*(32/(5*sqrt(2)*n))^(1/7) 
      wyF1=gauder(wy, sqrt(2)*aa, 8) 
      wyF2=gauder(wy, sqrt(2)*bb, 4) 
      F1=matrix(rows(xb)) 
      F2=matrix(rows(xb)) 
; 
      {xc, yc, or} = conv(xb, yb, wx, wyF1) 
      F1=paf(yc, or) 
      {xc, yc, or} = conv(xb, yb, wx, wyF2) 
      F2=paf(yc, or) 
      F1=(F1')*paf(yb, yb) 
      F2=(F2')*paf(yb, yb) 
      F1=(2./n.*F1+105)./n             
      F2=(2./n.*F2+3)./n
      C3=(21/(8.*sqrt(pi).*F1))^(1./9) .* (1./(2.*sqrt(pi).*F2))^(2./5) .* aa .* bb
;    
      g=C3*n.^(-23/45)*h.^(-2)
;    
      a=sqrt(2).*(h.^2+g.^2)^0.5     ; a,b,c are the rescale parameters ;
      b=(h.^2+2.*g.^2)^0.5
      c=sqrt(2).*g   
      wy1=gauder(wy, a', 0)   
      wy2=gauder(wy, b', 0)   
      wy3=gauder(wy, c', 0)   
;       
      r1=matrix(rows(xb))   
      r2=matrix(rows(xb))   
      r3=matrix(rows(xb))   
;       
      i=1   
      while(i<=m)      
        {xc, yc, or} = conv(xb, yb, wx, wy1[,i])
        nrow = rows(paf(yc,or))   
        r1 = r1[1:nrow,]~paf(yc, or)       
        {xc, yc, or} = conv(xb, yb, wx, wy2[,i])   
        nrow = rows(paf(yc,or))   
        r2 = r2[1:nrow,]~paf(yc, or)    
        {xc, yc, or} = conv(xb, yb, wx, wy3[,i])   
        nrow = rows(paf(yc,or))   
        r3 = r3[1:nrow,]~paf(yc, or)      
        i=i+1   
      endo   
      r1 = r1[,2:cols(r1)]   
      r2 = r2[,2:cols(r2)]   
      r3 = r3[,2:cols(r3)]
      yb = paf(yb, yb)   
      r1 = (r1')*yb   
      r2 = (r2')*yb   
      r3 = (r3')*yb  
;
      ch = 1./(sqrt(2*pi)*n*h) + 1/(n*(n-1))*(r1-2*r2+r3)   
      if (K!="gau")
        h=canker(h,"gau",K)
      endif
      ch = h~ch
      hopt = ch[minind(ch[,2]),1]
      break; JMP
; -----------------------------------------------------------
    case (crit=="pm") 
; -----------------------------------------------------------
;     pm = Park and Marron Plug-in
; -----------------------------------------------------------
      m = rows(h)   
      g=(189./(640.*sqrt(2)))^(1./13).*lambda^(3./13)    
      alpha=g.*h.^(10./13)    
      a=sqrt(2).*alpha    
;        
      wy1=gauder(wy, a', 4)    
      r1=matrix(rows(xb))    
;        
      i=1    
      while(i<=m)    
;        
        {xc, yc, or} = conv(xb, yb, wx, wy1[,i])    
        nrow = rows(paf(yc,or))   
        r1 = r1[1:nrow,]~paf(yc, or)       
        i=i+1    
      endo    
;        
      r1 = r1[,2:cols(r1)]       
      r1 = n^(-2).*((r1')*paf(yb, yb)-(3*n)./(sqrt(2.*pi).*(sqrt(2)*a)^5))       
;    
      ch = (1/(2*sqrt(pi)))^(0.2)*(r1*n)^(-0.2)-h      
      if (K!="gau")
        h=canker(h,"gau",K)
      endif
      ch = h~ch
; 
      ppp=(ch[,2].> 0)        ; Check that the score function has a zero
      ppp=sum(ppp)    
;        
      if ( ppp.*(ppp-rows(ch)) .= 0)   ; if this is true there is no zero 
        hopt=-1
      else
        j=minind(ch[,2]>0)-1
        x1=ch[j,1]
        x2=ch[j+1,1]
        y1=ch[j,2]
        y2=ch[j+1,2]
        hopt=x1+( y1.*(x1-x2) )./ (y2-y1)  ; interpolation formula
      endif
      break; PM
; -----------------------------------------------------------
    case (crit=="sj") 
; -----------------------------------------------------------
;     SJ = Sheather and Jones Plug-in
; -----------------------------------------------------------
      a=0.92*lambda*n^(-1/7) 
      b=0.912*lambda*n^(-1/9) 
;     
      wy=wx.*d 
      wy1=gauder(wy, a, 4) 
      wy2=gauder(wy, b, 6)      ; Note: here we don't need to loop 
                            ;       In fact, a and b are scalars 
;     
      {xc, yc, or} = conv(xb, yb, wx, wy1)    
      r1 = paf(yc, or)       
;    
      {xc, yc, or} = conv(xb, yb, wx, wy2)       
      r2 = paf(yc, or) 
;      
      frac1 =  1./(n.*(n-1)) 
      frac2 = -1./(n.*(n-1)) 
      r1 = (r1')*paf(yb, yb).*frac1 
      r2 = (r2')*paf(yb, yb).*frac2      
;          
      r12=r1./r2      
      alpha=(6.*sqrt(2))^(1./7) .* r12^(1./7) .* h^(5./7)      
      wy3=gauder(wy, alpha', 4)  
      r3=matrix(rows(xb))  
;      
      i=1                                       ; Note: here we need to loop  
      while(i<=m)                               ;       cause alpha is a vector
        {xc, yc, or} = conv(xb, yb, wx, wy3[,i])    
        nrow = rows(paf(yc,or))   
        r3 = r3[1:nrow,]~paf(yc, or)       
        i=i+1  
      endo  
;      
      r3 = r3[,2:cols(r3)]  
      frac3 = 1./(n.*(n-1))  
      r3 =(r3')*paf(yb, yb).*frac3  
      ch = (1./(2.*sqrt(pi)))^(0.2).*r3^(-0.2).*n^(-0.2)-h        
      if (K!="gau")
        h=canker(h,"gau",K)
      endif
      ch = h~ch
; 
      ppp=(ch[,2].> 0)        ; Check that the score function has a zero
      ppp=sum(ppp)    
;        
      if ( ppp.*(ppp-rows(ch)) .= 0)   ; if this is true there is no zero 
        hopt=-1
      else
        j=minind(ch[,2]>0)-1
        x1=ch[j,1]
        x2=ch[j+1,1]
        y1=ch[j,2]
        y2=ch[j+1,2]
        hopt=x1+( y1.*(x1-x2) )./ (y2-y1)  ; interpolation formula
      endif
      break; PM
; -----------------------------------------------------------
    default
; -----------------------------------------------------------
;     no such criterion!
; -----------------------------------------------------------
      error(1,"denbwcrit: Criterion "+crit+" does not exist!")
      break
  endsw
endp



