proc(fh)=denestp(x,h,K,d)
; -----------------------------------------------------------------
; Library        smoother
; -----------------------------------------------------------------
;  See_also      denest 
; -----------------------------------------------------------------
;   Macro        denestp
; -----------------------------------------------------------------
;   Keywords     kernel smoothing, kernel density estimation
; -----------------------------------------------------------------
;   Description  estimates a p-dimensional density by kernel density
;                estimation. The computation uses WARPing.
; -----------------------------------------------------------------
;   References   Scott (1992): Multivariate Kernel Density Estimation
;                Wand/Jones (1995): Kernel Smoothing
;                Haerdle/Mueller (1999): Multivariate and
;                                    Semiparametric Kernel Regression
; -----------------------------------------------------------------
;   Usage        fh = denestp(x {,h {,K} {,d} })
;   Input
;     Parameter  x  
;     Definition   n x p matrix, the data.
;     Parameter  h  
;     Definition   scalar or p x 1 vector, bandwidth. If not given, 
;                  the rule of thumb bandwidth computed by denrotp
;                  is used (Scott's rule of thumb).
;     Parameter  K  
;     Definition   string, kernel function on [-1,1]^p. If not given,
;                  the product Quartic kernel "qua" is used.
;     Parameter  d  
;                  scalar, discretization binwidth. d[i] must be 
;                  smaller than h[i]. If not given, the minimum of h/3 
;                  and (max(x)-min(x))'/r, with r=100 for p=1, and 
;                  r=(1000^(1/p)) for p>1 is used.
;   Output
;     Parameter  fh  
;     Definition   m x (p+1) matrix, the first p columns constitute 
;                  a grid and the last column contains the density 
;                  estimate on that grid.
; -----------------------------------------------------------------
;   Example   library("smoother")                                       
;             library("plot")
;             ;
;             x  = read("geyser")     ; read data         
;             fh = denestp(x)         ; estimate density
;             ;
;             fh = setmask(fh,"surface")
;             plot(fh)                ; graph density estimate
;             setgopt(plotdisplay,1,1,"title","ROTATE!")
; -----------------------------------------------------------------
;   Result    The kernel density estimate for the Geyser data is    
;             computed using the Quartic kernel and bandwidth
;             according to Scott's rule of thumb (default).
;             The display shows the surface of the resulting 
;             function.
; -----------------------------------------------------------------
;   Example   library("smoother")                                       
;             library("plot")
;             ;
;             x  = read("bank2.dat")            ; read data                    
;             x  = x[,4:6]                      ; columns 4 to 6
;             d  = (max(x)-min(x))'./7          ; large binwidth!         
;             fh = denestp(x,1.5,"qua",d)       ; estimate density
;             ;
;             c  = (max(fh[,4])-min(fh[,4])).*(1:4)./5 ; levels
;             cfh= grcontour3(fh,c,1:4)                ; contours
;             plot(cfh)                         ; graph contours
;             setgopt(plotdisplay,1,1,"title","ROTATE!")
; -----------------------------------------------------------------
;   Result    The kernel density estimate for the last three 
;             variables of the Swiss banknote data is computed 
;             using the Quartic kernel and bandwidth h=1.5.
;             The display shows a contour plot of the resulting 
;             function.
; -----------------------------------------------------------------
;   Author    Sigbert Klinke, 950225; Lijian Yang, 960504; 
;             Marlene Mueller, 990413
; -----------------------------------------------------------------
  n=rows(x)
  p=cols(x) 
  if (exist(h)==0)
    h=denrotp(x); h=2.6.*sqrt(var(x)').*(n^(-1/(p+4)))
  endif
  if (cols(h)>1)
    h=h'
  endif
  if (rows(h)==1)
    h = h.*matrix(p,1) 
  endif
;
  if (exist(K)==0)
    K="qua"
  endif
  if (exist(K)==1) ; no K but d
    d=K
    K="qua"
  endif
;
  if (exist(d)==0)
    r=100.*(p==1)+(1000^(1/p)).*(p>1)
    d=(max(x)-min(x))'/r
    d=min(d~(h./3),2)
  endif
  if (cols(d)>1)
    d=d'
  endif
  if (rows(d)==1)
    d = d.*matrix(p,1) 
  endif
;
  error(sum(h .<=d)>0,"denestp: h smaller than d -- Please modify h or d!")
;
  w1 = matrix(p,1)
  wx = w1-1
  {xb,yb} = bindata(x,d,wx) 
  m=max(xb)-min(xb)+1 
  error(prod(m).>10000, "denestp: can't create more than 10000 datapoints") 
  hd=h./d
  wy=symweigh(wx,d./h,hd,K)
  wx=grid(wx,w1,hd)
  {xc,yc,or} = conv(xb,yb,wx,wy)
  fh=((xc+0.5).*d')~(yc./(n*prod(d)))
  fh=sort(fh,1:p)
endp
