proc(fh)=denxestp(x,h,K,v)
; -----------------------------------------------------------------
; Library        smoother
; -----------------------------------------------------------------
;  See_also      denestp 
; -----------------------------------------------------------------
;   Macro        denxestp
; -----------------------------------------------------------------
;   Keywords     kernel smoothing, kernel density estimation
; -----------------------------------------------------------------
;   Description  estimates a p-dimensional density by kernel density
;                estimation. The computation uses WARPing.
; -----------------------------------------------------------------
;   Notes        This function does an exact computation, i.e.
;                requires O(n^2) operations for estimating the 
;                regression function on all observations. For
;                p>2 this is usually faster than the WARPing method.
; -----------------------------------------------------------------
;   References   Scott (1992): Multivariate Kernel Density Estimation,
;                Wand/Jones (1995): Kernel Smoothing,
;                Haerdle/Mueller (1999): Multivariate and
;                                    Semiparametric Kernel Regression
; -----------------------------------------------------------------
;   Usage        fh = denxestp(x {,h {,K} {,v} })
;   Input
;     Parameter  x  
;     Definition   n x p matrix, the data.
;     Parameter  h  
;     Definition   scalar or p x 1 vector, bandwidth. If not given, 
;                  the rule of thumb bandwidth computed by denrotp
;                  is used (Scott's rule of thumb).
;     Parameter  K  
;     Definition   string, kernel function on [-1,1]^p. If not given,
;                  the product Quartic kernel "qua" is used.
;     Parameter  v  
;     Definition   m x p, values of the independent variable on 
;                  which to compute the regression. If not given, 
;                  a grid of length 100 (p=1), length 30 (p=2)
;                  and length 8 (p=3) is used in case of p<4. When 
;                  p>=4 then v is set to x.
;   Output
;     Parameter  fh  
;     Definition   n x (p+1) or m x (p+1) matrix, the first p columns 
;                  contain the grid or the sorted  x[,1:p], the 
;                  second column contains the density estimate  
;                  on the values of the first p columns.
; -----------------------------------------------------------------
;   Example   library("smoother")                                       
;             library("plot")
;             ;
;             x  = read("geyser")     ; read data         
;             fh = denxestp(x)        ; estimate density
;             ;
;             fh = setmask(fh,"surface")
;             plot(fh)                ; graph density estimate
;             setgopt(plotdisplay,1,1,"title","ROTATE!")
; -----------------------------------------------------------------
;   Result    The kernel density estimate for the Geyser data is    
;             computed using the Quartic kernel and bandwidth
;             according to Scott's rule of thumb (default).
;             The display shows the surface of the resulting 
;             function.
; -----------------------------------------------------------------
;   Example   library("smoother")                                       
;             library("plot")
;             ;
;             x  = read("bank2.dat")            ; read data                    
;             x  = x[,4:6]                      ; columns 4 to 6
;             fh = denxestp(x,1.5,"qua")        ; estimate density
;             ;
;             c  = (max(fh[,4])-min(fh[,4])).*(1:4)./5 ; levels
;             cfh= grcontour3(fh,c,1:4)                ; contours
;             plot(cfh)                         ; graph contours
;             setgopt(plotdisplay,1,1,"title","ROTATE!")
; -----------------------------------------------------------------
;   Result    The kernel density estimate for the last three 
;             variables of the Swiss banknote data is computed 
;             using the Quartic kernel and bandwidth h=1.5. 
;             The display shows a contour plot of the resulting 
;             function.
; -----------------------------------------------------------------
;   Author    Marlene Mueller, 990413
; -----------------------------------------------------------------
  n=rows(x)
  p=cols(x) 
  ix=min(x[,1:p])
  mx=max(x[,1:p])
  rx=mx-ix
  one=matrix(p)
;
  if (exist(h)==0)
    h=denrotp(x); h=2.6.*sqrt(var(x)').*(n^(-1/(p+4)))
  endif
  if (cols(h)>1)
    h=h'
  endif
  if (rows(h)==1)
    h = h.*matrix(p,1) 
  endif
;
  if (exist(K)==0)
    K="qua"
  endif
  if (exist(K)==1) ; no K but v
    v=K
    K="qua"
  endif
;
  x=sort(x,1:p)
;
  if (exist(v)==0)
    switch
      case (p==1)
        v=grid(ix',rx'/99,100*one)
        break
      case (p==2)
        v=sort(grid(ix',rx'/29,30*one),1:p)
        break
      case (p==3)
        v=sort(grid(ix',rx'/7,8*one),1:p)
        break
      default
        v=x
        break
    endsw  
  else
    v=sort(v,1:p)
  endif
;
  if (K!="gau")
    tmp=sker(x,h',K,matrix(n),v)
  else
    tmp=sker(x,5.*h',"ngau",matrix(n),v)
  endif
  fh=v~tmp
endp
