proc()=doglm(x,y,opt)
; ----------------------------------------------------------------------------
; Library       glm
; ----------------------------------------------------------------------------
;  See_also     glmopt glmest glmcore glmstat glmout 
; ----------------------------------------------------------------------------
;   Macro       doglm
; ----------------------------------------------------------------------------
;   Description  doglm provides an interactive menu for preparing data
;                and fitting GLM in XploRe.
; ----------------------------------------------------------------------------
;   Keywords     GLM, Generalized Linear Model
; ----------------------------------------------------------------------------
;   Reference    McCullagh/Nelder, Generalized Linear Models, 1989
; ----------------------------------------------------------------------------
;   Link  ../tutorials/glmstart.html Tutorial: GLM in XploRe 
; ----------------------------------------------------------------------------
;   Usage        doglm(x,y{,opt})
;   Input
;     Parameter   x  
;     Definition        n x p  matrix, the predictor variables.
;     Parameter   y  
;     Definition        n x 1  vector, the response variables.
;                       (In the case of replicated data, the number of 
;                       replications should be given in opt.wx and y should 
;                       contain the sums of all responses for a replication.)
;     Parameter   opt
;     Definition        optional, a list with optional input. The macro
;                       "glmopt" can be used to set up this parameter.
;                       The order of the list elements is not important.
;                       Parameters which are not given are replaced by 
;                       defaults (see below).
;     Parameter   opt.code
;     Definition        text string, the short code for the model (e.g. 
;                       "bilo" for logit or "noid" for ordinary PLM).
;     Parameter   opt.weights
;     Definition        string, type of weights. Can be "frequency"
;                       for replication counts, or "prior" (default)
;                       for prior weights in weighted regression.
;     Parameter   opt.wx
;     Definition        scalar or n x 1 vector, frequency or prior 
;                       weights. If not given, set to 1.
;     Parameter   opt.off  
;     Definition        scalar or n x 1 vector, offset in linear predictor.
;                       If not given, set to 0.
;     Parameter   opt.name
;     Definition        string, prefix for the output. If not given, "glm"
;                       is used.
;     Parameter   opt.xvars
;     Definition        p x 1 string vector, variable names for the output. 
;                       Note, that only up to 15 characters are used.
;     Parameter   opt.title
;     Definition        string, title for the output. If not given, a default
;                       is set.
;     Parameter   opt.nopic
;     Definition        integer, if exists and =1, output is not shown
;                       in displays, but written to XploRe_out.
;   Output
;     Parameter   doglm  
;     Definition        or opt.name,
;                       list, global variable with the components:
;     Parameter   doglm.b
;     Definition        p x 1  vector, estimated coefficients.
;     Parameter   doglm.bv
;     Definition        p x p  matrix, estimated covariance matrix for b.
;     Parameter   doglm.stat
;     Definition        list with components as computed by glmstat:
;                       serror     (standard errors of coefficients), 
;                       tvalue     (t-values for coefficients), 
;                       pvalue     (p-values for coefficients), 
;                       df         (degrees of freedom),
;                       deviance   (deviance),
;                       pearson    (generalized pearson's chi^2),
;                       loglik     (log-likelihood),
;                       dispersion (estimated dispersion =pearson/df),
;                       r2         ((pseudo) coefficient of determination),
;                       adr2       (adjusted (pseudo) coefficient of determination),
;                       aic        (Akaike's AIC criterion),
;                       bic        (Schwarz' BIC criterion), and
;                       it         (number of iterations needed),
;                       ret        (return code, 
;                                     0 if everything went o.k., 
;                                     1 if maximal number of iterations reached, 
;                                    -1 if missing values have been encountered),
;                       nr         (number of replications found in x).
;     Parameter   doglmSummary  
;     Definition        or  opt.name+"Summary",
;                       display, containing summary statistics and
;                       show which variables are selected for fit.
;     Parameter   doglmsum  
;     Definition        or opt.name+"sum",
;                       string vector, global, containing summary 
;                       statistics as text.
;     Parameter   doglmOutput  
;     Definition        or  opt.name+"Output",
;                       display, containing estimation result in the left
;                       panel and a plot of the index x*b{+off} vs. y 
;                       (vs. y./wx for binomials) and a plot of the index
;                       vs. the link function.
;     Parameter   doglmout  
;     Definition        or opt.name+"out",
;                       string vector, global, containing estimation result 
;                       in the left panel as text.
; ----------------------------------------------------------------------------
;   Example   library("glm") 
;             n=100
;             b=-2|0|0|0|2
;             p=rows(b)
;             x=normal(n,p)-1
;             y=x*b+normal(n) >0
;             doglm(x,y)
; ----------------------------------------------------------------------------
;   Result    For the fit you should use "binomial" distribution and e.g.
;             "gaussian" link.
;             You may play around with different subsets of the explanatory
;             variables. 
; ----------------------------------------------------------------------------
;   Author    Marlene Mueller, 2000/05/18
; ----------------------------------------------------------------------------
;
; check initial errors
;
  havearray=(rows(dim(x))>2)
  error(havearray,"x must be vector or matrix")
  error(rows(x)!=rows(y),"x and y have different number of rows")
  error((rows(dim(y))>1),"y must be a vector")
;
  n=rows(x)
  p=cols(x)
;
  xvars=string("X%1.0f",1:cols(x))
  havexvars=0
;  
  haveopt=0
  if (exist(opt)==9)
    haveopt=1
  endif
;
; set the defaults
;
  wx=1
;  off=0
  shf=0
  norepl=0
  miter=10
  cnv=0.0001
  fscor=0
  pow=0
  nbk=1
  interc=1
  name="doglm"
  nopic=0
  title="none"
  code="none"
  weights="prior"
;
  glmmodels=getglobal("glmmodels")
;
  haveconst=0
  iconst=minind(abs(max(x)-min(x)),2)
  if (max(x[,iconst])-min(x[,iconst]) ==0)
    haveconst=1
    interc=0
  endif
;
  if (haveopt)
    if (comp(opt,"code")>0)
      notgood=(sum(opt.code==glmmodels.all)==0)
      warning(notgood>0, "opt.code was unusable, entered interactive mode")
      if (notgood==0)
        code=opt.code
      endif
    endif
    if (comp(opt,"wx")>0)
      notgood=(dim(dim(opt.wx))!=1)*(rows(opt.wx)!=n)+(rows(dim(opt.wx))!=1)
      notgood=notgood+(sum(abs(opt.wx))==0)+(min(opt.wx)<0)
      warning(notgood>0, "opt.wx was unusable, used opt.wx=1 instead")
      if (notgood==0)
        wx=opt.wx
      endif
    endif
    if (comp(opt,"weights")>0)
      notgood=(exist(opt.weights)!=2)
      error(notgood>0, "wrong type of weights in opt.weights!")
      if (notgood==0)
        weights=opt.weights
      endif
    endif
;    if (comp(opt,"off")>0)
;      notgood=(dim(dim(opt.off))!=1)*(rows(opt.off)!=n)+(rows(dim(opt.off))!=1)
;      warning(notgood>0, "opt.off not consistent with x, used off=1 instead")
;      if (notgood==0)
;        off=opt.off
;      endif
;    endif
    if (comp(opt,"shf")>0)
      shf=(opt.shf==1)*(dim(dim(opt.shf))==1)
    endif
    if (comp(opt,"fscor")>0)
      fscor=(opt.fscor==1)*(dim(dim(opt.fscor))==1)
    endif
    if (comp(opt,"norepl")>0)
      fscor=(opt.norepl==1)*(dim(dim(opt.norepl))==1)
    endif
    if (comp(opt,"pow")>0)
      notgood=(dim(dim(opt.pow))!=1)
      warning (notgood>0, "opt.pow was unusable, used default =0 instead")
      if (notgood==0)
        pow=opt.pow
      endif
    endif
    if (comp(opt,"nbk")>0)
      notgood=(dim(dim(opt.nbk))!=1)
      warning (notgood>0, "opt.nbk was unusable, used default =1 instead")
      if (notgood==0)
        nbk=opt.nbk
      endif
    endif
    if (comp(opt,"miter")>0)
      tmp=floor(opt.miter[1,1])
      notgood=(tmp<1)+(dim(dim(opt.miter))!=1)
      warning (notgood>0, "opt.miter was unusable, used default =10 instead")
      miter=notgood*miter+(1-notgood)*tmp
    endif
    if (comp(opt,"cnv")>0)
      tmp=opt.cnv[1,1]
      notgood=(tmp<=0)+(dim(dim(tmp))!=1)
      warning (notgood>0, "opt.cnv was unusable, used default =0.0001 instead")
      cnv=notgood*cnv+(1-notgood)*tmp
    endif
    if (comp(opt,"name")>0)
      notgood=(exist(opt.name)!=2) || (dim(dim(opt.name))!=1)
      warning (notgood>0, "opt.name not a single string, used default instead")
      if (notgood==0)
        name=opt.name
      endif
    endif
    if (comp(opt,"nopic")>0)
      notgood=(exist(opt.nopic)!=1)
      warning (notgood>0, "opt.nopic not a number, used default instead")
      if (notgood==0)
        nopic=(opt.nopic==1)*(dim(dim(opt.nopic))==1)
      endif
    endif
    if (comp(opt,"xvars")>0)
      notgood=(exist(opt.xvars)!=2) || (rows(dim(opt.xvars))!=1)
      notgood=notgood || (rows(opt.xvars)!=p)
      warning (notgood>0, "opt.xvars not consistent with x")
      if (notgood==0)
        xvars=opt.xvars
        havexvars=1
      endif
    endif
    if (comp(opt,"title")>0)
      notgood=(exist(opt.title)!=2) || (dim(dim(opt.title))!=1)
      warning (notgood>0, "opt.title not a single string, used default instead")
      if (notgood==0)
        title=opt.title
      endif
    endif
  else
    dummy=0
    opt=list(dummy)
  endif
;
  xorg=x       ; original x
  xall=x       ; all x (incl. created)
;
  xvars=xvars+"                    "
  xvars=substr(xvars,1,16)
;
  blank=string(" ",1:rows(xvars))
  xvarsnew=substr(xvars,1,2)
  j=1
  while (j<15)
    j=j+1
    sub=substr(xvars,j,j+1)
    if (1-prod(sub==blank)) 
      xvarsnew=xvarsnew+sub
    endif
  endo
  xvars=xvarsnew
;
  xorgvars=xvars
  xallvars=xvars
;
  jall =1:cols(x)
  jset =1:cols(x)
;
  havetrans=0
;
  finish=0
  while (finish==0)
    stmp="GLM main menu -- Select item"
    atmp="GLM main menu -- select ONE item"
    itmp=       " Descriptive statistics"
    itmp=itmp|  " Select variables"
    itmp=itmp|  " Transform variables"
    if (havetrans==1)
      itmp=itmp|" Reset variables"
    endif
    itmp=itmp|  " Marginal scatterplots"
    itmp=itmp|  " Do GLM fit"
    itmp=itmp|  " Change default settings"
    itmp=itmp|  " Stop"
    nummain=select1item(stmp,itmp,atmp)
;
    descdo  =(nummain==1)
    vardo   =(nummain==2)
    transdo =(nummain==3)
    if (havetrans)
      resetdo =(nummain==4)
      scattdo =(nummain==5)
      glmdo   =(nummain==6)
      optdo   =(nummain==7)
    else
      resetdo =0
      scattdo =(nummain==4)
      glmdo   =(nummain==5)
      optdo   =(nummain==6)
    endif
;
    switch
      case (descdo)
;       ===============================
;        Do descriptive statistics
;       ===============================
        jall=1:cols(xall)
        jstar=(min(abs(jall-jset'),2)==0)
        stars=string(" *",jstar)
        stars=substr(stars,jstar+1,jstar+2)
;
        descx=summarize(xall,string(" ",1:cols(xall)))
        descx=paf(descx,0|0|matrix(rows(descx)-2))
        descy=summarize(y," ")
        descy=paf(descy,0|matrix(rows(descy)-2)|0)
;
        desc= " "|"Descriptive Statistics"
        desc=desc|"----------------------"|" "
        txt="n="+string("%1.0f",rows(x))
        txt=txt+", "+string("%1.0f",cols(xall))+" variables"
        txt=txt+" ("+string("%1.0f",rows(jset))+" selected)"
        desc=desc|txt|" "
;
        vars=" "|" "|"Y"|" "|xallvars
        vars=vars+"                       "
        vars=substr(vars,1,19)
;
        blank=string(" ",1:rows(vars))
        varsnew=substr(vars,1,2)
        j=1
        while (j<15)
          j=j+1
          sub=substr(vars,j,j+1)
          if (1-prod(sub==blank)) 
            varsnew=varsnew+sub
          endif
        endo
        vars=varsnew
;
        txt=(" "|" "|" "|" "|stars)+vars+(descy|descx)
        desc=desc|txt
        desc=desc|" "|" *Variables selected for fit"
        exec( name+"sum=desc" )
        putglobal(name+"sum")
        if (nopic)
          exec( name+"sum" )
        else
          exec( name+"Summary=createdisplay(1,1)" )
          exec( "show("+name+"Summary,1,1,desc)" )
        endif
        break
;
      case (vardo)
;       ===============================
;        Choose from X
;       ===============================
        selx="SELECT variables for estimation"
        itmx=(" All (default)")|(string(" ",1:cols(xall))+xallvars)|(" Cancel")
        numx=selectitem(selx,itmx)
        some=(sum(abs( numx[2:rows(numx)-1]-0.*matrix(cols(xall)) ))>0)
        all =(sum(numx)==0)|| numx[1]==1
        some=some && (1-all)
        switch
          case (some)
            jset=numx[2:rows(numx)-1].*(1:cols(xall))
            jset=paf(jset,jset>0)
            break
          case (all)
            jset=1:cols(xall)
            break
          default ; change nothing
        endsw
        break
;
      case (transdo)
;       ===============================
;        Transform X
;       ===============================
        selx="SELECT variables to transform"
        itmx=(" All (default)")|(string(" ",1:cols(xall))+xallvars)|(" Cancel")
        numx=selectitem(selx,itmx)
        some=(sum(abs( numx[2:rows(numx)-1]-0.*matrix(cols(xall)) ))>0)
;
        all  =(sum(numx)==0)|| numx[1]==1
        some =some && (1-all)
;
        havetransvars=0
        switch
          case (some)
            jtrans=numx[2:rows(numx)-1].*(1:cols(xall))
            jtrans=paf(jtrans,jtrans>0)
            havetransvars=1
            break
          case (all)
            jtrans=1:cols(xall)
            havetransvars=1
            break
          default ; (cancel) = do nothing
        endsw
;
        if (havetransvars)
          havetrans=1
          stmp="SELECT transformation"
          atmp="Select ONE transformation"
          itmp=      " Categorize (create dummies)"
          itmp=itmp|" Standardize"
          itmp=itmp|" Transform to [0,max]"
          itmp=itmp|" Transform to [0,1]"
          itmp=itmp|" Logarithm"
          itmp=itmp|" Square"
          itmp=itmp|" Add polynomial terms"
          itmp=itmp|" Cancel"
          numtr=select1item(stmp,itmp,atmp) 
;
          dummy =(numtr==1)
          stand =(numtr==2)
          submin=(numtr==3)
          tr01  =(numtr==4)
          logar =(numtr==5)
          squar =(numtr==6)
          poly  =(numtr==7)
          cancel=(numtr==8)
;
          switch
            case (dummy)
              j=0
              while (j<rows(jtrans))
                j=j+1          ; j  is index for xtrans
                jj =jtrans[j]  ; jj is index for xall
;
                xj=xall[,jj]
                {xcat,f}=discrete(xj)
;
                if (rows(xcat>1))  ; make sure that we need to do something
                  xj=(xj==xcat')
                  xj=xj[,2:cols(xj)]
;          
                  p=cols(xall)
                  switch
                    case ((jj==1)*(jj==p)) ; have only 1 x-column!
                      xall=xj
                      xcurr=xallvars
                      xallvars=string(xcurr+"#%1.0f",2:rows(xcat))
                      break
                    case ((jj==1)*(jj<p))
                      xall=xj~xall[,2:p]
                      xcurr=paf(xallvars,(1:p)==1)
                      xallvars=paf(xallvars,(1:p)!=1)
                      xallvars=string(xcurr+"#%1.0f",2:rows(xcat))|xallvars
                      break
                    case ((jj==p)*(jj>1))
                      xall=xall[,1:p-1]~xj
                      xcurr=paf(xallvars,(1:p)==p)
                      xallvars=paf(xallvars,(1:p)!=p)
                      xallvars=xallvars|string(xcurr+"#%1.0f",2:rows(xcat))
                      break
                    default
                      xall=xall[,1:jj-1]~xj~xall[,jj+1:p]
                      xvarslow =paf(xallvars,(1:p)<jj)
                      xvarshigh=paf(xallvars,(1:p)>jj)
                      xcurr=paf(xallvars,(1:p)==jj)
                      xallvars=xvarslow|string(xcurr+"#%1.0f",2:rows(xcat))|xvarshigh
                  endsw 
                  jp=sum(jj-jset>=0)        ; all in jset until jj
                  if (jp>=1)
                    jsetlow=jset[1:jp]        ; -> keep them!
                  else
                    jsetlow=0
                  endif
                  if (min(abs(jj-jset))==0) ; variable was set
                    jsetlow=jsetlow|(jj+1:jj+rows(xcat)-2) ; append new 
                  endif
                  if (jp<rows(jset))        ; there is a rest
                    jset=jsetlow|(jset[jp+1:rows(jset)]+rows(xcat)-2)
                  else
                    jset=jsetlow
                  endif
                  jset=paf(jset,jset>0)
                  x=xall[,jset]
;
                  if (j<rows(jtrans))
                    jtrans[j+1:rows(jtrans)]=jtrans[j+1:rows(jtrans)]+rows(xcat)-2
                  endif
                endif
              endo
              break
;
            case (stand)
              j=0
              while (j<rows(jtrans))
                j=j+1
                jj=jtrans[j]
                xj=xall[,jj]
                if (var(xj)>0)
                  xj=(xj-mean(xj))./sqrt(var(xj))
                  xall[,jj]=xj
                endif
              endo
              break
;
            case (submin)
              j=0
              while (j<rows(jtrans))
                j=j+1
                jj=jtrans[j]
                xj=xall[,jj]
                xj=xj-min(xj)
                xall[,jj]=xj
              endo
              break
;
            case (tr01)
              j=0
              while (j<rows(jtrans))
                j=j+1
                jj=jtrans[j]
                xj=xall[,jj]
                xj=xj-min(xj)
                if (max(xj)>0)
                  xj=xj./max(xj)
                  xall[,jj]=xj
                endif
              endo
              break
;
            case (logar)
              j=0
              while (j<rows(jtrans))
                j=j+1
                jj=jtrans[j]
                xj=xall[,jj]
                if (min(xj)>0)
                  xj=log(xj)
                  xall[,jj]=xj
                endif
              endo
              break
;
            case (squar)
              j=0
              while (j<rows(jtrans))
                j=j+1
                jj=jtrans[j]
                xall[,jj]=xall[,jj]^2
              endo
              break
;
            case (poly)
              p=cols(xall)
              jj=jtrans[1]
              xvarsp="Order of polynomial for "+paf(xallvars,(1:p)==jj)
              j=1
              while (j<rows(jtrans))
                j=j+1
                jj=jtrans[j]
                tmp="Order of polynomial for "+paf(xallvars,(1:p)==jj)
                xvarsp=xvarsp|tmp
              endo
;
              ord=readvalue(xvarsp,matrix(rows(jtrans)))
              while (min(ord)<=0)
                ord=readvalue(xvarsp,ceil(max(matrix(rows(jtrans))~ord,2)))
              endo
              ord=ceil(ord)
;
              j=0
              while (j<rows(jtrans))
                j=j+1          ; j  is index for xtrans
                jj =jtrans[j]  ; jj is index for xall
;
                if (ord[j]>1)  ; make sure that we need to do something
                  xj=xall[,jj]
;
                  jo=1
                  while (jo<ord[j])
                    jo=jo+1
                    xj=xj~((xall[,jj])^jo)
                  endo
;
                  p=cols(xall)
                  switch
                    case ((jj==1)*(jj==p)) ; have only 1 x-column!
                      xall=xj
                      xcurr=xallvars
                      xallvars=xcurr|string(xcurr+"^%1.0f",2:ord[j])
                      break
                    case ((jj==1)*(jj<p))
                      xall=xj~xall[,2:p]
                      xcurr=paf(xallvars,(1:p)==1)
                      xallvars=paf(xallvars,(1:p)!=1)
                      xallvars=xcurr|string(xcurr+"^%1.0f",2:ord[j])|xallvars
                      break
                    case ((jj==p)*(jj>1))
                      xall=xall[,1:p-1]~xj
                      xcurr=paf(xallvars,(1:p)==p)
                      xallvars=paf(xallvars,(1:p)!=p)
                      xallvars=xallvars|xcurr|string(xcurr+"^%1.0f",2:ord[j])
                      break
                    default
                      xall=xall[,1:jj-1]~xj~xall[,jj+1:p]
                      xvarslow =paf(xallvars,(1:p)<jj)
                      xvarshigh=paf(xallvars,(1:p)>jj)
                      xcurr=paf(xallvars,(1:p)==jj)
                      xallvars=xvarslow|xcurr|string(xcurr+"^%1.0f",2:ord[j])|xvarshigh
                  endsw
                  jp=sum(jj-jset>=0)        ; all in jset until jj
                  if (jp>=1)
                    jsetlow=jset[1:jp]        ; -> keep them!
                  else
                    jsetlow=0
                  endif
                  if (min(abs(jj-jset))==0) ; variable was set
                    jsetlow=jsetlow|(jj+1:jj+ord[j]-1) ; append new 
                  endif
                  if (jp<rows(jset))        ; there is a rest
                    jset=jsetlow|(jset[jp+1:rows(jset)]+ord[j]-1)
                  else
                    jset=jsetlow
                  endif
                  jset=paf(jset,jset>0)
                  x=xall[,jset]
;
                  if (j<rows(jtrans))
                    jtrans[j+1:rows(jtrans)]=jtrans[j+1:rows(jtrans)]+ord[j]-1
                  endif
                endif
              endo
              break
;
            default ; (cancel) = do nothing
          endsw
        endif
        break
;
      case (scattdo)
;       =========================================
;        Show scatterplot of marginal dependence
;       =========================================
        fintmp=0
        while (fintmp==0)
          stmp="SELECT one variable for scatterplot"
          atmp="Select at most ONE variable for scatterplot"
          itmp=(string(" ",1:cols(xall))+xallvars)
          nump=select0item(stmp,itmp,atmp)
;
          if (nump>0)
            xj=xall[,nump]
            xjvars=xallvars[nump]
            optj=glmopt(opt,"xvars",xjvars,"name",name,opt)
            glmplot(xj,y,optj)
          else
            fintmp=1
          endif
        endo
;
        break
;
      case (optdo)
;       ===============================
;        Change Defaults
;       ===============================
        fintmp=0
        while (fintmp==0)
          itmp=     "General settings"
          itmp=itmp|"Model settings"
          itmp=itmp|"Iteration settings"
          stmp ="Select GROUP to change"
          stmp ="Select at most ONE group to change"
          nump=select0item(stmp,itmp,atmp)
;
          switch
            case (nump==1) ; General
              itmx=     " Intercept  (yes/no) "
              itmx=itmx|" Scan for replications  (yes/no) "
              itmx=itmx|" Show output in display  (yes/no) "
              itmx=itmx|" Name for output  "
              itmx=itmx|" Title for output  (none=no title)"
              valx=     paf("yes"|"no",interc==(1|0))
              valx=valx|paf("yes"|"no",norepl==(0|1))
              valx=valx|paf("yes"|"no",nopic==(0|1))
              valx=valx|name
              valx=valx|title
              bad=1
              while (bad)
                if (haveconst) ; do not ask for intercept if have one!
                  valxnew=readvalue(itmx[2:8],valx[2:8])
                  valxnew=0|valxnew
                else
                  valxnew=readvalue(itmx,valx)
                endif
                bad=(sum(sum(valxnew[1|2|3]==("yes"~"no")),2)!=3)
              endo     
              valx=valxnew     
              interc=(valx[1]=="yes")
              norepl=1-(valx[2]=="yes")
              nopic=1-(valx[3]=="yes")
              name =valx[4]
              title=valx[5]
              break
;
            case (nump==2) ; Model
              itmx=     " Model code              (none=choose later)"
              itmx=itmx|" Power for power link    (0=logarithm)  "
              itmx=itmx|" k for negative binomial (1=geometric)  "
              valx=     code
              valx=valx|string("%-1.8g",pow)
              valx=valx|string("%-1.8g",nbk)
              bad=1
              while (bad)
                valxnew=readvalue(itmx,valx)
                bad=(sum(valxnew[1]==(glmmodels.all|"none"))==0)
                tmp=atof(valxnew[2])
                bad=bad||(1-isNumber(tmp))
                tmp=atof(valxnew[3])
                bad=bad||(1-isNumber(tmp))||(tmp<1)
              endo     
              valx=valxnew
              code=valx[1]
              pow =atof(valx[2])
              nbk =atof(valx[3])
              break
;
            case (nump==3) ; Iteration
              itmx=     " Newton-Raphson algorithm     (yes/no=Fisher scoring) "
              itmx=itmx|" Convergence criterion                 "
              itmx=itmx|" Maximal number of iterations          "
              itmx=itmx|" Show iteration going on      (yes/no) "
              valx=     paf("yes"|"no",fscor==(0|1))
              valx=valx|string("%-1.8g",cnv)
              valx=valx|string("%-1.0f",miter)
              valx=valx|paf("yes"|"no",shf==(1|0))
              bad=1
              while (bad)
                valxnew=readvalue(itmx,valx)
                bad=(sum(sum(valxnew[1|4]==("yes"~"no")),2)!=2)
                tmp=atof(valxnew[2])
                bad=bad||(1-isNumber(tmp))||(tmp<=0)
                tmp=atof(valxnew[3])
                bad=bad||(1-isNumber(tmp))||(tmp<=1)
              endo     
              valx=valxnew     
              fscor=1-(valx[1]=="yes")
              cnv=atof(valx[2])
              miter=round(atof(valx[3]))
              shf  =(valx[4]=="yes")
              break
;       
            default ; cancel
              fintmp=1
              break
          endsw
        endo
; 
        if (1-haveopt)
          opt=glmopt("name",name)
        else
          if (exist(name)!=2)
            opt=glmopt("name",name,opt)
          endif
        endif
        opt=glmopt("cnv",cnv,"miter",miter,"norepl",norepl,opt)
        opt=glmopt("shf",shf,"fscor",fscor,"nopic",nopic,opt)
        if (title!="none")
          opt=glmopt("title",title,opt)
        endif
        if (code!="none")
          opt=glmopt("code",code,opt)
        endif
;
        break
;
      case (resetdo)
;       ===============================
;        Reset X
;       ===============================
        selx="RESET all X variables"
        itmx=(" Reset")|(" Cancel")
        numx=1|1
        while (sum(numx)!=1)
          numx=selectitem(selx,itmx)
        endo
;        
        if (numx[1]==1)
          xall=xorg
          x   =xorg
          jall=1:cols(xorg)
          jset=1:cols(xorg)
          xallvars=xorgvars
          xvars=xorgvars
        endif
        break
;
      case (glmdo)
;       ===============================
;        Do GLM fit
;       ===============================
        x=xall[,jset]
        jall=1:cols(xall)
        jstar=(min(abs(jall-jset'),2)==0)
        xvars=paf(xallvars,jstar)
        xvars=xvars+"                       "
        xvars=substr(xvars,1,19)
;
        blank=string(" ",1:rows(xvars))
        xvarsnew=substr(xvars,1,2)
        j=1
        while (j<18)
          j=j+1
          sub=substr(xvars,j,j+1)
          if (1-prod(sub==blank)) 
            xvarsnew=xvarsnew+sub
          endif
        endo
        xvars=xvarsnew
;
        if (interc)
          opt=glmopt("xvars","const"|xvars,"name",name,opt)
          glmfit(matrix(n)~x,y,opt)
        else
          opt=glmopt("xvars",xvars,"name",name,opt)
          glmfit(x,y,opt)
        endif
        break
;
      default; stop
;       ===============================
;        Cancel
;       ===============================
        finish=1
    endsw
  endo
endp
