proc(del)=dwade(x,y,h)
; -----------------------------------------------------------------
; Library        metrics
; -----------------------------------------------------------------
;  See_also      adeind adeslp
; -----------------------------------------------------------------
;   Macro        dwade
; -----------------------------------------------------------------
;   Keywords     single index model, semiparametric estimation, 
;		 average derivative
; -----------------------------------------------------------------
;   Description  dwade estimation of the density weighted average derivatives  
; -----------------------------------------------------------------
;   Reference    Powell, Stock and Stoker (1989), "Semiparametric Estimation
;		 of Index Coefficients", Econometrica, 57, 1403-1430
; -----------------------------------------------------------------
;   Usage        d = dwade(x,y,h)    
;   Input
;     Parameter   x  
;     Definition      n x d matrix , the observed explanatory variable    
;     Parameter   y 
;     Definition      n x 1 matrix , the observed response variable       
;     Parameter   h  
;     Definition      d x 1 or 1 x 1 matrix , chosen bandwidth            
;   Output                      
;     Parameter   d 
;     Definition      d x 1 matrix, the density weighted average          
;                     derivative estimator  
; -----------------------------------------------------------------
;   Example   library("metrics")
;	      randomize(0)
;             n   = 100
;             x   = normal(n,3)
;             z   = 0.2*x[,1] - 0.7*x[,2] + x[,3]
;             eps = normal(n,1) * sqrt(0.5)
;             y   = 2 * z^3 + eps
;             h   = 0.3
;             d   = dwade(x,y,h)
;             d   
; -----------------------------------------------------------------
;   Result    the density weighted average derivative estimator of 
;             Powell, Stock and Stoker, Econometrica (1989) 
; -----------------------------------------------------------------
;   Author    Sperlich & Stockmeyer 951111  
; -----------------------------------------------------------------
  error(rows(x)!=rows(y),"x and y have different number of rows")
  error(cols(y)!=1 ,"y has more than one column")
  d  = cols(x)
  n  = rows(x)
  if (rows(h)<2)                ;  reshape h to matrix
     h = matrix(1,1,d)*h
  else
     reshape ("h",#(1,1,d))
  endif
  reshape ("x",#(n,1,d))
  xt = (x-x')./h 
  xs = prod(pdfn(xt),3)         ;  mult. over 3. dimension        
  xt = sum((xt.*xs),1)          ;  sum over 1. dimension
  reshape("xt",#(n,d))
  del = xt'*y
  reshape("h",#(d,1,1))
  del = -2*del ./ ( n*(n-1)*prod(h,1)*h)
endp
  