proc(ll1,ll2)=glmlld(code,eta,y,opt)
; ----------------------------------------------------------------------------
; Library       glm
; ----------------------------------------------------------------------------
;  See_also     glmcore
; ----------------------------------------------------------------------------
;   Macro       glmlld
; ----------------------------------------------------------------------------
;   Description  glmlld computes the first and second derivative of the 
;                individual log-likelihood in dependence of the linear
;                index eta and y.
; ----------------------------------------------------------------------------
;   Link  ../tutorials/glmstart.html Tutorial: GLM in XploRe 
; ----------------------------------------------------------------------------
;   Usage        {ll1,ll2} = glmlld(code,eta,y{,opt})
;   Input
;     Parameter   code 
;     Definition        text string, the short code for the model (e.g. 
;                       "bipro" for probit or "noid" for linear model).
;                       Append "fs" to the code for the expected ll2
;                       instead of ll2 (-> Fisher scoring algorithm).
;     Parameter   eta  
;     Definition        n x d  matrix, the index values.
;     Parameter   y  
;     Definition        n x d  matrix, the response values.
;     Parameter   opt
;     Definition        optional, a list with optional input. The macro
;                       "glmopt" can be used to set up this parameter.
;     Parameter   opt.pow
;     Definition        scalar, power for power link. If not given, set 
;                       to 0 (logarithm).
;     Parameter   opt.nbk
;     Definition        scalar, extra parameter k for negative binomial
;                       distribution. If not given, set to 1 (geometric 
;                       distribution).
;   Output
;     Parameter   ll1  
;     Definition        n x d  matrix, 1st derivative of log-likelihood.
;     Parameter   ll2  
;     Definition        n x d  matrix, 2nd derivative of log-likelihood.
; ----------------------------------------------------------------------------
;   Example   library("glm")
;             y=1
;             eta=2
;             {ll1,ll2}=glmlld("bilo",eta,y)
;             ll1
;             ll2
; ----------------------------------------------------------------------------
;   Result    The derivatives of the individual log-likelihood for a logit
;             model are computed at y=1 and eta=2:
;             Contents of ll1
;             [1,]   0.1192 
;             Contents of ll2
;             [1,] -0.10499  
; ----------------------------------------------------------------------------
;   Author    Marlene Mueller, 2000/05/18
; ----------------------------------------------------------------------------
;
; classify our algo
;
  glmmodels=getglobal("glmmodels")
;
  binomial  = sum(code==(glmmodels.binomial)) >0
  binomial  = binomial + sum(code==(glmmodels.binomial+"fs")) >0
  gaussian  = sum(code==(glmmodels.gaussian)) >0
  gaussian  = gaussian + sum(code==(glmmodels.gaussian+"fs")) >0
  poisson   = sum(code==(glmmodels.poisson)) >0
  poisson   = poisson + sum(code==(glmmodels.poisson+"fs")) >0
  gamma     = sum(code==(glmmodels.gamma)) >0
  gamma     = gamma + sum(code==(glmmodels.gamma+"fs")) >0
  igaussian = sum(code==(glmmodels.igaussian)) >0
  igaussian = igaussian + sum(code==(glmmodels.igaussian+"fs")) >0
  nbinomial = sum(code==(glmmodels.nbinomial)) >0
  nbinomial = nbinomial + sum(code==(glmmodels.nbinomial+"fs")) >0
;
  pow=0
  nbk=1
;
  if (exist(opt)>0)
    if (comp(opt,"pow")>0)
      pow=opt.pow
    endif
    if (comp(opt,"nbk")>0)
      nbk=opt.nbk
    endif
  endif

  switch
;   ========================================================
    case (binomial)
;   ========================================================
    switch
      case ((code=="bilo") || (code=="bilofs"))
        e=exp(eta)
        g=e./(1+e)
        ll1 = y - g
        ll2 = -g./(1+e) 
        break
      case (code=="bipro")  
        g   = cdfn(eta)
        gg  = g.*cdfn(-eta)
        gg8 = cdfn(8).*(1-cdfn(8)) 
        gg  = gg.*(gg.>gg8) + gg8.*(gg.<=gg8)  
        g1  = pdfn(eta)
        ll1 = (y - g).*g1./gg
        g2  = -eta.*g1 
        g11 = g1.*g1
        ll2 = g2/gg - (1-2.*g).*g11./(gg.*gg)
        ll2 = ll2.*(y-g)
        ll2 = ll2 - g11./gg
        break
      case (code=="bicll")  
        e = exp(eta)
        ee= exp(-e)
        g = 1-ee
        ll1= (y-g).*e/g
        ll2= ll1 - ee.*e/(g.*g)
        break
      case (code=="biprofs")  
        g   = cdfn(eta)
        gg  = g.*cdfn(-eta)
        g1  = pdfn(eta)
        ll1 = (y - g).*g1./gg
        ll2 = -g1.*g1./gg
        break
      case (code=="bicllfs")  
        e = exp(eta)
        ee= exp(-e)
        g = 1-ee
        ll1= (y-g).*e/g
        ll2= -ee.*e/(g.*g)
        break
      default
        error(1, "'"+code+"' is not a valid model")
        break
    endsw
    break
;   ========================================================
    case (gaussian)
;   ========================================================
    switch
      case ((code=="noid") || (code=="noidfs"))
        ll1=(y-eta)
        ll2=-1
        break
      case (code=="nopow")  
        if (pow==0)
          g=exp(eta)
          ll1=(y-g).*g
          ll2=(y-2.*g).*g
        else
          g=eta^(1/pow)
          ll1=(1/pow)*(y-g).*g./eta
          ll2=(1/pow)*g.*( (1/pow-1).*(y-g)-(1/pow).*g )./(eta.*eta)
        endif
        break
      case (code=="nopowfs")  
        if (pow==0)
          g=exp(eta)
          ll1=(y-g).*g
          ll2=-g.*g
        else
          g=eta^(1/pow)
          e=(1/pow)*g./eta
          ll1= (y-g).*e
          ll2=-e.*e
        endif
        break
      default
        error(1, "'"+code+"' is not a valid model")
        break
    endsw
    break
;   ========================================================
    case (poisson)
;   ========================================================
    switch
      case ((code=="polog") || (code=="pologfs")) 
        g=exp(eta)
        ll1= y-g
        ll2= -g
        break
      case (code=="popow")  
        if (pow==0)
          g=exp(eta)
          ll1= y-g
          ll2= -g
        else
          g=eta^(1/pow)
          ll1= (1/pow).*(y-g)./eta
          ll2=-(1/pow).*(y-(1/pow-1)*g)./(eta.*eta)
        endif
        break
      case (code=="popowfs")  
        if (pow==0)
          g=exp(eta)
          ll1= y-g
          ll2= -g
        else
          g=eta^(1/pow)
          ll1= (1/pow)*(y-g)./eta
          ll2=-(1/pow)*(2-1/pow).*g./(eta.*eta)
        endif
        break
      default
        error(1, "'"+code+"' is not a valid model")
        break
    endsw
    break
;   ========================================================
    case (gamma)
;   ========================================================
    switch
      case ((code=="gacl") || (code=="gaclfs")) 
        g=1./eta
        ll1= -y+g
        ll2= -g.*g
        break
      case (code=="gapow")  
        if (pow==0)
          e=y.*exp(-eta)
          ll1= e-1
          ll2= -e
        else
          e=eta^(-1/pow)
          ll1= (1/pow)*(y.*e-1)./eta
          ll2=-(1/pow)*((1/pow+1)*y.*e-1)./(eta.*eta)
        endif
        break
      case (code=="gapowfs")  
        if (pow==0)
          e=y.*exp(-eta)
          ll1= e-1
          ll2= -e
        else
          ll1= (1/pow)*(y.*eta^(-1/pow)-1)./eta
          ll2=-(1/(pow*pow))./(eta.*eta)
        endif
        break
      default
        error(1, "'"+code+"' is not a valid model")
        break
    endsw    
    break
;   ========================================================
    case (igaussian)
;   ========================================================
    switch
      case ((code=="igcl") || (code=="igclfs")) 
        g=1./sqrt(eta)
        ll1= 0.5*(-y+g)
        ll2= -0.25*g./eta
        break
      case (code=="igpow")  
        if (pow==0)
          e=exp(-eta)
          ll1= (y.*e-1).*e
          ll2=-(2*y.*e-1).*e
        else
          e=eta^(-1/pow)
          ee=y.*e.*e
          ll1= (1/pow)*(ee-e)./eta
          ll2=-(1/pow)*((2/pow+1)*ee-(1/pow+1)*e)./(eta.*eta)
        endif
        break
      case (code=="igpowfs")  
        if (pow==0)
          e=exp(-eta)
          ll1= (y.*e-1).*e
          ll2=-(2*y.*e-1).*e
        else
          e=eta^(-1/pow)
          ll1= (1/pow)*(y.*e.*e-e)./eta
          ll2=-(1/(pow*pow))*e./(eta.*eta)
        endif
        break
      default
        error(1, "'"+code+"' is not a valid model")
        break
    endsw
    break
;   ========================================================
    case (nbinomial)
;   ========================================================
    switch
      case ((code=="nbcl") || (code=="nbclfs")) 
        e=exp(eta)
        ee=1-e
        ll1= y-nbk*e./ee
        ll2=-nbk./(ee.*ee)
        break
      case (code=="nbpow")
        if (pow==0)
          e=exp(-eta)
          ee=nbk+e
          ll1= y-(y+nbk).*log(ee)
          ll2=-nbk*(y+nbk).*e./(ee.*ee)
        else
          e=eta^(1/pow)
          ee=eta.*eta
          ll1= (1/pow)*( y./eta - (y+nbk).*e./((e+nbk).*eta) )
          ll2=-(1/pow)*( y./ee - (y+nbk).*((1/pow-1)*nbk*e-e.*e)./((e+nbk).*ee) )
        endif
        break
      case (code=="nbpowfs")  
        if (pow==0)
          e=exp(-eta)
          ll1= y-(y+nbk).*log(e+nbk)
          ll2=-nbk*e./(nbk+e)
        else
          e=eta^(1/pow)
          ee=eta.*eta
          ll1= (1/pow)*( y./eta - (y+nbk).*e./((e+nbk).*eta) )
          ll2=-(1/pow)*( e./ee - ((1/pow-1)*nbk*e - e.*e)./(e+nbk) )
        endif
        break
      default
        error(1, "'"+code+"' is not a valid model")
        break
    endsw
    break
;   ========================================================
    default     
;   ========================================================
      error(1, "'"+code+"' is not (yet?) a valid model")
      break
  endsw
endp



