proc(b)=gls(x, y, om)
; -----------------------------------------------------------------------
; Library      xplore
; -----------------------------------------------------------------------
; Macro        gls
; -----------------------------------------------------------------------
; Description  Computes the Generalized Least Squares estimate for
;              the coefficients of a linear model when the errors have 
;              as covariance matrix  sigma^2 * om. 
; -----------------------------------------------------------------------
;   Notes      The estimate is given by b = inv(x'*om*x)*x'*om*y.
; -----------------------------------------------------------------------
;   Keywords   linear regression, generalized linear regression,
;              least squares
; -----------------------------------------------------------------------
; Usage        b = gls (x, y{, om})
; Input
;   Parameter  x 
;   Definition n x p x d1 x ... x dn array    
;   Parameter  y 
;   Definition n x 1 x d1 x ... x dn array   
;   Parameter  om
;   Definition optional, n x n x d1 x ... x dn array   
; Output
;   Parameter  b
;   Definition p x 1 x d1 x ... x dn array   
; -----------------------------------------------------------------------
; Example      library("xplore")
;              randomize(1964)
;              n = 50
;              x = matrix(n)~normal(n,2)
;              beta = #(10, 2, 3)
;              u = 0.5 * normal(n)
;              y = x*beta .+ u
;              b = gls (x, y)
;              b
; -----------------------------------------------------------------------
; Result       Contents of b
;              [1,]     9.97 
;              [2,]   1.9116 
;              [3,]   3.0123 
; -----------------------------------------------------------------------
; Example      library("xplore")
;              randomize(1964)
;              n = 50
;              x = matrix(n)~normal(n,2)
;              beta = #(10, 2, 3) 
;              covar = (0.5.*x[,2] .+ 0.3.*x[,3]).^2
;              y = x*beta .+ sqrt(covar).*u
;              b2 = gls (x, y, diag(covar))
;              b2
; -----------------------------------------------------------------------
; Result       Contents of b2
;              [1,]   9.9977 
;              [2,]   1.9946 
;              [3,]   3.0093  
; -----------------------------------------------------------------------
; Author       Sigbert Klinke, 920424, 960327, 960425
; -----------------------------------------------------------------------
  xtom = x'
  if(exist(om))
    xtom = xtom*inv(om)
  endif
  b    = inv(xtom*x)*(xtom*y)
endp

