; -----------------------------------------------------------------
; Library        stats
; -----------------------------------------------------------------
;  See_also     skewness kurtosis
; -----------------------------------------------------------------
;   Macro       jarber
; -----------------------------------------------------------------
;   Keywords    normality test
; -----------------------------------------------------------------
;   Description Quantlet for testing the normality of a data set.
; -----------------------------------------------------------------
;   Usage       {jb,probJB,sk,k} = jarber(resid,printout)
;   Input
;       Parameter   resid
;       Definition  (n x 1) matrix of observations
;       Parameter   printout
;       Definition  scalar, 0: no printout, 1: printout
;   Output
;       Parameter   jb
;       Definition  test statistic of Jarque-Bera test
;       Parameter   probjB
;       Definition  probability value of test statistics
;       Parameter   sk
;       Definition  skewness
;       Parameter   k
;       Definition  kurtosis
; -----------------------------------------------------------------
;   Example     x = normal(100,1)
;               {jb,probjb,sk,k} = jarber(x,1)
; -----------------------------------------------------------------
;   Result  prints and returns JB-test statistic and its
;           probability value, the standard deviation, skewness 
;           and kurtosis of the data in vector x
; -----------------------------------------------------------------
;   Reference Bera, A.K. and Jarque, C.M. (1982). Model Specification
;             Tests: a Simultaneous Approach, Journal of Econometrics,
;             20, 59-82.
; -----------------------------------------------------------------
;   Author    Tschernig  000427
; -----------------------------------------------------------------
proc (jb,probjb,sk,k) = jarber(resid,printout)
    resid   = resid - mean(resid)
    T       = rows(resid)
    m2      = sum(resid^2) / (T-1)
    m3      = sum(resid^3) / (T-1)
    m4      = sum(resid^4) / (T-1)
    sk      = m3 / var(resid)^(3/2)
    k       = m4/m2^2
    jb      = T/6*(sk^2 + (k-3)^2 / 4)
    probjb = 1-cdfc(jb,2)
    if (printout == 1)
        o = string("", (1:8))
        o[1] = ""
        o[2] = "Bera-Jarque (1992) test for normality"
        o[3] = ""
        o[4] = string("JB-test statistic: %g",jb);
	o[5] = string("probability value: %g", probjb)
        o[6] = string("std.dev.:          %g", sqrt(var(resid)));
        o[7] = string("skewness:          %g", sk);
        o[8] = string("kurtosis:          %g", k);
        o
    endif
endp
