proc(Estmu, EstF, EstQ, EstR) = kem(y,mu,Sig,H,F,Q,R,LoopLimit) 

; -----------------------------------------------------------------
; Library        times
; -----------------------------------------------------------------
;  See_also      kfilter, ksmoother
; -----------------------------------------------------------------
;   Macro        kem
; -----------------------------------------------------------------
;   Description  Calculates estimates of mu, F, Q and R in a
;                state-space model using EM-algorithm.
;                The state-space model is assumed to be in the
;                following form:
;
;                y_t = H x_t + v_t
;
;                x_t = F x_t-1 + w_t
;
;                x_0 ~ (mu,Sig), v_t ~ (0,Q), w_t ~ (0,R)
;
;                Parameters Sig and H are assumed known.
;
; -----------------------------------------------------------------
;   Keywords     Kalman filter, Kalman smoother, filter
; -----------------------------------------------------------------
;   Usage         {Estmu, EstF, EstQ, EstR} = kem(y,mu,Sig,H,F,Q,R,LoopLimit)    
;   Input
;     Parameter   y  
;     Definition      T x m matrix of observed time series, 
;                     T is the number of observations,
;                     m is the dimension of time series
;     Parameter   mu    
;     Definition      n x 1 vector, an initial value for iterations
;     Parameter   Sig    
;     Definition      n x n covariance matrix of the initial state
;     Parameter   H    
;     Definition      m x n matrix
;     Parameter   F    
;     Definition      n x n matrix, an initial value for iterations
;     Parameter   Q   
;     Definition      m x m variance-covariance matrix, an initial
;                     value for iterations
;     Parameter   R
;     Definition      n x n variance-covariance matrix, an initial
;                     value for iterations 
;   Output                                                           
;     Parameter   Estmu    
;     Definition      n x 1 vector, the estimate of the mean of the 
;                     initial state
;     Parameter   EstF    
;     Definition      n x n matrix, the estimate of the state
;                     transition matrix
;     Parameter   EstQ   
;     Definition      m x m matrix, the estimate of the covariance 
;                     matrix
;     Parameter   EstR
;     Definition      n x n matrix, the estimate of the covariance 
;                     matrix
; -----------------------------------------------------------------
;   Example   library("xplore")
;             library("times")
;             library("plot")
;
;             serie = read("kalman2.dat")
;             y = serie[,2]
;             mu = #(0,0)
;             T = 100
;             Sig = #(0,0)~#(0,0)
;             H = #(1,0)'
;             F = #(5,1)~#(1,5)
;             R = #(1,0)~#(0,1)
;             Q = 10
;
;             {Estmu, EstF, EstQ, EstR} = kem(y,mu,Sig,H,F,Q,R,10)
;             Estmu
;             EstF
;             EstQ
;             EstR
;             smoothed = ksmoother(y,Estmu,Sig,H,EstF,EstQ,EstR)
;             orig = vec(1:T)~y
;             smoot = vec(1:T)~smoothed
;             orig = setmask(orig, "line", "red", "thin")
;             smoot = setmask(smoot, "line", "blue", "medium")
;             disp = createdisplay(1,1)
;             show(disp,1,1, orig, smoot)
;             setgopt(disp,1,1, "title", "AR(2) with noise - EM estimated")
; -----------------------------------------------------------------
;   Result    Original series is displayed with red colour,
;             filtered series is displayed with blue colour.
;             (y is an AR(2) process with errors.)
;
;             Contents of Estmu

;             [1,] -0.049524 
;             [2,]  0.011351 
;             Contents of EstF

;             [1,]  0.70331  0.11066 
;             [2,]  0.80357   4.0443 
;             Contents of EstQ

;             [1,]   3.6792 
;             Contents of EstR

;             [1,]  0.19393 -0.018777 
;             [2,] -0.018777  0.81767 
; -----------------------------------------------------------------
;   Author    P.Franek 990507
; -----------------------------------------------------------------

 
; setting of dimensions
 
  T = rows(y)  
  dimY = cols(y)
  dimX = rows(Sig)
 
; dimension check

  error(rows(H)<>dimY,"Matrix H and y dim mismatch.")
  error(cols(H)<>dimX,"Matrix H and x dim mismatch.")
 
  Loop = 1                   

  do                         ; start of main cycle
    
; kalman iteration
    
    smoothed = ksmoother(y,mu,Sig,H,F,Q,R)
    KFOutPut = read("KFOutPut.dat")
    KSOutPut = read("KSOutPut.dat")


    A = matrix(dimX,dimX).*0
    B = matrix(dimX,dimX).*0
    C = matrix(dimX,dimX).*0
    D = matrix(dimY,dimY).*0 
    
    i=rows(KSOutPut)                                                   ; T+1 in fact
    
; P_T,T-1|T initialization - by Shumway&Stoffer
    
    pomP = reshape(KFOutPut[i-1,dimX+1:dimX+dimX^2],#(dimX,dimX))      ; get P_T-1|T-1
    pom = F * pomP * F' + R                                            ; count P_T|T-1
    KT = pom * H' * ginv(H * pom * H' + Q)                             ; count K_T
    
    Pcov = (unit(dimX) - KT * H) * F * pomP                            ; Pcov = P_T,T-1|T

; A, B, C, D matrices evaluation
    
    while (i>=2)
      
      pomXcur = KSOutPut[i,1:dimX]'                                         ; get x_i|T
      pomPcur = reshape( KSOutPut[i,dimX+1:dimX+dimX^2], #(dimX,dimX))      ; get P_i|T
      pomXprev = KSOutPut[i-1,1:dimX]'                                      ; get x_i-1|T
      pomPprev = reshape( KSOutPut[i-1,dimX+1:dimX+dimX^2], #(dimX,dimX))   ; get P_i-1|T
      
; evaluation of A
      
      A = A + pomXprev * pomXprev' + pomPprev
      
; evaluation of B
      
      B = B + pomXcur * pomXprev' + Pcov
      
      if (i>2)         ; if there is not beginning of data, update Pcov 
        
        pomPprev1 = reshape( KFOutPut[i-1,dimX+1:dimX+dimX^2], #(dimX,dimX)) ; get P_i-1|i-1
        pomPprev2 = reshape( KFOutPut[i-2,dimX+1:dimX+dimX^2], #(dimX,dimX)) ; get P_i-2|i-2
        
        pom = F * pomPprev1 * F' + R
        pominv = ginv(pom)
        PStar1 = pomPprev1 * F' * pominv      ; corresponds to J_(t-1)
        
        pom = F * pomPprev2 * F' + R
        pominv = ginv(pom)
        PStar2 = pomPprev2 * F' * pominv      ; corresponds to J_(t-2)
        
        Pcov = pomPprev1 * PStar2' + PStar1 * (Pcov - F * pomPprev1) * PStar2' 
        
      endif
      
; evaluation of C
      
      C = C + pomXcur * pomXcur' + pomPcur
      
; evaluation of D
      
      D = D + 1/T * ((y[i]' - H * pomXcur)*(y[i]' - H * pomXcur)' + H * pomPcur * H')
      
      i = i-1
    endo
    
    mu = KSOutPut[1,1:dimX]'
    F = B * ginv(A)
    R = 1/T * (C - B * ginv(A) * B')
    Q = D

    Loop = Loop + 1    
until (Loop > LoopLimit)        ; end of main cycle

  Estmu = mu
  EstF  = F
  EstR  = R
  EstQ  = Q
 
endp
