proc (kp, critvalue)=kpss(y,t,output)
; ----------------------------------------------------------------------
; Library         times
; ----------------------------------------------------------------------
;  See_also       adf, rvlm, lo, robwhittle, roblm, lobrob, gph, sclmtest, neweywest
;----------------------------------------------------------------------
;   Macro         kpss
; ----------------------------------------------------------------------
;   Description   Calculation of the KPSS statistics for I(0) against long-memory
;                 alternatives. We consider the two tests, denoted by KPSS_mu and KPSS_t,
;                 and respectively based on a regression on a constant mu, and
;                 on a constant and a time trend t. 
;                 The quantlet returns the value of the estimated statistic for two
;                 type of the tests, i.e., const or trend and
;                 the critical values for a 10, 5 and 1 
;                 percent confidence interval for I(0) (const, trend). 
;
; ----------------------------------------------------------------------
;   Usage         {kp, critvalue} = kpss(y {,t {,output}})
;   Input 
;     Parameter   y 
;     Definition  vector
;     Parameter   t 
;     Definition  scalar, truncation lag for the autocorrelation consistent variance estimator,
;                       if t is not specified, the kpss statistic is calculated for t=0, 7 and 21
;     Parameter   output 
;     Definition  0 or 1, a text output is shown in the XploRe output window if output = 1
;     Output
;     Parameter   kp
;     Definition  vector, KPSS statistics KPSS_mu and KPSS_t
;     Parameter   critvalue
;     Definition  matrix, critic. values for 10% (1. row), 5% (2. rows) and 
;                       1% (3. row) for the kpss test with a constant (1. col)
;                       and with a constant and linear trend (2. col)
; ----------------------------------------------------------------------
;   Example     ;Non-stationarity test on the series dmus58.dat. 
;               library("times")
;               x = read("dmus58.dat") ; 25477 obs: log FX rates
;               x = x[1:1000]
;               k=kpss(x,0,1) 
;               k
; ----------------------------------------------------------------------
;   Result
;      Contents of kpss
;      
;      [1,] "   Order   Test   Statistic          Crit. Value       "
;      [2,] "                                  0.1   0.05   0.01    "
;      [3,] "_______________________________________________________"
;      [4,] "       0  const      77.702     0.347  0.463  0.739    "
;      [5,] "       0  trend      11.117     0.119  0.146  0.216    "
;      
;      Contents of k.kp
;      
;      [1,]   77.702 
;      [2,]   11.117 
;      
;      Contents of k.critvalue
;      
;      [1,]    0.347    0.119 
;      [2,]    0.463    0.146 
;      [3,]    0.739    0.216 
;
; ----------------------------------------------------------------------
;   Reference
;       Kwiatkowski,D., Phillips, P.C.B., Schmidt, P., and Y. Shin (1992):
;       Testing the Null Hypothesis of Stationarity Against the Alternative
;       of a Unit Root: How Sure Are We that Economic Series Have a Unit
;       Root, Journal of Econometrics, 54, 159-178.
; ----------------------------------------------------------------------
;  Keywords
;       long memory, tests, nonstationarity
; ---------------------------------------------------------------------
; Link http://www.quantlet.de/scripts/fin/fhh.pdf
;      Statistics of financial markets
; Link http://www.quantlet.de/codes/fin/SFMAdfKpss.html
;      Application in financial markets
; ----------------------------------------------------------------------
;   Author        Kleinow, Teyssiere, 20000509
; ----------------------------------------------------------------------
  n = rows(y) 
  rat1 = (n/100)^(0.25)
  if (!exist("t")) t=0 endif
  bandwidth = rint(t*rat1)
  temps = cumsum(matrix(n))
  ctemps = temps - mean(temps)
  cy = y - mean(y)
  betahat = sum(ctemps.*cy)/sum(ctemps^2)  
  constanthat = mean(y) - betahat*mean(temps)
  e1 = cy
  e2 = y - constanthat -betahat*temps
  S1 = sum(cumsum(e1)^2)
  S2 = sum(cumsum(e2)^2)
  ;;
  kp = matrix(2)
  sac1 = neweywest(e1,bandwidth)
  sac2 = neweywest(e2,bandwidth)
  kp[1] = S1/sac1[1]
  kp[2] = S2/sac2[1]
  kp = kp/(n^2)
  critvalue = #(0.347,  0.463,  0.739)~#(0.119,  0.146,  0.216)
  if (exist("output"))
    if (output != 0)
      line1 ="   Order   Test   Statistic          Crit. Value       "
      line2 ="                                  0.1   0.05   0.01    "
      line3 ="_______________________________________________________"
      ts = "                      "+string("%2.0f",t)
      ts = substr(ts, strlen(ts)-7,strlen(ts)+1)
      kps = "                      "+string("%2.3f",kp[1])
      kps = substr(kps, strlen(kps)-7,strlen(kps)+1)
      line4   =ts +    "  const    "+kps+"     0.347  0.463  0.739    "
      kps = "                      "+string("%2.3f",kp[2])
      kps = substr(kps, strlen(kps)-7,strlen(kps)+1)
      line5   =ts +    "  trend    "+kps+"     0.119  0.146  0.216    "
      kpss = line1|line2|line3|line4|line5
      kpss
    endif
  endif
endp

