proc (mh) = lpregest(x,h,p,K,d)
; -----------------------------------------------------------------
; Library        smoother
; -----------------------------------------------------------------
;  See_also      lpderest lpregrot
; -----------------------------------------------------------------
;   Macro        lpregest
; -----------------------------------------------------------------
;   Keywords     kernel smoothing, kernel regression,
;                local polynomial estimator
; -----------------------------------------------------------------
;   Description  estimates a regression function using
;                local polynomial kernel regression.
;                The computation uses WARPing.
; -----------------------------------------------------------------
;   Reference    Fan and Gijbels (1995): Local Polynomial Fitting
;
;                Fan and Marron (1994): Binning for local polynomials
;
;                Haerdle (1991): Smoothing Techniques
; -----------------------------------------------------------------
;   Usage        mh = lpregest (x, h {,p {,K} {,d}})
;   Input
;     Parameter  x
;     Definition   n x 2, the data. In the first column the
;                  independent, in the second column the
;                  dependent variable.
;     Parameter  h
;     Definition   scalar, bandwidth. If not given, the rule of thumb
;                  bandwidth computed by lpregrot is used.
;     Parameter  p
;     Definition   integer, order of polynomial. If not given,
;                  p=1 (local linear) is used. p=0 yields the
;                  Nadaraya-Watson estimator.
;     Parameter  K
;     Definition   string, kernel function on [-1,1] or Gaussian
;                  kernel "gau". If not given, the Quartic kernel
;                  "qua" is used.
;     Parameter  d
;     Definition   scalar, discretization binwidth. d must be smaller
;                  than h. If not given, the minimum of h/3 and
;                  (max(x[,1])-min(x[,1]))/100 is used.
;   Output
;     Parameter  mh
;     Definition   m x 2 matrix, the first column is a grid and the
;                  second column contains the regression estimate on
;                  that grid.
; -----------------------------------------------------------------
;   Example   library("smoother")
;             library("plot")
;             ;
;             x = 4.*pi.*(uniform(200)-0.5)   ; independent variable
;             m = cos(x)                      ; true function
;             e = uniform(200)-0.5            ; error term
;             x = x~(m+e)
;             ;
;             mh = lpregest(x,1)                ; estimate function
;             ;
;             mh = setmask(mh, "line","blue")
;             m  = setmask(sort(x[,1]~m) , "line","black","thin")
;             plot(x,mh,m)
; -----------------------------------------------------------------
;   Result    The Nadaraya-Watson regession estimate (blue) using
;             Quartic kernel and bandwidth h=1 and the true
;             regression function (thin black) are pictured.
; -----------------------------------------------------------------
;   Author    Marlene Mueller, 2000/03/28 - 14:04
; -----------------------------------------------------------------
  n=rows(x)
  error(cols(x)<>2,"lpregest: cols(x) <> 2")
;
  if (exist(h)==0)
    h=lpregrot(x)
  endif
  if (exist(p)==0)
    p=1            ; default is local linear
  endif
  if (exist(K)==0)
    K="qua"
  endif
  if (exist(K)==1) ; no K but d
    d=K
    K="qua"
  endif
  if (exist(d)==0)
    d=(max(x[,1])-min(x[,1]))/100
    d=min(d~(h./3),2)
  endif
;
  error(h .<=d,"lpregest: h smaller than d -- Please modify h or d!")
;
  {xb,yb}=bindata(x[,1],d,0,x[,2])        ; bin data
  g=aseq(0,h/d,d/h)
  wy=_K(g)                                ; kernel weights
  wx=aseq(0,rows(wy),1)
  {xc,c,or}=conv(xb,yb,wx,wy)
  s=c[,1]
  t=c[,2]
  nc=rows(c)
;
  if (p==0)
    s1 = s
  endif
  if (p.>0)
    i=1
    while (i.<= p)
      wy=wy.*g
      sym=i-2.*floor(i./2)
      {xc,c,or}=conv(xb,yb,wx,wy,sym)
      s=s~c[,1]
      t=t~c[,2]
      i=i+1
    endo
    while (i.<= 2*p)
      wy=wy.*g
      sym=i-2.*floor(i./2)
      {xc,c,or}=conv(xb,yb[,1],wx,wy,sym)
      s=s~c
      i=i+1
    endo
    s1=s[,1:(p+1)]
    i=2
    while (i.<= p+1)
      s1=s1|s[,i:(p+i)]
      i=i+1
    endo
  endif
  is=#(aseq(0,p+1,nc)*matrix(1,nc))
  is=is+sort(#(aseq(1,nc,1)*matrix(1,p+1)))
;
  s1=index(s1, is)
  s1= s1'
  reshape("s1",#(p+1,p+1,nc))
  s1 = inv(s1)
  reshape("s1",#(p+1,(p+1)*nc,1))
  s = s1'
;
  is=#(aseq(1,p+1,1)*matrix(1,nc))
  s=paf(s, is.=1)
  mh=sum(s.*t,2)
  mh=(d.*xc+d.*0.5)~mh
endp