proc (mh)=lregxestp(x,h,K,v)   
; -----------------------------------------------------------------
; Library        smoother
; -----------------------------------------------------------------
;  See_also      regxestp lpregxest lregestp 
; -----------------------------------------------------------------
;   Macro        lregxestp
; -----------------------------------------------------------------
;   Keywords     kernel smoothing, kernel regression, 
;                local linear estimator
; -----------------------------------------------------------------
;   Description  estimates a multivariate regression function using 
;                local polynomial kernel regression with
;                Quartic kernel.
; -----------------------------------------------------------------
;   Notes        This function does an exact computation, i.e.
;                requires O(n^2) operations for estimating the 
;                regression function on all observations. For
;                p>2 this is usually faster than the WARPing method.
; -----------------------------------------------------------------
;   Reference    Fan and Gijbels (1995): Local Polynomial Fitting
; -----------------------------------------------------------------
;   Usage        mh = lregxestp(x {,h {,K} {,v} })
;   Input
;     Parameter  x  
;     Definition   n x (k+1), the data. In the first p columns 
;                  the independent variables, in the last column 
;                  the dependent variable.
;     Parameter  h 
;     Definition   scalar or k x 1 vector, bandwidth. If not 
;                  given, 20% of the volume of x[,1:k] is used.
;     Parameter  v  
;     Definition   m x p, values of the independent variable on 
;                  which to compute the regression. If not given, 
;                  a grid of length 100 (k=1), length 30 (k=2)
;                  and length 8 (k=3) is used in case of k<4. When 
;                  k>=4 then v is set to x.
;   Output
;     Parameter  mh  
;     Definition   n x (k+1) or m x (k+1) matrix, the first k columns 
;                  contain the grid or the sorted  x[,1:k], the 
;                  second column contains the regression estimate  
;                  on the values of the first k columns.
; -----------------------------------------------------------------
;   Example   library("smoother") 
;             library("plot")
;             ;
;             x = 2.*pi.*(uniform(200,2)-0.5)  ; independent variable
;             m = sum(cos(x),2)                ; true function
;             e = uniform(200)-0.5             ; error term             
;             x = x~(m+e)                             
;             ;
;             mh = lregxestp(x,2)              ; estimate function
;             mh = setmask(mh,"surface","blue")
;             plot(x,mh)                       ; surface plot                  
;             setgopt(plotdisplay,1,1,"title","ROTATE!")
; -----------------------------------------------------------------
;   Result    The local linear regression estimate (blue) using   
;             Quartic kernel and bandwidth h=2 and the data are
;             pictured.
; -----------------------------------------------------------------
;   Author    Marlene Mueller, 990413
; -----------------------------------------------------------------
  n=rows(x)
  k=cols(x)-1
  ix=min(x[,1:k])
  mx=max(x[,1:k])
  rx=mx-ix
  one=matrix(k)
;
  if (exist(h)==0)
    h=((0.2)^(1/k)).*rx
  endif
  if (cols(h)>1)
    h=h'
  endif
  if (cols(h)==1)
    h=h.*matrix(k)
  endif
;
  x=sort(x,1:k)
  y=x[,k+1]
  x=x[,1:k]
;
  if (exist(v)==0)
    switch
      case (k==1)
        v=grid(ix',rx'/99,100*one)
        break
      case (k==2)
        v=sort(grid(ix',rx'/29,30*one),1:k)
        break
      case (k==3)
        v=sort(grid(ix',rx'/7,8*one),1:k)
        break
      default
        v=x
        break
    endsw  
  endif
;
  mh=locpol(x, v, y, h, k-1) 
  mh=v~mh[,1]
endp


