proc(net)=nnrnet (x, y, weights, size, param, wts)
; -----------------------------------------------------------------------
; Library      nn
; -----------------------------------------------------------------------
; See_also     nnrpredict ann
; -----------------------------------------------------------------------
; Macro        nnrnet
; -----------------------------------------------------------------------
; Description  trains a one hidden layer feed forward network. The
;              optional parameter param consists of 8 values. Boolean
;              values for linear output, entropy error function, log
;              probability models and for skip connections.
;              The fifth value is the maximum value for the starting 
;              weights, the sixth the weight decay, the seventh the 
;              number of maximal iterations and the last value generates
;              some output if equal to one.
; -----------------------------------------------------------------------
; Usage        net=nnrnet (x, y, weights, size{, param {, wts}})
; Input
;   Parameter  x
;   Definition n x p matrix input variables 
;   Parameter  y
;   Definition n x q matrix output variables
;   Parameter  weights
;   Definition n x 1 vector of weights
;   Parameter  size
;   Definition scalar       number of hidden units
;   Parameter  param
;   Definition 7 x 1 vector of parameters
;   Parameter  wts
;   Definition vector       of predefined weights 
; Output
;   Parameter  net.n       
;   Definition 3 x 1 vector number of input, hidden and output units
;   Parameter  net.nunits   
;   Definition scalar 
;   Parameter  net.nconn
;   Definition vector
;   Parameter  net.conn
;   Definition vector
;   Parameter  net.decay
;   Definition scalar       weight decay parameter
;   Parameter  net.entropy
;   Definition scalar     
;   Parameter  net.softmax
;   Definition scalar     
;   Parameter  net.value
;   Definition scalar       value of error function 
;   Parameter  net.wts
;   Definition vector       of weights    
;   Parameter  net.yh
;   Definition n x q        estimated y's 
; -----------------------------------------------------------------------
; Note        Tracing is as default equal to 1. Check in the output 
;             window if the network specified is the network you
;             want to use. Wrong parameter setting will cause strange
;             results!
;
;             To improve the optimization algorithm it is strongly
;             suggested to standardize the data, either by normalizing
;             them z = (x-mean(x))/sqrt(var(x)) or uniformize them by
;             z = (x-min(x))/(max(x)-min(x))   
; -----------------------------------------------------------------------
; Example     library("nn")
;             x = read("bank2")
;             y = (1:rows(x)).>100
;             x = (x-min(x))./(max(x)-min(x))
;             par = 0|1|0|1|0.7|1.0e-3|1500|0
;             net = nnrnet (x, y, matrix(rows(x)), 10, par)
; -----------------------------------------------------------------------
; Result      runs a neural network with 10 hidden units for the swiss
;	      banknote data (1 forged banknote, 0 genuine banknote).
; -----------------------------------------------------------------------
; Reference   Venables and Ripley (1994). Modern Applied Statistics with
;             S-Plus. p. 261-266
; -----------------------------------------------------------------------
; Author      Sigbert Klinke, 970523, 981207
; -----------------------------------------------------------------------
  error (rows(x)<>rows(y), "nnetr : rows(x)<>rows(y)")
  linout  = 0
  entropy = 0
  softmax = 0
  skip    = 0
  rang    = 0.7
  decay   = 0
  maxit   = 100
  tracing = 1
  if (exist("param")==1)
    linout  = param[1]
    switch 
      case (rows(param).>1)
        entropy = param[2]
      case (rows(param).>2)
        softmax = param[3]
      case (rows(param).>3)
        skip    = param[4]
      case (rows(param).>4)
        rang    = param[5]
      case (rows(param).>5)
        decay   = param[6]
      case (rows(param).>6)
        maxit   = param[7]
      case (rows(param).>7)
        tracing   = param[8]
    endsw
  endif
  error (linout && entropy, "nnrnet: entropy fit only for logistic units")
  error (size<1, "nnrnet: #hidden units<1")
  if (softmax)
    linout  = 1
    entropy = 0
  endif
  n      = cols(x)|size|cols(y)
  nunits = 1+sum(n)
  nconn  = 0.*matrix(1+nunits)
  conn   = 0
;; norm
  nc = cumsum(n)
  {nconn, conn}=addr(nunits, nconn, conn, 1:nc[1], (1+nc[1]):nc[2])
  {nconn, conn}=addr(nunits, nconn, conn, (1+nc[1]):nc[2], (1+nc[2]):nc[3])
;; end of norm
  if (skip)
    {nconn, conn}=addr(nunits, nconn, conn, 1:n[1], (1+nc[2]):(nunits-1))
  endif
  conn = conn[1:rows(conn)-1]
  if (linout)
    nsunits = nunits-n[3]
  else
    nsunits = nunits
  endif
  if (tracing)
    ts = string ("A %.0f - %.0f - %.0f network:", n[1],n[2],n[3])
    ts = ts|string ("# weights     : %.0f", rows(conn))
    if (linout)
      ts = ts|"linear output : yes"
    else
      ts = ts|"linear output : no"
    endif
    if (entropy)
      ts = ts|"error function: entropy"
    else
      ts = ts|"error function: least squares"
    endif
    if (softmax)
      ts = ts|"log prob model: yes"
    else
      ts = ts|"log prob model: no"
    endif
    if (skip)
      ts = ts|"skip links    : yes"
    else
      ts = ts|"skip links    : no"
    endif
    ts = ts|string("max. weight   : %.2f", rang)
    ts = ts|string("decay         : %g", decay)
    ts = ts|string("max. Iterat   : %.0f", maxit)
    ts
  endif
  if (exist("wts")<>1)
    wts = rang*(2*uniform(rows(conn))-1)
  endif
  error (rows(wts)<>rows(conn), "nnrnet: weights vector of incorrect length")
  nnrsetnet (n|nsunits|entropy|softmax|decay, nconn, conn)
  ntr  = rows(x)
  nout = cols(y)
  if (exist("weights")<>1)
    weights = matrix(ntr)
  endif
  error ((rows(weights)<>ntr) || sum(weights<0), "nnrnet: invalid weights vector")
  nnrsettrain (ntr, x~y, weights)
  {wts, value} = nnrdovm(rows(wts)|maxit|tracing, wts)
  yh  = nnrtest(ntr|nout, x~y, wts)
  net = list(n, nunits, nsunits, nconn, conn, decay, entropy, softmax, value, wts, yh)
endp
