proc(output)=pancoint(z,d,m,lag,T)
; ---------------------------------------------------------------------
; Library     metrics
; ---------------------------------------------------------------------
; See_also    panunit
; ---------------------------------------------------------------------
; Macro       pancoint
; ---------------------------------------------------------------------
; Description Computes FM-OLS estimates for cointegration equations with homogenous cointegration parameters and heterogeneous short-run dynamics and deterministics
;		
; ---------------------------------------------------------------------
; Usage       (output)=pancoint(z,d,m,lag, {T} )
; Input       
; Parameter   z	   
; Definition     NT x {2+} k  data matrix
; Parameter   d    
; Definition     d=0 for no deterministics, =1 for a constant, =2 for trend
; Parameter   m    
; Definition     truncation lag for the kernels used for the FM-OLS
; Parameter   lag  
; Definition     number of lagged differences for the ADF test on the residuals
; Parameter   T    
; Definition     Common number of time periods for balanced panel (optional)
; Output      
; Parameter   
; Definition     string with output table
; ---------------------------------------------------------------------
; Notes       
; ---------------------------------------------------------------------
; Example     library("metrics")
              z=read("dynpanel")
	      pancoint(z,1,6,1)	
; ---------------------------------------------------------------------
; Result      
;[ 1,] "====================================================="
;[ 2,] "FM-OLS for homogenous panels (Pedroni/Phillips/Moon) "
;[ 3,] "====================================================="
;[ 4,] "beta[ 1 ]=          1.0494        0.1436        7.306"
;[ 5,] "beta[ 2 ]=        -0.99501        0.1849       -5.382"
;[ 6,] "====================================================="
;[ 7,] "Number of groups (N):                30              "
;[ 8,] "Total number of obs. (NT):          600              "
;[ 9,] "Truncation lag:                       4              "
;[10,] "====================================================="
;[11,] "                                                     "
;[12,] "    Unit-Root tests applied to the OLS-residuals     "
;[13,] "====================================================="
;[14,] "Pooled Dickey-Fuller Regression:  1'th variable      "
;[15,] "====================================================="
;[16,] "PARAMETERS        Estimate     robust SE      t-value"
;[17,] "====================================================="
;[18,] "Lag[1]=            -0.3736       0.0622        -6.002"
;[19,] "Delta[ 1]=         -0.3322       0.0355        -9.364"
;[20,] "const=              0.0767       0.0671         1.144"
;[21,] "====================================================="
;[22,] "N*T=    600      N=   30           With constant     "
;[23,] "Unit root statistics:                                "
;[24,] "STATISTIC      Value  crit. Value (5%)  mean variance"
;[25,] "====================================================="
;[26,] "B/M (1994)     -5.792     -1.65       0.000     1.000"
;[27,] "L/L (1993)     -8.123     -1.65      -0.565     0.877"
;[28,] "mod. L/L      -14.115     -1.65       0.000     1.000"
;[29,] "I/P/S (1997)  -10.942     -1.65      -1.488     0.772"
;[30,] "====================================================="
; ---------------------------------------------------------------------
; Keywords   Cointegration, FM-OLS, Unit root tests
; ---------------------------------------------------------------------
; Reference   Kao and Chung (1997) working paper, Syrakrus University.
; ---------------------------------------------------------------------
; Link        
; ---------------------------------------------------------------------
; Author      breitung
; ---------------------------------------------------------------------
  NT=rows(z)
  k=cols(z)  
 
  if (exist(T))  

    if (T>0) 
      N=NT/T
      error(N-floor(N)>0,"  The data set is not balanced (NT/N is not an integer)")
      error(T<2,         "  T must be at least 2")
      ind=(1:N)  
      ind=kron(ind,matrix(T,1))
      z=ind~z
      k=k+1
    endif
  else
    it=z[,1:2]
    z=z[,1]~z[,3:k]
    k=k-1 
	endif	 

; Eingabeanalyse

    error(d>2,         "  d must be at most 2")
    error(d<0,         "  d must be at least 0")


;                 Pooled Regression

  sxx=0
  sxy=0
  N=0
  pos=1
  while(pos < NT-1) 
    ind=z[pos,1]
    yi=z[pos,2:k]
    pos=pos+1
    while (z[pos,1]==ind)
      yi=yi|z[pos,2:k]
      pos=pos+1
    endo
    Ti=rows(yi)

    y=yi[,1]
    x=yi[,2:k-1]
    ct=matrix(Ti,1)
    if (d==2)
      ct=ct~#(1:Ti)
    endif
    if (d>0)
      yadj=y-ct*inv(ct'*ct)*ct'*y
      xadj=x-ct*inv(ct'*ct)*ct'*x
    endif

    sxx=sxx+xadj'*xadj
    sxy=sxy+xadj'*yadj
    N=N+1
  endo 

  beta=inv(sxx)*sxy
  uvec=matrix(NT,1)
  sxx=0
  sxy=0
  pos=1
  while(pos < NT-1) 
    Ti0=pos 
    ind=z[pos,1]
    yi=z[pos,2:k]
    pos=pos+1
    while (z[pos,1]==ind)
      yi=yi|z[pos,2:k]
      Ti1=pos
      pos=pos+1
    endo
    Ti=rows(yi)
    
    y=yi[,1]  
    x=yi[,2:k-1]

    if (d>0) 
      x=x~matrix(Ti,1)
      if (d==2)
        x=x~#(1:Ti)
      endif
    endif
    uvec[Ti0:Ti1]=y-x[,1:k-2]*beta

    if (d>0)
      ct=x[,k-1:k-2+d]
      xadj=x[,1:k-2]-ct*inv(ct'*ct)*ct'*x[,1:k-2]
      yadj=y-ct*inv(ct'*ct)*ct'*y
    else
      xadj=x
      yadj=y 
    endif 
    dx=xadj[2:Ti,]-xadj[1:Ti-1,]  
    u=yadj-xadj*beta
    e =u[2:Ti]~dx
    sig = e'*e
    lam = 0 
    j = 1
    while (j<=m)
      lam = lam + (e[1:Ti-1-j,]'*e[1+j:Ti-1,])*(m-j+1)/(m+1)
      j=j+1
    endo
    delta = sig + lam
    omega = sig + lam + (lam') 

    p=rows(omega)
    omstar=omega[1,1]-omega[1,2:p]*inv(omega[2:p,2:p])*omega[2:p,1]
    g=inv(omega[2:p,2:p]'*omega[2:p,2:p])*omega[2:p,2:p]'*omega[2:p,1]
    delg=delta[2:p,1] - delta[2:p,2:p]*g
                                               ;delg = delg |(0*matrix(k1+1,p1))
    ystar=yadj[2:Ti] - dx*g
    xadj=xadj[2:Ti,]

    sxx=sxx+xadj'*xadj/omstar
    sxy=sxy+((xadj'*ystar) - delg)/omstar
    bfmols=inv(sxx)*sxy
    se=sqrt(xdiag(inv(sxx)))
    tstat=bfmols./se
    
            
;    sg = (omega[1,1] - omega[1,2:p]*g)/(Ti-1);
;    v = xxi*sg;
;    se = sqrt(diag(v))

  betastr=string("beta[%2.0g ]= %15.5g", 1:rows(bfmols), bfmols)	
  betasestr=string("   %11.4g",se)
  tvalstr=string(" %12.3f", tstat)  
  parametertext=betastr+betasestr+tvalstr
    
  endo 

    line1="FM-OLS for heterogenous panels (Kao and Chiang 1997) "
    line2="=====================================================" 
    line3="PARAMETERS        Estimate     robust SE      t-value"    
    line4=string("Number of groups (N):      %12.0f",N) +"              "
    line5=string("Total number of obs. (NT): %12.0f",NT)+"              "
    line6=string("Truncation lag:            %12.0f",m) +"              "    
    line7="                                                     "
    line8="    Unit-Root tests applied to the OLS-residuals     "
    output1=line2|line1|line2|parametertext|line2|line4|line5|line6|line2|line7|line8
    if (exist(T))  
       output2=panunit(uvec,1,lag,d,T)
    else
       output2=panunit(it~uvec,1,lag,d)
    endif  
    output=output1|output2
endp 

