proc(output,bgmm)=pandyn(z,p,IVmeth,T) 
; ---------------------------------------------------------------------
; Library     metrics
; ---------------------------------------------------------------------
; See_also    panfix, panrand, panhaus
; ---------------------------------------------------------------------
; Macro       pandyn
; ---------------------------------------------------------------------
; Description Estimaties dynamic panel data model with GMM 
; ---------------------------------------------------------------------
; Usage       (output,beta)=pandyn(z,p,IVmeth {,T})
; Input        
; Parameter   z       
; Definition          NT x {2}+1+k data matrix
; Parameter   p       
; Definition          Autoregressive order for dependent variable 
; Parameter   IVmeth  
; Definition          Indicator for the Instruments 
;                     setting IV-meth=0 lets program choose the IV-method
; Parameter   T       
; Definition          Common time periods for balanced panel (optional)
; Output      
; Parameter   output  
; Definition          Output table (string)
; Parameter   beta    
; Definition          (k+p) x 1  vector of coefficient estimates
; ---------------------------------------------------------------------
; Notes       See Arellano and Bond (1991) for a method description
; ---------------------------------------------------------------------
; Example     z=read("dynpanel.dat")   ; read data set
;             {output,b}=pandyn(z,1,0) ; compute first step estimator
;             output                   ; print output
; ---------------------------------------------------------------------
; Result      
; [ 1,] "====================================================="
; [ 2,] "GMM-Estimation of the Dynamic Fixed-Effect Model:    "
; [ 3,] "y(i,t) = y(i,t-1)'gamma + x(i,t)'beta + a(i) + e(i,t)"
; [ 4,] "====================================================="
; [ 5,] "PARAMETERS        Estimate        SE          t-value"
; [ 6,] "====================================================="
; [ 7,] "beta[ 1 ]=          0.9591       0.0704        13.625"
; [ 8,] "beta[ 2 ]=         -1.0949       0.0771       -14.203"
; [ 9,] "Lag [ 1 ]=          0.7643       0.0490        15.609"
; [10,] "====================================================="
; [11,] "N*T=    375      N=   25      DF=    12              "
; [12,] "R-square (differenced model):    0.5955              "
; [13,] "Hansen's J-statistic (p-val):   18.4000 (0.1041)     "
; [14,] "Instruments according to method: 2                   "
; [15,] "====================================================="
; ---------------------------------------------------------------------
; Keywords    GMM, dynamic panel data model 
; ---------------------------------------------------------------------
; Reference   Arellano and Bond (1991)
; ---------------------------------------------------------------------
; Link        
; ---------------------------------------------------------------------
; Author      Joerg Breitung (000524)
; ---------------------------------------------------------------------

  NT=rows(z)   
  k=cols(z)  
  if (exist(T)) 
    Tmin=1
    Tmax=T
  else  
    Tmin=min(z[,2])
    Tmax=max(z[,2])
  endif

; Eingabeanalyse

    error(IVmeth>4,       "  IVmeth must be smaller than 4")

  zy=0
  zx=0
  zwz=0
  yy=0
  xy=0
  xx=0
  sx=0
  sy=0
  sxx=0
  sxy=0
  ssy=0
  ssx=0
  syy=0
  sT=0
                          ; balanced case
  if (exist(T)) 
    N=NT/T
    error(N-floor(N)>0,"  The data set is not balanced (NT/N is not an integer)")
    error(T<2,         "  T must be at least 2")
    T0=T-p-1
      error(T0<2,"  T must be at least p+2")
    range0=T0
    range1=T0
    if (IVmeth==0)
      ivn1=T0*(T0+1)/2 + (k-1) + N/10
      ivn2=T0 + T0*(k-1) + N/10
      ivn3=T0+k-1+N/10
      IVmeth=4
      if (ivn1>N)
        IVmeth=3
        if (ivn2>N)
           IVmeth=2
           if (ivn3>N)
              IVmeth=1
           endif
        endif
      endif   
    endif


    w=0*matrix(T0,T0)
    w[1,1:2]=2~(-1) 
    w[T0,T0-1]=-1
    w[T0,T0]=2
    i=2
    while (i<=T0-1)
      w[i,i-1:i+1]=(-1)~2~(-1)
      i=i+1;
    endo
                         
    i=1 
    while (i<=N)
      i0=(i-1)*T+1
      i1=i*T
      yi=z[i0:i1,1] 
      xi=z[i0:i1,2:k] 
      dxi=xi[p+2:T,]-xi[p+1:T-1,]

     if (IVmeth==1) 

       zi=yi[p:T-2]
       j=2
       while (j<=p)
         zi=zi~yi[p-j+1:T-j-1]
         j=j+1
       endo
       dyi=yi[p+2:T]-yi[p+1:T-1]
       dxi=xi[p+2:T,]-xi[p+1:T-1,]
       zi=zi~(xi[p+1:T-1,]-xi[p:T-2,])~dxi
       zi=zi'
     endif


      if (IVmeth==2) 

        zi=(matrix(T0,1)*yi[p])
        j=p+1
        while (j<=T-2)
          z0=(0*matrix(j-p,1))|(matrix(T-j-1,1)*yi[j])
          zi=zi~z0
          j=j+1
        endo
        dyi=yi[p+2:T]-yi[p+1:T-1]
        dxi=xi[p+2:T,]-xi[p+1:T-1,]
        zi=zi|(dxi')
      endif

      if (IVmeth==3)  

        zi=(matrix(T0,1)*yi[p])
        j=p+1
        while (j<=T-2)
          z0=(0*matrix(j-p,1))|(matrix(T-j-1,1)*yi[j])
          zi=zi~z0
          j=j+1
        endo 
        dyi=yi[p+2:T]-yi[p+1:T-1]
        dxi=xi[p+2:T,]-xi[p+1:T-1,]

        ein=unit(T0)
        z0=matrix((k-1)*T0,T0)
        j=1
        while (j<=T0)
          z0[,j]=kron(ein[,j],(dxi[j,]'))
          j=j+1
        endo
        zi=zi|z0 
      endif 

      if (IVmeth==4)

        zi=yi[1:p]~(0*matrix(p,T0-1))  
        j=p+1
        while (j<=T-3)
          z0=yi[1:j]~(0*matrix(j,T-j-2))
          z0=(0*matrix(j,j-p))~z0
          zi=zi|z0
          j=j+1
        endo
        z0=(0*matrix(T-2,T-2-p))~yi[1:T-2]
        zi=zi|z0
        zi=zi|(dxi')
        dyi=yi[p+2:T]-yi[p+1:T-1]
        dxi=xi[p+2:T,]-xi[p+1:T-1,]
      endif


      j=1
      while (j<=p)
        dxi=dxi~(yi[p+2-j:T-j,]-yi[p+1-j:T-j-1,])
        xi=xi[p+1:T,]~yi[p-j+1:T-j]
        j=j+1
      endo 

      sxy=sxy+dxi'*dyi
      syy=syy+dyi'*dyi
      sxx=sxx+dxi'*dxi
      sy=sy+sum(dyi)
      sx=sx+sum(dxi)' 

      zy=zy+zi*dyi 
      zx=zx+zi*dxi
      zwz=zwz+zi*w*zi'
      sT=sT+T0
 
    i=i+1 
    endo
    k=k-1
    
   else 

                           ; unbalanced case
     T=Tmax-Tmin+1
     T0=T-p-1
     N=NT/T 
     range0=Tmax
     range1=Tmin
     if (IVmeth==0)
       ivn1=T0*(T0+1)/2 + (k-3) + N/10
       ivn2=T0 + T0*(k-3) + N/10
       ivn3=T0+k-3+N/10
       IVmeth=4
       if (ivn1>N)
         IVmeth=3
         if (ivn2>N)
            IVmeth=2
            if (ivn3>N)
              IVmeth=1
           endif
         endif
       endif   
     endif
    
     w=0*matrix(T0,T0)
     w[1,1:2]=2~(-1)
     w[T0,T0-1]=-1
     w[T0,T0]=2
     i=2
     while (i<=T0-1)
       w[i,i-1:i+1]=(-1)~2~(-1)
       i=i+1;
     endo

     pos=1
     N=0
     while (pos < NT-2) 
       yi=0*matrix(1,1)
       xi=0*matrix(1,k-3)    
       dyi=0*matrix(2,1) 
       dxi=0*matrix(2,k+p-3)
       ctl=dyi
       dvec=0*matrix(T,1)
       Ti=0
       year=Tmin 
       while (year<=Tmax)
         if (z[pos,2]==year)
           dvec[year-Tmin+1]=1 
           yi=yi|z[pos,3]
           xi=xi|z[pos,4:k]
           pos=pos+1
         else  
           yi=yi|(0*z[pos,3])
           xi=xi|(0*z[pos,4:k])
         endif
         
         if (year>=Tmin+p+1)
           d=0
           m=rows(yi)
           if (sum(dvec[m-p-2:m-1])==p+2)           
              d=1
           endif
           ctl=ctl|d
           dyi=dyi|(d*(yi[m]-yi[m-1]))
           dxi=dxi|(d*((xi[m,]-xi[m-1,])~(yi[m-1:m-p]-yi[m-2:m-p-1])' ))
           Ti=Ti+d
         endif
  
       year=year+1
       endo

       yi=yi[2:T+1]
       xi=xi[2:T+1,]
       dyi=dyi[3:rows(dyi)]
       dxi=dxi[3:rows(dxi),]
       ctl=ctl[3:rows(ctl)]
                                       
       if (IVmeth==1)  

         zi=yi[p:T-2] 
         j=2
         while (j<=p)
           zi=zi~yi[p-j+1:T-j-1]
           j=j+1
         endo
         zi=zi~(xi[p+1:T-1,]-xi[p:T-2,])~dxi[,1:k-3] 
         zi=zi'
       endif

       if (IVmeth==2)   

         zi=(matrix(T0,1)*yi[p])
         j=p+1
         while (j<=T-2)
           z0=(0*matrix(j-p,1))|(matrix(T-j-1,1)*yi[j])
           zi=zi~z0
           j=j+1
         endo

         zi=zi|(dxi[,1:k-3]')
       endif

       if (IVmeth==3)   

         zi=(matrix(T0,1)*yi[p])
         j=p+1
         while (j<=T-2)
           z0=(0*matrix(j-p,1))|(matrix(T-j-1,1)*yi[j])
           zi=zi~z0
           j=j+1
         endo

         ein=unit(T0)
         z0=matrix((k-3)*T0,T0)
         j=1
         while (j<=T0)
           z0[,j]=kron(ein[,j],(dxi[j,1:k-3]'))
           j=j+1
         endo
         zi=zi|z0 
       endif

       if (IVmeth==4)   

         z0=yi[1:p]~(0*matrix(p,T-p-2))
         zi=z0   
         i=p+1
         while (i<=T-3) 
           z0=yi[1:i]~(0*matrix(i,T-i-2))
           z0=(0*matrix(i,i-p))~z0
           zi=zi|z0
           i=i+1
         endo
         z0=(0*matrix(T-2,T-2-p))~yi[1:T-2]
         zi=zi|z0
         zi=zi|(dxi[,1:k-3]')
       endif
         
       Ti=Ti-1      
       range0=min(range0|Ti)
       range1=max(range1|Ti)
       y=yi[p+1:T]
       x=xi[p+1:T,]      
       zi=(zi'.*ctl)'
       sxx=sxx+dxi'*dxi
       sxy=sxy+dxi'*dyi
       syy=syy+dyi'*dyi
       sy=sy+sum(dyi)
       sx=sx+sum(dxi)' 
       zy=zy+zi*dyi  
       zx=zx+zi*dxi
       zwz=zwz+zi*w*zi'
       yy=yy+dyi'*dyi
       xy=xy+dxi'*dyi
       xx=xx+dxi'*dxi
       sT=sT+sum(ctl)
       N=N+1
     endo       
     k=k-3
   endif 

     error(rows(zwz)>N,"  number of instruments exceeds N")
   zwz=inv(zwz)
   {r,im,v}=eisrg(zx'*zwz*zx)
   condind=log(max(r)/min(r)) 
   bgmm=inv(zx'*zwz*zx)*zx'*zwz*zy
   sige=syy-2*bgmm'*sxy+bgmm'*sxx*bgmm
   sige=sige/2/sT 

   SEbgmm=sige*inv(zx'*zwz*zx) 
   SEbgmm=sqrt(xdiag(SEbgmm))	
 
   m=zy-zx*bgmm
   J=m'*zwz*m/sige
;                          computation of R2 as correlation coefficient

   cov=bgmm'*(sxy-sx*sy/sT)/sT
   var1=bgmm'*(sxx-sx*sx'/sT)*bgmm/sT
   var2=(syy-sy^2/sT)/sT
   R2=cov^2/var1/var2

   df=rows(zx)-k-p 
   Jpval=1-cdfc(J,df)  
   
   betastr=string("beta[%2.0f ]= %15.4f", 1:k, bgmm[1:k])	
   betastr=betastr|string("Lag [%2.0f ]= %15.4f", 1:p, bgmm[k+1:k+p])	
   betasestr=string("     %8.4f",SEbgmm)
   tvalues=bgmm./SEbgmm
   tvalstr=string("  %12.3f", tvalues)     
   DFstr  =      string("N*T=%7.0f",NT)
   DFstr  =DFstr+string("      N=%5.0f",N)
   DFstr  =DFstr+string("      DF=%6.0f              ",df)
   R2str  =string("R-square (differenced model):    %5.4f              ",R2)  
   Jstr   =string("Hansen's J-statistic (p-val):%10.4f ",J)   
   Jstr   =Jstr+string("(%5.4f)     ",Jpval)
   METHstr=string("Instruments according to method: %1.0f                   ",IVmeth)

//TEXTOUTPUT	

  line1="GMM-Estimation of the Dynamic Fixed-Effect Model:    "
  line1a="y(i,t) = y(i,t-1)'gamma + x(i,t)'beta + a(i) + e(i,t)"
	line2="=====================================================" 
	line3="PARAMETERS        Estimate        SE          t-value"    

	parametertext=betastr+betasestr+tvalstr 
	output=line1|line1a|line2|line3|line2|parametertext|line2
  output=line2|output|DFstr|R2str|Jstr|METHstr|line2
  if (condind > 10)
     output=output|"WARNING:  Moments are poorly identified (weak instruments)"
  endif  
endp  
 

