proc(d)=robwhittle(y,bdvec)
; ----------------------------------------------------------------------
; Library 	  times
; ----------------------------------------------------------------------
;  See_also	   roblm, lobrob, sclm, gph, lo, kpss, rvlm,  pgram, 
;----------------------------------------------------------------------
;   Macro	  robwhittle
; ----------------------------------------------------------------------
;   Description   Semiparametric Gaussian estimator of the degree of long memory
;                 of a time series, based on the Whittle estimator. The first 
;                 argument is the series, the second argument is the vector of 
;                 bandwidths, i.e., the number of frequencies after zero that are
;                 considered. 
;                 By default, the bandwidth vector m = n/4, n/8, n/16, where
;                 n is the sample size.
;                 This quantlet displays the estimated parameter d, with the number
;                 of frequencies considered.
; ----------------------------------------------------------------------
;   Usage         d = robwhittle(x{,bdvec})
;   Input 
;     Parameter   x 
;     Definition  vector
;     Parameter   bdvec
;     Definition  vector of bandwidths
;     Output
;     Parameter   d
;     Definition  vector
; ----------------------------------------------------------------------
;   Example     ;nonparametric estimation of degree of long-memory in volatility
;               ;Since no vector of bandwidths is provided, the default vector of 
;               ;bandwidth parameter is used. 
;               library("times")
;               x = read("dmus58.dat")
;               x=x[1:1000]
;               y = abs(tdiff(x)) 
;               d = robwhittle(y)
;               d
; ----------------------------------------------------------------------
;   Result           
;                 Contents of d
;
;               [1,] "     d     Bandwidth"
;               [2,] "_____________________"
;               [3,] ""
;               [4,] "  0.0982      250"
;               [5,] "  0.1200      125"
;               [6,] "  0.0805       62"
;
; ----------------------------------------------------------------------
;   Example     ;nonparametric estimation of degree of long-memory in volatility
;               ;In this case, the vector of bandwidths m is provided 
;               library("times")
;               x = read("dmus58.dat")
;               x=x[1:1000]
;               y = abs(tdiff(x)) 
;               m = #(50,100,150)
;               d = robwhittle(y,m)
;               d
; ----------------------------------------------------------------------
;   Result
;                 Contents of d

;
;               [1,] "     d     Bandwidth"
;               [2,] "_____________________"
;               [3,] ""
;               [4,] "  0.0669       50"
;               [5,] "  0.0940      100"
;               [6,] "  0.1269      150"
;
; ----------------------------------------------------------------------
;   Reference
;     Robinson, P.M. (1995): Gaussian Semiparametric Estimation of 
;     Long-Range Dependence, The Annals of Statistics, 23, 1630-1661.
;
; ----------------------------------------------------------------------
;   Keywords
;      long memory
; ----------------------------------------------------------------------
;   Author        Gilles Teyssiere, 980423, revised 981210
; ----------------------------------------------------------------------
  ;; This part of the code is from the macro pgram
  ;; written by Rong Chen and Christian Hafner
  opt = 0 ; tests the presence of the optional matrix of bandwidth parameters
  if (exist(bdvec)==1)
    opt=1   
  endif
  ;;
  n = rows(y) 
  y = y-mean(y)
  p = log(n) / log(2)
  p = rint(exp(log(2)*ceil(p))) ; next power of 2 
  if (n != p)
    y = y|(matrix(p-n)-1)       ; append series 
  endif
  y = y~(matrix(p).*0)
  ftr = fft(y)                  ; fourier transform
  I = (sum((ftr.*ftr)')/p)'     ; periodogram
  ind=2:(floor(n/2))
  I = I[ind]
  lambda = ind *2*pi/n
  ;;
  putglobal("I")
  putglobal("lambda")
  ;;   
  if (opt == 0)
    imax = 3
    range = #(rint(n/4), rint (n/8), rint (n/16)); default bandwidths
  else
    imax =  rows(bdvec)
    range = bdvec
  endif 
  dest = matrix(imax)
  i = 1
  while (i <= imax)
    error(range[i] > n/2,"Bandwidth greater than the Nyquist frequency")      
    mbandwidth = rint(range[i]); in case of error in the bandwidth
    putglobal("mbandwidth")     
    dm = bfgs(0.0,"fwhit","dfwhit",100)
    dest[i] = dm[1]
    i = i+1  
  endo
  ;;
  d = dest
  ;;Graphical display of results
	line1 ="     d     Bandwidth"
  line2 ="_____________________"
  st1 = string("%8.4f",dest[,1])
  st2 = string("%9.0f",range[,1])
	st3 = st1+st2
  d = line1|line2|""|st3
endp
;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; Whittle approximate likelihood function
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
proc (vf) = fwhit(d)
  ;;
  I = getglobal("I")
  lambda = getglobal("lambda")
  mbandwidth = getglobal("mbandwidth")
  ;;
  wI = I[1:mbandwidth]
  wlambda = lambda[1:mbandwidth]
  plambda = wlambda^(2*d)
  ;;
  sum1 = log(mean(wI.*plambda))
  sum2 = mean(log(wlambda))*2*d
  vf = sum1 - sum2;  
endp
;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; Gradient of Whittle approximate likelihood function
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
proc (dvf) = dfwhit(d)
  ;;
  I = getglobal("I")
  lambda = getglobal("lambda")
  mbandwidth = getglobal("mbandwidth")
  ;;
  wI = I[1:mbandwidth]
  wlambda = lambda[1:mbandwidth]
  plambda = wlambda^(2*d)
  sum1 = 2*mean(wI.*log(wlambda).*plambda)/(mean(wI.*plambda))
  sum2 = mean(log(wlambda))*2
  dvf = sum1 - sum2
endp
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; end of macro robwhittle
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
