proc(med,depths) = simdep(x, mem)
; ----------------------------------------------------------------------------
; Library      stats
; ----------------------------------------------------------------------------
;   See_also    median
; ----------------------------------------------------------------------------
;   Macro       simdep
; ----------------------------------------------------------------------------
;   Description Computes the simplicial depth estimate of location
; ----------------------------------------------------------------------------
;   Keywords    median, robust estimate of location
; ----------------------------------------------------------------------------
;   Usage       {med,depths} = simdep(x{, mem})
;
;   Input
;     Parameter   x  
;     Definition        n x p matrix (n observations of dimension p)
;
;     Parameter   mem
;     Definition        optional parameter that affects the amount of memory
;                       used by the routine; the values belongs to interval (0,1),
;                       where 0 represents the minimal memory requirements 
;                       and the lowest speed of computation,
;                       and 1 the maximal memory requirements and highest speed. 
;                       Default value is 1.
;                       The higher is the dimension of observations (p),
;                       the bigger is the effect of the maximal memory setting
;                       (mem = 1).
;
;   Output
;     Parameter   med
;     Definition        p x 1 vector containing the estimate of location
;
;     Parameter   depth 
;     Definition        n x 1 vector containing for every observation the 
;                       number of simpleces covering it
; ----------------------------------------------------------------------------
;   Example   library("stats")
;             ; 
;             ; simulate data
;             ;
;             randomize(101)
;             x = uniform(100)
;             ;
;             ; estimate the location by simdep
;             ;
;             {med,d} = simdep(x)
;             med
;             ;
;             ; estimate the location by median
;             ;
;             median(x)
; ----------------------------------------------------------------------------
;   Result    There are two results confirming that the simplicial depth 
;             estimator is equivalent to median for one dimensional data:
;             Contents of med
;             [1,]  0.48251 
; 
;             Contents of med
;             [1,]  0.48251 
; ----------------------------------------------------------------------------
;   Example   library("stats")
;             library("xplore")
;             ; 
;             ; simulate data
;             ;
;             randomize(10)
;             x = uniform(30,2)
;             ;
;             ; estimate the location by simdep
;             ;
;             {med,dep} = simdep(x)
;             med
;             ;
;             ; draw a depth graph and median
;             ;
;             d = createdisplay(1,1)
;             dat = x
;             col = round(5*dep/max(dep))
;             setmaskp(dat, col, 3, 8)
;             median = med'
;             setmaskp(median,4,12,15)    ; median is red big star
;             show(d, 1, 1, dat, median)
;             setgopt(d, 1, 1, "title", "Simplicial depth")
; ----------------------------------------------------------------------------
;   Result    There are two types of output. First, in the output window, the
;             following estimate appears:
;             Contents of med
;             [1,]  0.53705
;             [2,]  0.33484
;
;             Second, there is also a graph showing the estimate as a big read
;             star and all data points as circles of different colors. The color
;             is given to every point according to its depth, i.e., the number of
;             simpleces that covers it. So, points with the lowest depths are
;             green, points with higher depths are blue, and so on.
; ----------------------------------------------------------------------------
;   Author   Pavel Cizek, 2000-05-04
; ----------------------------------------------------------------------------
;
; load dll
  if (getenv("os") == "unix")
    handle = dlopen("ssd.so")
  else
    handle = dlopen("ssd.dll")
  endif
;
; check parameter existence and their values
  error(exist(x) <> 1, "Missing data matrix!")
  p = cols(x)
  n = rows(x)
  
  if (exist(mem) <> 1)
    mem = 1
  endif

; prepare parameters for call
;EXTERN void EXPORT SSD(double *x_, double *n_, double *p_, double *med_, 
;                       double *telim_, double *numtempelim_, double *counts_,
;                       double *intbuf)
;// RETURN VALUE:
;// med - pointer to the vector of length p (dimension)
;//
;// x - matrix p x n, each column contains one observation
;// n - number of observations
;// p - number of dimensions
;// telim - memory for numtempelim temporary elimination matrices of size p x (n+p)
;// numtempelim - actual number of supplied temporary elimination matrices
;//               has to be at least 1!
;// counts - array to hold counts of simpleces covering points (length n)
;// intbuf - temp storage for different integer arrays (length (2+numtempelim) * p) 
  numtemp = 1 + mem * (p+2)
  if (numtemp > p + 2)
    numtemp = p + 2
  endif
  if (numtemp > 30)
    numtemp = 30
  endif

; x ... matrix p x n
  x = x'
; med ... vector p x 1
  med = 0 * matrix(p)
; telim matrices ... numtemp * p * (n+p)
  telim = 0 * matrix(numtemp * p * (n+p))
; depths ... n 
  depths = 0 * matrix(n)
; intbuf ... (2+numtempelim) * p
  intbuf = 0 * matrix((2 + numtemp) * p)

; call dll
  if (getenv("os") == "unix")
    ret = dlcall(handle, "SSD", x,n,p,med,telim,numtemp,depths,intbuf)
  else
    ret = dlcall(handle, "_SSD", x,n,p,med,telim,numtemp,depths,intbuf) 
  endif
  
; close dll
  dlclose(handle)
;end
endp

