proc(summ)=summarize(x,xvars)
; ----------------------------------------------------------------------------
; Library       stats
; ----------------------------------------------------------------------------
;  See_also     descriptive frequency mean var min max quantile 
; ----------------------------------------------------------------------------
;   Macro       summarize
; ----------------------------------------------------------------------------
;   Description  provides a short summary table (min, max, mean, median
;                standard error) for all columns of a data matrix.
;                An additional vector of name  strings can be given 
;                to identify columns by names.
; ----------------------------------------------------------------------------
;   Usage        summ=summarize(x{,xvars})
;   Input
;     Parameter   x  
;     Definition        n x p  matrix, the columns correspond to the variables.
;     Parameter   xvars
;     Definition        optional, p x 1  string vector, variable names. Note,
;                       that only up to 15 characters are used.
;   Output
;     Parameter   summ
;     Definition        string output, containing summary table.
; ----------------------------------------------------------------------------
;   Notes     summarize will first remove all missing and infinity
;             values. Summary statistics is only produced for
;             the remaining observations. For a separate analysis
;             of each variable use the macro descriptive.
; ----------------------------------------------------------------------------
;   Example   library("stats") 
;             x=read("bank2")
;             xvars="length"|"l_height"|"r_height"|"u_frame"|"l_frame"|"diagonal"
;             summarize(x,xvars)
; ----------------------------------------------------------------------------
;   Result     
;             Contents of summ
;             [ 1,] "          "
;             [ 2,] "              Minimum     Maximum        Mean      Median   Std.Error"
;             [ 3,] "         -------------------------------------------------------------"
;             [ 4,] "length          213.8       216.3       214.9       214.9     0.37655"
;             [ 5,] "l_height          129         131      130.12       130.2     0.36103"
;             [ 6,] "r_height          129       131.1      129.96         130     0.40407"
;             [ 7,] "u_frame           7.2        12.7      9.4175         9.1      1.4446"
;             [ 8,] "l_frame           7.7        12.3       10.65        10.6     0.80295"
;             [ 9,] "diagonal        137.8       142.4      140.48       140.3      1.1523"
;             [10,] " "
; ----------------------------------------------------------------------------
;   Author    Marlene Mueller, 970903
; ----------------------------------------------------------------------------
  if (exist(x)==9)
    x=x{1}
  endif
  if (exist(xvars)>0)
    if (cols(xvars)>1)
      xvars=reshape(xvars, cols(xvars)|rows(xvars))
    endif
  endif
;   
  error(rows(dim(x))>2,"first argument must be vector or matrix")
  error(exist(x)!=1,"first argument must be numeric")
;
  nmiss=0
  n=rows(x)
  if (max(prod(isNumber(x)==1,2))==1)
    x=paf(x,prod(isNumber(x)==1,2))
    nmiss=n-rows(x)
  else
    nmiss=n
  endif
;
  if (nmiss<n)
    x=(0*matrix(rows(x),cols(x))).*(x==0)+x.*(x!=0)
;
    if (exist(xvars)>0)
      error(exist(xvars)!=2,"variable names should be strings!")
      error(rows(dim(xvars))>1,"variable names have wrong dimension")
      error(rows(xvars)!=cols(x),"variable names have wrong dimension")
    else
      xvars=string("Variable %2.0f",1:cols(x))
    endif
;
    summ=min(x)'~max(x)'~mean(x)'~quantile(x,0.5)'~sqrt(var(x))'
;
    j=0
    while (j<cols(summ))
      j=j+1
      if (j==1)
        summtxt=string(" %11.5g",summ[,1])
      else
        summtxt=summtxt+string(" %11.5g",summ[,j])
      endif
    endo
;
    summ=     ("     Minimum"|"------------")
    summ=summ+("     Maximum"|"------------")
    summ=summ+("        Mean"|"------------")
    summ=summ+("      Median"|"------------")
    summ=summ+("   Std.Error"|"-------------")
    summ=" "|summ|summtxt
;
    if (exist(xvars)==2)
      if ( (rows(xvars)==cols(x))&&(dim(dim(xvars))==1) )
        xvars=(" ")|(" ")|(" ")|xvars
        xvars=xvars+"                       "
        xvars=substr(xvars,1,16)
;
        blank=string(" ",1:rows(xvars))
        j=16
        while (j>1)
          j=j-1
          sub=substr(xvars,j,j+1)
          if (prod(sub==blank))
            xvars=substr(xvars,1,j)
          else
            j=1
          endif
        endo
;
        summ=xvars+" "+summ
      endif
    endif
    summ=summ|" "
    if (nmiss>0)
      summ=summ|string(" %1.0f",nmiss)+" observation(s) containing {-Inf,Inf,NaN} excluded."|" "
    endif
  else
    summ=" "|" All observations contain {-Inf,Inf,NaN}."|" "
  endif
endp
