/*

	PISEEK.C

	Search pi=3.14159... for a given search pattern
	or read the sequence at a given position
	It uses pi files of 10 million digits with 1 digit per byte and nothing else in them.
	The pi files are named <pidir>/piNN.dat (N = 01..NMAX)
	First pi file: pi01.dat, starts with 1415...

Hae/99-04-12
*/

/*
	Pattern:
	Positive Examples:  314         finds 314
			                314??9?     finds 3141592
			                314*1?**6   finds 31415926 but also 
	Negative Examples:  ?14         starts not with digit
                      314*        ends not with digit
			                3*4*5*6*7*8 more than 5 parts
			                $A;         unallowed characters

	Pattern := {digit * ?}...

	A search pattern may consist of up to 5 (=PARTS_MAX) digit-parts
	with interspersed *?-parts.
	It must start and end with a digit-part.

	A digit-part consists of one or more digits.
	A *?-part consists of one or more ? and/or *

	A ? means a single unspecified digit
	A * means 0 to 10 (= STAR_SPAN) unspecified digits
	Using more than one adjacent * prolongs the unspecified digits each by factor 10.

	A Pattern must be not longer than 20 (=PATLEN_MAX) characters.
	In its expanded form (10 chars per *) it must be not longer than 99 (=PATLEN_EXP_MAX) characters.

*/

#define PARTS_MAX 		   5 // max. no of digit-parts of the search pattern
#define STAR_SPAN       10 // max. size of a span defined by *
#define PATLEN_MAX      20 // max length of an unexpanded search pattern
#define PATLEN_EXP_MAX  99 // max total length of an expanded search pattern
#define DIGITS          16 //  no of digits printed if action=rp

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <unistd.h>
#ifdef __DJGPP__
  #include <pc.h>  	// kbhit()
#else
int kbhit(void) 
{
    return 0;
}

#endif

#include "myutils.h"
#include "pifile.h"   // fopen_pi, fread_pi, fseek_pi
                      // extern decl of the var's of pifile.c    

#define FALSE		0
#define TRUE		!FALSE

char  	acBuf[20];

// global data
char    pat[PATLEN_MAX+1],
			  pat_intern[PATLEN_MAX+1];
char    cmd_user[20];
int 	  bDone = 0;
long    digitsmax = 250000000L;  // default size of pi file set

// values of each pattern part
char    *p_dp     [PARTS_MAX];  // address of digit-part
int     len_digits[PARTS_MAX];  // no of digits of digit-part
int     len_qm    [PARTS_MAX];  // no of ? in *?-part
int     len_star  [PARTS_MAX];  // (no of *) times STAR_SPAN in *?-part

int     nparts;                 // no of pairs of digit- and *?-parts
size_t 	r0;                     // no of bytes at bufbeg from previous read
size_t  r;                      // no of bytes recently read
char    *pBufEnd;               // == acBuf+strlen(acBuf)
long 	  nPos = 0;               // filepos at pPos
char 	  *pPos;	      	        // address in acBuf where nPos points
char    *pFound;                // address in acBuf of matching pattern
int     patlen_exp;             // total

int  	  scanpattern(void);
void 	  search(void);
int 	  getinput(int argc, char *argv[]);
int 	  usage(void);
int     getpifile(void);

int main(int argc, char *argv[])
{
  strcpy(pidir, "../");  // default dir of the pifiles piNN.dat: above current

		// main loop
	while (!bDone)
	{
		if (getinput(argc, argv) == FALSE)
			continue;
		getpifile(); // open and check pi file and fill acBuf
		if (nPos != fseek_pi(nPos)) {
			printf("Position %ld not available.\n", nPos);
			continue;
		}
		r0 = 0;
		r = fread_pi(acBuf, sizeof(acBuf));
		pBufEnd = acBuf + r;
		pPos = acBuf;
		if (*cmd_user == 's') { // search
			search();
		} else { // read from position
			printf("Position: %ld\n", nPos);
			printf("Sequence: %.*s\n", DIGITS, pPos);
		}
		++nPos;
	}
	return 0; // end of program
}

/*******************************************************************************
*
* Function:            scanpattern()
*
* scanpattern checks the user search pattern and
* transforms it in a internal format and fills
* the parts arrays.
*
* Each digit-part is ended by an '\0'.
* For the i-th part, len_digits[i], len_qm[i] and len_star[i] are
* set to its span length.
*
* Returns 0 if wrong, else !0
*
* E.g.: user pattern = "98?765*??*45"
* is transformed to "98" "" "765" "??*" "45"
* and arrays are set to:
*   nparts  p_dp  len_digits len_qm len_star
*  ---------------------------------
*   0      "98"   2
*   1      "765"  3      1
*   2      "45"          2      20
*  ---------------------------------
*
*******************************************************************************/

int scanpattern(void)
{
	char 		now,
				  prev = 'x';
	char		*p = pat_intern;
	int      i;

	if (*pat == '\0')
	  strcpy(pat, "314159");
	strcpy(p, pat);
	nparts = *len_digits = 0;
	for ( ; *p; ++p) { // scan char by char of the user pattern
		now = *p;
		if (isdigit(now)) { // a digit starts or continues a digit-part
			++len_digits[nparts];
			if (prev != 0)  
			    p_dp[nparts] = p;
			prev = 0;
		} else { 
			if (prev == 0) {
				if (++nparts >= PARTS_MAX) {
					printf("Pattern has too many digit parts.\n");
					return FALSE;
				}
				len_digits[nparts] = len_qm[nparts] = len_star[nparts] = 0;
				*p = '\0';
			}
			prev = now;
			if (now == '?') ++len_qm[nparts];
			else
			if (now == '*') len_star[nparts] += STAR_SPAN;
			else {
				printf("Pattern contains invalid characters.\n");
				return FALSE;
			}
		}
	}
	for (patlen_exp = i = 0; i <= nparts; ++i)
		patlen_exp += len_digits[i] + len_qm[i] + len_star[i];
	if (patlen_exp > PATLEN_EXP_MAX
			|| patlen_exp > sizeof(acBuf)-2) {
		printf("Pattern expands too large.\n");
		return FALSE;
	}
	// first and last part must be digit-part
	if (len_qm[0] || len_star[0] || len_digits[nparts] ==  0) {
		printf("Pattern does not begin or end with a digit.\n");
		return FALSE;
	}
	return TRUE;
}

/*******************************************************************************
*
* Function: search()
*
* Searches for a pattern match.
* The user pattern was parsed and dispatched in scanpattern().
* First step is searching for the first digitpart of pattern.
* Then the remaining parts are searched for.
*
* Returns : TRUE or exits the program
*
*******************************************************************************/

void search(void)
{
	size_t mm1 = len_digits[0] - 1;
	char   *q, *ptmp;
	char   qsave;
	int    i;

	// use a dummy pattern if one is missing
	if (*pat == '\0') {
		strcpy(pat, "314159");
		scanpattern();
	}
	// search starts with a full buffer
	while (r > 0) {	// while not found && not fileend
		// search first digit-part
		pFound = strstr(pPos, p_dp[0]);
		if (pFound == NULL) { // not in this buffer load
			// move last mm1 digits to buffer beg
			memmove(acBuf, pBufEnd-mm1, r0=mm1);
			nPos += pBufEnd-mm1-pPos;
			pPos = acBuf;
			// read
			r = fread_pi(acBuf+r0, sizeof(acBuf)-r0);
			pBufEnd = acBuf+r0+r;
			if (kbhit()) {
				printf("Search aborted.\n");
				r = 0;
			}

			continue;
		}
 		// first digit-part found
		// assure max. length in acBuf
		nPos += pFound - pPos;
		pPos = pFound;
		if (pBufEnd-pPos < patlen_exp) {
			r0 = (size_t)(pBufEnd-pPos);
			memmove(acBuf, pPos, r0);
			r = fread_pi(acBuf+r0, sizeof(acBuf)-r0);
			pBufEnd = acBuf+r0+r;
			pFound = pPos = acBuf;
		}
		// search remaining parts
		for (i=1; i <= nparts; ++i) {
			// increment by length of prev. digit part and no of ?'s
			pPos += len_digits[i-1] + len_qm[i];
			nPos += len_digits[i-1] + len_qm[i];
			// if no * ask current pos for the next digit part
			if (len_star[i] == 0) {
				if (strncmp(pPos, p_dp[i], len_digits[i]) != 0)
					break; // not found, try whole pattern search again
			} else {
				// else search for the next digit part
				// among the no of * times STAR_SPAN plus
				// the length of the next digit part
				q = pPos + len_star[i] + len_digits[i];
				// insert a \0
				qsave = *q;
				*q = '\0';
				ptmp = strstr(pPos, p_dp[i]);
				// restore
				*q = qsave;
				if (ptmp == NULL)
					break; // not found, try whole pattern search again
				nPos += ptmp - pPos;
				pPos = ptmp;
			}
		}
		if (i > nparts)
			break; // pattern matches
		// set back to pFound+1
		nPos -= (pPos - pFound - 1);
		pPos = pFound + 1;
	} // end while not found
	if (r != 0) {
		// match
		i = (int)(pPos - pFound + len_digits[nparts]);
		if (nparts > 0)
			printf("Pattern : %s\n", pat);
		printf("Sequence: %*.*s\n", i, i, pFound);
		nPos -= (pPos - pFound);
		printf("Position: %03ld %03ld %03ld\n"
				 , nPos/1000000L, (nPos/1000)%1000, nPos%1000);
		pPos = pFound;
	} else {
		printf("Pattern : %s does not match any sequence before\n", pat);
		printf("Position: %03ld %03ld %03ld\n"
				 , nPos/1000000L, (nPos/1000)%1000, nPos%1000);
		nPos = -1;
	}
	return;
}

/*******************************************************************************
*
* Function: getinput()
*
* Requests the next command from the user.
* Either from the cmdline or from console.
* Handles cases: quit, help, and wrong input
*
* Returns : TRUE
*
*******************************************************************************/

int getinput(int argc, char *argv[])
{
	static int 		bCmdlineread = 0;
         int    bIsConsole  = (isatty(fileno(stdin)) != 0);
	static int 		bFirst = TRUE;
			  char  	inbuf[400];
			  int     d;
			  char   *p;
			  int     bOK = FALSE;
			  char    arg[40] = "";
  
	while (!bOK && !bDone) {
    *cmd_user = '\0';
	  if (!bCmdlineread && argc > 1) { // get input from commandline
	    // ask pidir option
	    for (d=1; d < argc; ++d)
	    {
		    if (   strncasecmp(argv[d], "-d", 2) == 0
		    		|| strncasecmp(argv[d], "/d", 2) == 0)
		    {
		      sscanf(argv[d]+2, "%s", pidir);
		      break;
		    }
	    }
			if (argc >= 2 && d >= 2) strcpy(cmd_user, argv[1]);
			if (argc >= 3 && d <  2) strcpy(cmd_user, argv[2]);
			if (argc >= 3 && d >= 3) strcpy(arg, argv[2]);
			if (argc >= 4 && d <  3) strcpy(arg, argv[3]);
			if (*cmd_user)
				bDone = 1;
			bCmdlineread = 1;
		} else {
			// request user input from stdin
			if (bFirst && bIsConsole) {
			  bFirst = FALSE;
				usage();
      } 
			//	Current search position is nPos
      if (bIsConsole)
        printf("\nEnter command: sb [<pat>] | sc [<pat>] | rp <pos> | h | q :\n\n");
			fgets(inbuf, sizeof(inbuf),stdin);
      if (!bIsConsole)
        fprintf(stdout, "\n%s", inbuf);
      inbuf[strlen(inbuf)-1] = '\0'; // eliminate \n
      sscanf(inbuf, "%s", cmd_user);
      for (p=cmd_user; *p; ++p)
        *p = tolower(*p);
			p = inbuf + strlen(cmd_user);
			strcpy(arg, p);
			if (!*cmd_user)  {
				printf("No command\n");
				continue;
			}
		}
		if (*cmd_user == 'q')
			exit(0);
		if (*cmd_user == 'h' || *cmd_user == '?') {
			usage();
			continue;
		}
		if (*arg) {
			// eliminate blanks
			while ((p = strchr(arg, ' ')) != NULL)
				memmove(p, p+1, strlen(p));
		}
		if (strcmp(cmd_user, "rp") == 0) {
			if (!*arg) {
				printf("Missing readposition\n");
				continue;
			}
			nPos = atol(arg);
			return TRUE;
		}
		// sb or sc
		if (strcmp(cmd_user, "sb") && strcmp(cmd_user, "sc")) {
			printf("Wrong command\n");
			continue;
		}
		if (*arg)
			strcpy(pat, arg);
		if (scanpattern() == FALSE)
			continue; // error msg was issued in scanpattern()
		if (strcmp(cmd_user, "sb") == 0)
			nPos = 0; // search from begin
		bOK = TRUE;
	} // end while
	return bOK;
}

/*******************************************************************************
*
* Function: getpifile()
*
* If not yet done:
* Opens all files of the pi file set in order to
* find fno_last and the total no of digits.
* Exits the program when no file or wrong file
* Returns : TRUE or exits the program
*
*******************************************************************************/

int getpifile(void)
{
	int fno;

  if (fno_last >= fno_first)
   	return TRUE;
  digitsmax = 0;
  
  for (fno=fno_first; 1; ++fno) {
    if (fopen_pi(fno) == NULL)
   		break;
	  digitsmax += fsize[fno];
  }
  if (fno_last == -1)
   	progexit(1, "Missing pi file %s.\n"
      	"No sense to continue.\n", fn_pi);
	return TRUE;
}

/*******************************************************************************
*
* Function: usage()
*
* Prints a help text
*
* Returns : TRUE
*
*******************************************************************************/

int usage(void)
{
	printf("\n");
	printf("         piseek 1.00 -- Search a sequence of pi = 3.14159...\n"
			 "Usage:   piseek <command> [-d<pidir>]\n"
			 "Example: piseek sb 987??2*10\n"
			 "\n<commands>\n"
			 "  sb [<pat>]: search <pat> (or previous pattern) from begin of file\n"
			 "  sc [<pat>]: search <pat> (or previous pattern) from current file position\n"
			 "  rp <pos>  : read 16 digits from position <pos>\n"
			 "  h         : help\n"
			 "  q         : quit\n"
			);
	printf("\n");
	printf("<pat>  : sequence of digits, * and ?, must have a digit at both ends.\n");
	printf("         a digit is meant literally,\n"
			 "         a ? means a single unspecified digit,\n"
			 "         a * means 0 to 10 unspecified digits.\n");
	printf("<pos>  : decimal number <= 	%03ld %03ld %03ld\n"
				 , digitsmax/1000000L, (digitsmax/1000)%1000, digitsmax%1000);
	printf("<pidir>: directory of the pi files (pi01.dat, pi02.dat etc.). Default=./\n");
	return TRUE;
}

