#include "ap.h"


// The bigint functions
// Bigints are simply arrays of unsigned integers, they can be seen as
// base-2^24 numbers. Arithmetic on them is quite fast.
// The least significant word is stored first.

const double base = 16777216.0;                 // 2^24
const double invbase = 1.0 / base;              // 2^-24
const double halfbase = 0.5 * base;             // 2^23

// Adds n words from s to d, returns overflow carry bit
rawtype bigadd (rawtype *d, rawtype *s, size_t n, size_t c)
{
    size_t t;
    double r;
    rawtype b = 0.0;

    for (t = 0; t < n; t++)
    {
        r = (double) *d + (double) *s + (double) b;

        if (r >= base)
        {
            r -= base;
            b = 1.0;
        }
        else b = 0.0;

        *d = r;

        d++;
        s++;
    }


    for (t = 0; t < c && b != 0.0; t++)
    {
        if ((r = (double) *d + (double) b) == base)
            r -= base;
        else
            b = 0.0;

        *d = r;
        d++;
    }

    return b;
}

// Subtracts n words s from d, returns subtract carry bit (1)
rawtype bigsub (rawtype *d, rawtype *s, size_t n, size_t c)
{
    size_t t;
    double r;
    rawtype b = 0;

    for (t = 0; t < n; t++)
    {
        r = (double) *d - (double) *s - (double) b;

        if (r < 0.0)
        {
            r += base;
            b = 1.0;
        }
        else b = 0.0;

        *d = r;

        d++;
        s++;
    }

    for (t = 0; t < c && b != 0.0; t++)
    {
        if ((r = (double) *d - (double) b) == -1.0)
            r += base;
        else
            b = 0.0;

        *d = r;
        d++;
    }

    return b;
}

// Multiplicates n words in s by f, stores result in d, returns overflow word
rawtype bigmul (rawtype *d, rawtype *s, rawtype f, size_t n)
{
    size_t t;
    rawtype b = 0;
    double r;

    for (t = 0; t < n; t++)
    {
        r = (double) *s * (double) f + (double) b;

        b = quick_floor (invbase * (r + 0.5));

        *d = r - base * (double) b;

        d++;
        s++;
    }

    return b;
}

// Divides n words in s by f, stores result in d, returns remainder
rawtype bigdiv (rawtype *d, rawtype *s, rawtype f, size_t n)
{
    size_t t;
    rawtype b;
    double r, invf = 1.0 / (double) f;

    if (!n) return 0;

    d += n;
    s += n;

    if ((b = *(s - 1)) < f)
    {
        d--;
        s--;
        n--;
        *d = 0;
    }
    else b = 0;

    for (t = 0; t < n; t++)
    {
        d--;
        s--;

        r = base * (double) b + (double) *s;

        *d = quick_floor (invf * (r + 0.5));

        b = r - (double) *d * (double) f;                    // b = r % f;
    }

    return b;
}

// Shifts s right one bit to d, returns carry bit
int bigshr (rawtype *d, rawtype *s, size_t n)
{
    size_t t;
    rawtype tmp1, tmp2;
    int b = 0;

    if (!n) return 0;

    d += n;
    s += n;

    for (t = 0; t < n; t++)
    {
        d--;
        s--;

        tmp1 = *s * 0.5;
        tmp2 = quick_floor (tmp1);
        *d = tmp2 + (b ? halfbase : 0);
        b = (tmp1 == tmp2 ? 0 : 1);
    }

    return b;
}

// Compares n words, returns -1 if d < s, 1 if d > s, 0 if d == s
int bigcmp (rawtype *d, rawtype *s, size_t n)
{
    size_t t;

    d += n - 1;
    s += n - 1;

    for (t = 0; t < n; t++)
    {
        if (*d < *s) return -1;
        if (*d > *s) return 1;

        d--;
        s--;
    }

    return 0;
}
