
#include <math.h>
#include "fxtaux.h"

/*
 A Duhamel-Hollman split-radix dif fft
 Ref: Electronics Letters, Jan. 5, 1984
 Complex input and output data in arrays x and y
 Length is n.

 by: Dave Edelblute, edelblut@cod.nosc.mil, 05 Jan 1993
 Modified: R. Mayer
 Modified: Joerg Arndt 
*/


#define USE_SINCOS3 
// whether sincos is used for 3*angle
// else: use algebraic relation



void 
duhamel_fft(double *x, double *y, ulong ldn, int is)
{
    x--;  // fortran convention := on 
    y--;

    const ulong n = (1<<ldn);

    if(n==1)  return;

    ulong n2 = n+n;

    for(ulong k=1; k<=ldn; k++)
    {
        n2 >>= 1;
        ulong n4 = n2>>2;

        double e = 2.0*M_PI/n2;
        double a = 0.0;

        for(ulong j=1; j<=n4; j++)
        {
	    double cc1,ss1, cc3,ss3;
            sincos(&cc1,&ss1,a);

#if defined USE_SINCOS3 
            sincos(&cc3,&ss3,3.0*a);
#else
            cc3=4.0*cc1*(cc1*cc1-0.75); 
            ss3=4.0*ss1*(0.75-ss1*ss1);
#endif

            a = j*e;
            ulong ix = j;

            ulong id=(n2<<1);

            while( ix<n )
            {
                for(ulong i0=ix; i0<n; i0=i0+id)
                {
                    ulong i1=i0+n4;
                    ulong i2=i1+n4;
                    ulong i3=i2+n4;

		    double r1,r2,s1,s2,s3;

                    r1 = x[i0]-x[i2];
                    x[i0] += x[i2];

                    r2 = x[i1]-x[i3];
                    x[i1] += x[i3];

                    s1 = y[i0]-y[i2];
                    y[i0] += y[i2];

                    s2 = y[i1]-y[i3];
                    y[i1] += y[i3];

                    s3 = r1-s2;
                    r1 += s2;

                    s2 = r2-s1;
                    r2 += s1;

                    x[i2] =  r1*cc1-s2*ss1;
                    y[i2] = -s2*cc1-r1*ss1;

                    x[i3] =  s3*cc3+r2*ss3;
                    y[i3] =  r2*cc3-s3*ss3;
               }
               
               ix = (id<<1)-n2+j;
               id <<= 2;
            }
        }
    }


    //------------------ last stage, length=2 butterfly -----------
    ulong ix = 1;
    ulong id = 4;

    while( ix<n )
    {
        for(ulong i0=ix; i0<=n; i0+=id)
        {
            ulong i1=i0+1;

	    sumdiff2(x[i0],x[i1]);
	    sumdiff2(y[i0],y[i1]);
        }

        ix = (id<<1)-1;
        id <<= 2;
    }


    x++;  // fortran convention := off
    y++;

    scramble(x,y,n);

    if( is>0 )   // backtransform
    {
        for(ulong k=1; k<n/2; ++k)  SWAP(x[k],x[n-k]);

        for(ulong k=1; k<n/2; ++k)  SWAP(y[k],y[n-k]);
    }
}
//========================== end DUHAMEL_FFT ===============================

