// operations with arrays of doubles and LIMBs

#include <math.h>
#include <stdlib.h>
#include <iostream.h>
#include <iomanip.h>
#include <assert.h>


#include "auxid.h"
#include "mybuiltin.h"  // for MIN


// set array to zero:
void
d_null(double *dst, ulong n)
{
    while ( n-- )  dst[n] = 0;
}
//=============== end D_NULL ==============

void
i_null(LIMB *dst, ulong n)
{
    while ( n-- )  dst[n] = 0;
}
//=============== end I_NULL ==============


// fill array with one value:
void
d_fill(double *dst, ulong n, double d)
{
    while ( n-- )  dst[n] = d;
}
//=============== end D_FILL ==============

void
i_fill(LIMB *dst, ulong n, LIMB d)
{
    while ( n-- )  dst[n] = d;
}
//=============== end I_FILL ==============


/*  memcpy doc:
  void * memcpy (void *TO, const void *FROM, size_t SIZE)
  undefined if the two arrays TO and FROM overlap;

  void * memmove (void *TO, const void *FROM, size_t SIZE)
  ... even if those two blocks of space overlap.
*/
// copy array:
void
d_copy(const double *src, double *dst, ulong n)
{
    //    while ( n-- )  dst[n] = src[n];
    memcpy(dst,src,n*sizeof(double));
}
//=============== end D_COPY ==============

void
i_copy(const LIMB *src, LIMB *dst, ulong n)
{
    //    while ( n-- )  dst[n] = src[n];
    memcpy(dst,src,n*sizeof(LIMB));
}
//=============== end I_COPY ==============

void
id_copy(const LIMB *src, double *dst, ulong n)
{
    while ( n-- )  dst[n] = (double)src[n];
}
//=============== end ID_COPY ==============

void
di_copy(const double *src, LIMB *dst, ulong n)
{
    while ( n-- )  dst[n] = (LIMB)src[n];
}
//=============== end DI_COPY ==============


// copy as much as makes sense
// from src[] (length ns) to dst[] (length nd):
void
d_copy(const double *src, ulong ns, double *dst, ulong nd)
{
    ulong k,n;

    n = MIN(nd,ns);
    for (k=0; k<n; ++k)  dst[k] = src[k];

    if ( nd>n )  for ( ; k<nd; ++k)  dst[k] = 0;
}
//=============== end D_COPY ==============

void
i_copy(const LIMB *src, ulong ns, LIMB *dst, ulong nd)
{
    ulong k,n;

    n = MIN(nd,ns);
    for (k=0; k<n; ++k)  dst[k] = src[k];

    if ( nd>n )  for ( ; k<nd; ++k)  dst[k] = 0;
}
//=============== end I_COPY ==============

void
id_copy(const LIMB *src, ulong ns, double *dst, ulong nd)
{
    ulong k,n;
    
    n = MIN(nd,ns);
    for (k=0; k<n; ++k)  dst[k] = (double)src[k];

    if ( nd>n )  for ( ; k<nd; ++k)  dst[k] = 0;
}
//=============== end ID_COPY ==============

void
di_copy(const double *src, ulong ns, LIMB *dst, ulong nd)
{
    ulong k,n;

    n = MIN(nd,ns);
    for (k=0; k<n; ++k)  dst[k] = (LIMB)src[k];

    if ( nd>n )  for ( ; k<nd; ++k)  dst[k] = 0;
}
//=============== end DI_COPY ==============


ulong
i_first(const LIMB *a, ulong n)
// search first LIMB != 0
{
    ulong k = 0;
    while ( (k<n) && (a[k]==0) )  k++;
    return k;
}
//=============== end I_FIRST ==============

ulong
i_first_val(const LIMB *a, ulong n, LIMB v)
// search first LIMB == v
{
    ulong k = 0;
    while ( (k<n) && (a[k]!=v) )  k++;
    return k;
}
//=============== end I_FIRSTVAL ==============

ulong
i_first_nonval(const LIMB *a, ulong n, LIMB v)
// search first LIMB != v
{
    ulong k = 0;
    while ( (k<n) && (a[k]==v) )  k++;
    return k;
}
//=============== end I_FIRSTVAL ==============


ulong
s_first(const char *a, long n)
// return index of first char != blank
// returns n if only blanks
{
    long k = 0;
    while ( k<n )
    {
        if ( (a[k]!=' ') && (a[k]!='\t') )  break;
        k++;
    }
    return k;
}
//=============== end S_FIRST ==============

ulong
s_last(const char *a, long n)
// return index of last char != blank
// returns -1 if only blanks
{
    n--;
    while ( n>=0 )
    {
        if ( (a[n]!=' ') && (a[n]!='\t') )  break;
        n--;
    }
    return n;
}
//=============== end S_LAST ==============


void
d_shift_left(double *c, ulong n)
//
// shift (by one position)
// towards more significant digits
//
{
    for (ulong k=0; k<n-1; ++k)  c[k] = c[k+1];
    c[n-1] = 0;
}
//================ end D_SHIFT_LEFT ===============


void
i_shift_left(LIMB *c, ulong n)
//
// shift (by one position) 
// towards more significant digits
//
{
    for (ulong k=0; k<n-1; ++k)  c[k] = c[k+1];
    c[n-1] = 0;
}
//================ end I_SHIFT_LEFT ===============


void
d_shift_right(double *c, ulong n)
//
// shift (by one position)
// towards less significant digits
//
{
    for (ulong k=n-1; k!=0; --k)  c[k] = c[k-1];
    c[0] = 0;
}
//================ end D_SHIFT_RIGHT ===============


void
i_shift_right(LIMB *c, ulong n)
//
// shift (by one position)
// towards less significant digits
//
{
    for (ulong k=n-1; k!=0; --k)  c[k] = c[k-1];
    c[0] = 0;
}
//================ end I_SHIFT_RIGHT ===============


void
d_shift_left(double *c, ulong n, ulong sh)
//
// shift (by sh positions)
// towards more significant digits
//
{
    if ( sh>=n )
    {
        while ( n-- )  *(c++) = 0;
        return;
    }

    ulong k;
    for (k=0; k<n-sh; ++k)  c[k] = c[k+sh];

    for ( ; k<n; ++k)  c[k] = 0;
}
//================ end D_SHIFT_LEFT ===============


void
i_shift_left(LIMB *c, ulong n, ulong sh)
//
// shift (by sh positions)
// towards more significant digits
//
{
    if ( sh>=n )
    {
        while ( n-- )  *(c++) = 0;
        return;
    }

    ulong k;
    for (k=0; k<n-sh; ++k)  c[k] = c[k+sh];

    for ( ; k<n; ++k)  c[k] = 0;
}
//================ end I_SHIFT_LEFT ===============


void
d_shift_right(double *c, ulong n, ulong sh)
//
// shift (by sh positions)
// towards less significant digits
//
{
    if ( sh>=n )
    {
        d_null(c,n);
        return;
    }

    ulong k;
    for (k=n-1; k>=sh; --k)  c[k] = c[k-sh];

    do  { c[k]=0; }  while ( k-- );
}
//================ end D_SHIFT_RIGHT ===============


void
i_shift_right(LIMB *c, ulong n, ulong sh)
//
// shift (by sh positions)
// towards less significant digits
//
{
    if ( sh>=n )
    {
        i_null(c,n);
        return;
    }

    ulong k;
    for (k=n-1; k>=sh; --k)  c[k] = c[k-sh];

    do  { c[k]=0; }  while ( k-- );
}
//================ end I_SHIFT_RIGHT ===============


void
i_bit_shift_left(LIMB *c, ulong n, ulong sh, int rxbits)
//
// shift (by sh bits)
// towards more significant digits
// shift must be < rxbits*n
//
{
    ulong k,sl,sb;
    ulong t,nine;

    sl = sh/rxbits;
    if ( sl )  i_shift_left(c,n,sl);

    sb = sh-sl*rxbits;
    if ( sb==0 )  return;

    n -= sh;

    nine = (1<<(rxbits-1))-1;
    t = 0;
    for (k=0; k<n; ++k)
    {
	t |= (int)c[k]<<sh;
	c[k] = t&nine;
	t >>= rxbits;
    }
}
//================ end I_BIT_SHIFT_LEFT ===============


void
i_bit_shift_right(LIMB *c, ulong n, ulong sh, int rxbits)
//
// shift (by sh bits)
// towards less significant digits
// shift must be < rxbits*n
//
{
    ulong k,sl,sb;
    ulong t,cy,nine;

    sl = sh/rxbits;
    if ( sl )  i_shift_right(c,n,sl);

    sb = sh-sl*rxbits;
    if ( sb==0 )  return;

    n -= sh;
    c += sh;

    nine = (1<<(rxbits-1))-1;
    t = 0;
    for (k=n-1; k>=1; --k)
    {
	t |= c[k];
	cy = t&1;
	c[k] = t>>sb;
	t = cy<<rxbits;
    }
}
//================ end I_BIT_SHIFT_RIGHT ===============



void
d_rand(double *a, ulong n)
{
//    srand(time(0));

    while ( n-- )  a[n] = (double)rand()*(1.0/RAND_MAX);
}
// ================== end D_RAND ===================


void
i_rand(LIMB *a, ulong n, int nine)
{
//    srand(time(0));

    int rn = (RAND_MAX/nine);

    while ( n-- )  a[n] = (LIMB)((int)rand()/rn);
}
// ================== end I_RAND ===================


void
d_print(char *what, const double *ar, ulong n)
{
    cout << what;

    for (ulong k=0; k<n; ++k)
    {
        //        cout.form("%20.8f, ",CHOP(ar[k]));
        cout << CHOP(ar[k]) << "' ";
        if ( k%4==3 )  cout << "\n";
    }

    cout<<flush;
}
//=============== end D_PRINT ===========


void
reimabs_print(char *what, const double *ar, const double *ai, ulong n)
{
    cout << what << "\n";

    for (ulong k=0; k<n; ++k)
    {
        //        cout.form(" (%-+13.5g, %-+13.5g)  |.|=%-+13.5g ",
        //                  CHOP(ar[k]),
        //                  CHOP(ai[k]),
        //                  CHOP(sqrt(ar[k]*ar[k]+ai[k]*ai[k])));

        cout.setf(ios::showpos);
        cout << " (" << CHOP(ar[k])
             << ", " << CHOP(ai[k])
             << ")  |.|=" << CHOP(sqrt(ar[k]*ar[k]+ai[k]*ai[k]));
        cout << "\n";
    }

    cout << flush;
}
//=============== end D_PRINT ===========


void
i_print(char *what, const LIMB *ar, ulong n)
{
    cout << what;

    cout.precision(5);
    cout.fill('0');

    for (ulong k=0; k<n; ++k)
    {
        //        cout.form("%05d",ar[k]);
        cout << setw(5) << ar[k];

        cout << ", ";
        if ( k%8==7 )  cout<<"\n";
    }

    cout << flush;

    //    cout.precision(19);
    cout.fill(' ');

}
//=============== end I_PRINT ===========


double
d_sumofdigits(const double *a, ulong n, double nine, double s)
{
    for (ulong k=0; k<n; ++k)
    {
        s += a[k];
        if  ( s>=nine )  s -= nine;
    }

    return s;
}
//========== end D_SUMOFDIGITS ===========

