#include <math.h>

#include "fxtaux.h"



/****************************************************************************
*
*    split_radix_real_complex_fft(double x[],int ldn)
*    a real-valued, in-place, split-radix fft program
*    decimation-in-time, cos/sin in second loop
*    input: double x[0]...x[n-1] (C style)
*    length is n=2^ldn
*    output in x[0]...x[n-1], in order:
*         [re(0), re(1),..., re(n/2), im(n/2-1),..., im(1)]
*
*  imag part is negative wrt. output of fht_real_complex_fft() !
*
*
* original Fortran code by Sorensen; published in H.V. Sorensen, D.L. Jones,
* M.T. Heideman, C.S. Burrus(1987)Real-valued fast fourier transform
* algorithms.  IEEE Trans on Acoustics, Speech, & Signal Processing, 35,
* 849-863.  Adapted to C by Bill Simpson, 1995  wsimpson@uwinnipeg.ca
* further editing by joerg arndt  arndt@spektracom.de
*
****************************************************************************/



void 
split_radix_real_complex_fft(double x[], int ldn)
/*
 ordering on output:

 f[0]     = re[0] 
 f[1]     = re[1] 
         ... 
 f[n/2-1] = re[n/2-1] 
 f[n/2]   = re[n/2](==nyquist freq) 

 f[n/2+1] = im[n/2-1] (wrt. compl fft with is=-1)
 f[n/2+2] = im[n/2-2] 
         ... 
 f[n-1]   = im[1]

 corresponding real and imag parts (with the exception of
 zero and nyquist freq) are found in f[i] and f[n-i]

 note that the order of imaginary parts 
 is reversed wrt. fft_real_complex_fft()

 This is a decimation-in-time, split-radix algorithm.
*/
{
    long n=(1<<ldn);		/* n=2^ldn */

    long   i, i0, i1, i2, i3, i4, i5, i6, i7, i8, is, id;
    long   j, k, n2, n4, n8;
    double  a, cc1, ss1, cc3, ss3, e, r1;
    double  t1, t2, t3, t4, t5, t6;


    scramble(x,n);

    x--;  /* C convention */ 

/*----length two butterflies---------------------------------------------*/
    is=1;
    id=4;
    do
    {
	for(i0=is; i0<=n; i0+=id)
	{
	    i1=i0+1;
	    r1=x[i0];
	    x[i0]=r1+x[i1];
	    x[i1]=r1-x[i1];
	}

	is=2*id-1;
	id=4*id;
    }
    while(is<n);


/*---- L shaped butterflies-----------------------------------------------*/
    n2=2;
    for(k=2; k<=ldn; k++)
    {
	n2 <<= 1;
	e=2.0*M_PI/n2;
	n4=n2/4;
	n8=n2/8;
	is=0;
	id=n2<<1;

	do
	{
	    for(i=is; i<n; i+=id)
	    {
		i1=i+1;
		i2=i1+n4;
		i3=i2+n4;
		i4=i3+n4;

		t1    =  x[i4]+x[i3];
		x[i4] -= x[i3];

		x[i3] =  x[i1]-t1;
		x[i1] += t1;

		if(n4!=1)
		{
		    i1+=n8;
		    i2+=n8;
		    i3+=n8;
		    i4+=n8;

		    t2=(x[i3]-x[i4])*SQRT12;
		    t1=(x[i3]+x[i4])*SQRT12;

		    x[i4]= x[i2]-t1;
		    x[i3]=-x[i2]-t1;

		    x[i2]=x[i1]-t2;
		    x[i1]+=t2;
		}
	    }

	    is=2*id-n2;
	    id=4*id;
	}
	while(is<n);

	a=e;

	for(j=2; j<=n8; j++)
	{
	    sincos(&cc1, &ss1, a);

#if defined USE_SINCOS3
	    sincos(&cc3, &ss3, 3.0*a);
#else
	    cc3=4.0*cc1*(cc1*cc1-0.75);
	    ss3=4.0*ss1*(0.75-ss1*ss1);
#endif
	    a=(double)j*e;

	    is=0;
	    id=2*n2;

	    do
	    {
		for(i=is; i<n; i+=id)
		{
		    i1=i+j;
		    i2=i1+n4;
		    i3=i2+n4;
		    i4=i3+n4;

		    i5=i+n4-j+2;
		    i6=i5+n4;
		    i7=i6+n4;
		    i8=i7+n4;

		    t2=x[i7]*cc1-x[i3]*ss1;
		    t1=x[i3]*cc1+x[i7]*ss1;

		    t4=x[i8]*cc3-x[i4]*ss3;
		    t3=x[i4]*cc3+x[i8]*ss3;

		    t5=t1+t3;
		    t3=t1-t3;

		    t6=t2+t4;
		    t4=t2-t4;

		    t2   =x[i6]+t6;
		    x[i3]=t6-x[i6];

		    x[i8]=t2;

		    t2    =x[i2]-t3;
		    x[i7]=-x[i2]-t3;

		    x[i4]=t2;

		    t1   =x[i1]+t5;
		    x[i6]=x[i1]-t5;

		    x[i1]=t1;

		    t1   =x[i5]+t4;
		    x[i5]-=t4;

		    x[i2]=t1;
		}

		is=2*id-n2;
		id=4*id;
	    }
	    while(is<n);
	}
    }


    return;
}
/* ============== end SPLIT_RADIX_REAL_COMPLEX_FFT ================ */
