
#include <math.h>
#include <assert.h>

#include "fxtaux.h"
#include "fxt.h"


double *w;  // workspace

// always call walsh_ndim(), don't call e.g. walsh_3dim() directly !

void walsh_2dim(double *f, ulong d1, ulong d2);
void walsh_3dim(double *f, ulong d1, ulong d2, ulong d3);
void walsh_4dim(double *f, ulong d1, ulong d2, ulong d3, ulong d4);
void walsh_5dim(double *f, ulong d1, ulong d2, ulong d3, ulong d4, ulong d5);


// depending on this define you get the wak_, pal_ or wal_ version:
#define WALSH(f,n)  dit2_walsh_wal(f,ld(n))


void
walsh_ndim(double *f, ulong ndim, ulong *ldn)
//
// ndim must be 1,2,3,4 or 5
// ldn[] contains base 2 logarithms of dimensions
//
{
    if( (ndim<1) || (ndim>5) )
    {
        assert(0*(int)" invalid ndim in ndim_walsh ");
    }

    if(ndim==1)  // 1-dim walsh
    {
        WALSH(f,1<<ldn[0]);
        return;
    }

    // find max(d1,d2,...,dn):
    ulong nw=0;
    for(ulong k=0; k<ndim; ++k)
    {
        if( ldn[k]>nw )  nw=ldn[k];
    }
    nw=(1<<nw);

    w=new double[nw];

    ulong d1=(1<<ldn[0]);
    ulong d2=(1<<ldn[1]);
    if(ndim==2)
    {
        walsh_2dim(f,d1,d2);
    }
    else
    {
        ulong d3=(1<<ldn[2]);
        if(ndim==3)
        {
            walsh_3dim(f,d1,d2,d3);
        }
        else
        {
            ulong d4=(1<<ldn[3]);
            if(ndim==4)
            {
                walsh_4dim(f,d1,d2,d3,d4);
            }
            else
            {
                ulong d5=(1<<ldn[4]);
                if(ndim==5)
                {
                    walsh_5dim(f,d1,d2,d3,d4,d5);
                }
            }
        }
    }

    delete [] w;
}
// =============== end WALSH_NDIM =========== 


void
walsh_2dim(double *f, ulong d1, ulong d2)
{
    ulong n=d1*d2;
    ulong k;

    // rows:
    for(k=0; k<n; k+=d1)  WALSH(f+k,d1);

    // columns:
    for(k=0; k<d1; k++)  wide_walsh(f+k,d2,d1,w);
}
// =============== end WALSH_2DIM =========== 



void
walsh_3dim(double *f, ulong d1, ulong d2, ulong d3)
{
    ulong d12=d1*d2;
    ulong n=d12*d3;
    ulong k;

    for(k=0; k<n; k+=d12)  walsh_2dim(f+k,d1,d2);

    for(k=0; k<d12; k++)  wide_walsh(f+k,d3,d12,w);
}
// =============== end WALSH_3DIM =========== 



void
walsh_4dim(double *f, ulong d1, ulong d2, ulong d3, ulong d4)
{
    ulong d123=d1*d2*d3;
    ulong n=d123*d4;
    ulong k;

    for(k=0; k<n; k+=d123)  walsh_3dim(f+k,d1,d2,d3);

    for(k=0; k<d123; k++)  wide_walsh(f+k,d4,d123,w);
}
// =============== end WALSH_4DIM =========== 



void
walsh_5dim(double *f, ulong d1, ulong d2, ulong d3, ulong d4, ulong d5)
{
    ulong d1234=d1*d2*d3*d4;
    ulong n=d1234*d5;
    ulong k;

    for(k=0; k<n; k+=d1234)  walsh_4dim(f+k,d1,d2,d3,d4);

    for(k=0; k<d1234; k++)  wide_walsh(f+k,d5,d1234,w);
}
// =============== end WALSH_5DIM =========== 

