
#include <math.h>
#include <iostream.h>
#include <assert.h>
#include <stdlib.h>

#include "hfloatfu.h"
#include "mybuiltin.h"
#include "iteratio.h"

void approx_exp(const hfloat &d, hfloat &c);


// for debug:
#define  PR(x)
#define  SHRT  10

#undef   CHECK_LAST_STEP
#define  CHECK_LAST_STEP  (0)

int
exp_iteration(const hfloat &di, hfloat &x, ulong startprec)
//
//   the equation exp(d)==x*(exp(d-log(x))) gives the
//
//   iteration for exp(d):
//
//   x <- x*(1+y+y^2/2!+y^3/3!+...) where y := d-log(x)
//   or:
//   x += x*(  y+y^2/2!+y^3/3!+...)
//
{
    startprec = 0;  // avoid compiler warning

    static const ulong maxdeg = 10;
    static long num[maxdeg+1];
    static long den[maxdeg+1];

    // setup polynom for exp()-1:
    num[0] = 0;
    den[0] = 1;
    for (ulong k=1; k<=maxdeg; ++k)
    {
        num[k] = 1;
        den[k] = den[k-1] * k;  // k!
    }

    //    for (ulong k=0; k<=maxdeg; ++k)
    //        cout << " k=" << k << ": "
    //             << num[k] << "/" << den[k]
    //             << endl;

    static const ulong deg = 1;

    int ret=0;
    int j;
    //    ulong prec;
    //    ulong dp0 = di.prec();
    ulong newprec;
    ulong precgoal = x.prec();
    ulong n;
    hfloat t(precgoal);
    hfloat d(di);   // clone

    if ( same_mantissa(d,x) )  d.make_own_data(x.prec());

    if (d.is_zero())
    {
        x = 1;
        return 0;
    }

    int  xs = d.sign();
    d.sign( +1 );


    PR( print("\n d=",d,SHRT); );
    approx_exp(d,x);        PR( print("\n ==  x0=",x,SHRT); );
    newprec = 2;  // assume 2 limbs are correct

    n=START_CALC_PREC;
    t.prec(n);
    x.prec(n);
    d.prec(n);


    j=0;
    do
    {
        PR( print("\n x=",x,SHRT); );
        log(x,t);   PR( print("\n log(x)=",t,SHRT); );
        sub(d,t,t); PR( print("\n y:= d-log(x)=",t,SHRT); );

        polynom(t,deg,num,den,t);  PR( print("\n  poly=",t,SHRT); );

        mul(x,t,t); PR( print("\n  x*y=",t,SHRT); );

        newprec = x.exp()-t.exp();
        //            newprec *= (deg+1);
        PR( cout << "\n => newprec=" << newprec; );

        add(x,t,x);       PR( print("\n x+x*y=",x,SHRT); );

        //            PREC_INFO(newprec,n);

        if (++j>20)  ITER_ERR_1(j);
    }
    while(newprec < PREC_THRES/2);


    //    prec=0;
    for(j=0; j<MAX_REC; ++j)  // ---------------- ITERATION --------------
    {
        //        PREC_INFO(newprec,n);
        OOOPSQ(newprec,n);

        n = (deg+1)*n;
        n = MIN(n,precgoal);

        t.prec(n);
        x.prec(n);
        d.prec(n);

        log(x,t);     PR( print("\n log(x)=",t,SHRT); );

        sub(d,t,t);   PR( print("\n y:= d-log(x)=",t,SHRT); );


        if ( n!=precgoal )
        {
            polynom(t,deg,num,den,t);  PR( print("\n  poly=",t,SHRT); );
        }
        else
        {
            polynom(t,deg+2,num,den,t);  PR( print("\n  ++poly=",t,SHRT); );
        }

        mul(x,t,t);   PR( print("\n  x*poly=",t,SHRT); );

        //        prec = newprec;
        newprec = x.exp()-t.exp();
        //        newprec *= (deg+1);

        if ( BREAK_COND(newprec) )
        {
            //            PREC_INFO(newprec,n);
            break;
        }

        add(x,t,x);         PR( print("\n x+x*poly=",x,SHRT); );

        if ( !CHECK_LAST_STEP && EARLY_COND )  break;
    }
    // ---------------- end of ITERATION ---------------------


    if ( j==MAX_REC )  ITER_ERR_2(j);

    if ( xs==-1 )
    {
        inv(x,t);
        x = t;
    }

    return ret;
}
//========================= end EXP_ITERATION ==========================



#define  SIMPLE_EXP  (0)

void
approx_exp(const hfloat &d, hfloat &c)
//
// c = exp(d)
//
// to avoid overflow for big arguments
// we use:
// set c := M*R^X
// then d = log(M) + X * log(R)
//
// X = floor(d/log(R))  == "     div(d,log(R) "
// M = exp(d-d*X)       == " exp(mod(d,log(R)) "
//
{
    double dd;
    hfloat2d(d,dd);

#if ( SIMPLE_EXP )
    dd = exp(dd);
    d2hfloat(dd,c);
#else

    double lrx = log(hfloat::radix());
    double x;
    double m;
    int s = SIGN(dd);
    dd = fabs(dd);

    m = modf(dd/lrx,&x);
    m = exp(m);

    d2hfloat(m,c);

    if ( s<0 )  c.sign( -1 );

    c.exp( c.exp()+(long)x );

#endif
}
//=================== end APPROX_EXP ========================

