///////////////////////////////
//
// Piologie V1.2.1
// multi-precision arithmetic
// Digit / NumberBase
//
// Sebastian Wedeniwski
// 03/18/1998
//

#include "digit.h"

#ifdef _Piologie_Debug_
# define CONDITION(expr)     assert(expr)

# define NATURALCONDITION(a)                            \
  if ((a).root) {                                       \
      CONDITION(*(a).root == 0 &&                       \
                *((a).p-1) == 0 &&                      \
                (a).root < (a).p &&                     \
                (a).size > 0 &&                         \
                ((a).size == 1 || *(a).p != 0));        \
      const Digit* _pE = (a).p;                         \
      for (Digit* _pA = (a).root; _pA != _pE; ++_pA)    \
        CONDITION(*_pA == 0);                           \
  }

# define NATURAL_FOR_CHECK(a, b)                        \
  Natural (a) = (b);

# define INTEGERCONDITION(a)                            \
  CONDITION((a).highest() == 0 && sign(a) == 0 ||       \
            (a).highest() != 0 &&                       \
            (sign(a) == 1 || sign(a) == -1));

# define INTEGER_FOR_CHECK(a, b)                        \
  Integer (a) = (b);

#else
# define CONDITION(__ignore)        ((void)0)
# define NATURALCONDITION(__ignore) ((void)0)
# define INTEGERCONDITION(__ignore) ((void)0)
# define NATURAL_FOR_CHECK(__ignore1, __ignore2) ((void)0)
# define INTEGER_FOR_CHECK(__ignore1, __ignore2) ((void)0)
#endif

#ifndef _Old_STD_
# define NOTHROW_NEW  new(nothrow)
#else
# define NOTHROW_NEW  new
#endif



///////////////////// Class NumberBase /////////////////////////////

const size_t     SieveSize = (size_t(1) << ((BETA > 32)? 16 : BETA/2)) - 1;

class NumberBase_init : private NumberBase {
private:
  static NumberBase_init PrimeNumbers;

public:
  NumberBase_init(char);                // generating prime numbers
  ~NumberBase_init();
};

unsigned char*  NumberBase::primes     = 0;
unsigned char*  NumberBase::primPos    = 0;
Digit           NumberBase::primNumber = 2;
unsigned char   NumberBase::idx        = 1;
NumberBase_init NumberBase_init::PrimeNumbers(' ');

NumberBase_init::NumberBase_init(char)
{
  CONDITION(primes == 0);

  // Testing the bits for low ending
  const Digit d = 1 << 1;
  if (d != 2) errmsg(0, "Bitproblem!");
  
  // Generating prime numbers
  const Digit n = 8*SieveSize;
  unsigned char* p = primes = NOTHROW_NEW unsigned char[SieveSize];
  if (!primes) errmsg(2, "(constructor)");
  firstPrime();
  const unsigned char* pE = p+SieveSize;
  do *p++ = (unsigned char)~0; while (p != pE);
  p -= SieveSize;
  
  Digit c = GAMMA;
  Digit k = 1;
  Digit q = 2*(sqrt(Digit(24*SieveSize))/3);
  
  for (Digit i = 3; i <= q; i += 2) {
    k += 3;
    Digit j = c += 2*k;
    k += 3;
  
    while (j < n) {
      p[j/8] &= (unsigned char)~(1 << (j%8)); j += i;
      if (j >= n) break;
      p[j/8] &= (unsigned char)~(1 << (j%8)); j += k;
    }
  
    i += 2;
    if (i > q) break;
    j = c += ++k; ++k;
  
    while (j < n) {
      p[j/8] &= (unsigned char)~(1 << (j%8)); j += k;
      if (j >= n) break;
      p[j/8] &= (unsigned char)~(1 << (j%8)); j += i;
    }
  }
  
}

NumberBase_init::~NumberBase_init()
{
  if (this == &PrimeNumbers) {
    delete[] primes;
    primPos = primes = 0;
  }
}

size_t* NumberBase::quad_convergence_sizes(const size_t a, size_t& b) const
// Algorithm:  s := c.quad_convergence_sizes(a, b)
// Input:      c in NumberBase, a,b in size_t where a >= 2.
// Output:     b in size_t, s in size_t^b such that b = log2(a)+2,
//             s[i] = [a/2^i]+epsilon, 0 <= i < b ||
{
  CONDITION(a >= 2);

  size_t* s = NOTHROW_NEW size_t[log2(a)+2];
  if (!s) errmsg(2, "(quad_convergence_sizes)");
  size_t i = 1;
  size_t j = a+2;
  s[0] = j; s[1] = j /= 2;
  do { ++j; s[++i] = j /= 2; } while (j > 1);
  b = i+1;

  CONDITION(b >= 2);

  return s;
}

void default_piologie_error_handler(const int IDerr, const char* msg)
{
  switch(IDerr) {
    case 0: break;
//    case 1: cerr << "Overflow!"; break;
    case 2: cerr << "Out of memory!"; break;
    case 3: cerr << "Result negative (No Natural solution)!"; break;
    case 4: cerr << "Division by zero!"; break;
    case 5: cerr << "Same location error!"; break;
    case 6: cerr << "Discriminant negative (No Real solution)!"; break;

    default: cerr << "Internal error!";
  }
  cerr << msg << endl;
  exit(1);
}

static piologie_error_handler_t
 piologie_error_handler = default_piologie_error_handler;

piologie_error_handler_t set_piologie_error_handler(piologie_error_handler_t a)
{
  piologie_error_handler_t b = piologie_error_handler;
  piologie_error_handler = a;
  return b;
}

void NumberBase::errmsg(const int a, const char* b) const
{
  (*piologie_error_handler)(a, b);
}

bool NumberBase::nextPrime(Digit& a) const
// Algorithm:  c := n.nextPrime(a)
// Input:      n in NumberBase.
// Output:     a in Digit, c in bool
//             such that if a prime then c = true else c = false ||
{
  if (primNumber == 2) { a = 3; primNumber = 5; }
  else {
    Digit p;
    do {
      if (primPos - primes == SieveSize) return false;
      p = *primPos & idx;
      if (p) a = primNumber;
      if (idx == 1 << (CHAR_BIT-1)) { idx = 1; ++primPos; primNumber += 4; }
      else {
        if (idx == 2 || idx == 8 || idx == 32) primNumber += 2;
        primNumber += 2;
        idx *= 2;
      }
    } while (p == 0);
  }
  return true;
}

Digit NumberBase::lastPrime() const
// Algorithm:  c := n.lastPrime()
// Input:      n in NumberBase.
// Output:     c in Digit such that c prime ||
{
  static Digit lprim = 0;
  if (lprim) return lprim;
  Digit i = SieveSize;
  while (primes[--i] == 0);
  const Digit p = log2(Digit(primes[i]));
  i *= 24; i += 3*p + 5;
  return lprim = i - (p&1);
}

size_t NumberBase::NumberOfPrimes(Digit n) const
// Algorithm:  c := n.NumberOfPrimes(a)
// Input:      n in NumberBase, a in Digit.
// Output:     c in Digit
//             such that if a <= n.lastPrime() then c = pi(a) else c = 0 ||
{
  if (n < 5) return size_t(n/2 + (n == 3));
  n -= 5;
  if (n >= 24*SieveSize) return 0;
  const unsigned char* q = primes + n/24;
  n %= 24;
  size_t i = 2;
  for (unsigned char j = char(1 << (n/3 + ((n%6) == 2))); j >= 1; j /= 2)
    if (*q & j) ++i;

  for (unsigned char* p = primes; p < q; ++p) {
    for (unsigned char j = 1; j; j *= 2)
      if (*p & j) ++i;
  }
  return i;
}


#ifdef _DigitAsm_
#if _M_IX86 >= 300 && _MSC_VER
void NumberBase::digitmul(const Digit a, const Digit b, Digit& x, Digit& y) const
{
  __asm {
    mov eax,a
    mov ebx,y
    mov ecx,x
    mul b
    mov [ebx],eax
    mov [ecx],edx
  }
}

void NumberBase::digitdiv(const Digit a, const Digit b, const Digit c,
                          Digit& q, Digit& r) const
{
  __asm {
    mov eax,b
    mov edx,a
    mov ebx,q
    mov ecx,r
    div c
    mov [ebx],eax
    mov [ecx],edx
  }
}

#endif
#endif

#ifndef _DigitAsm_
void NumberBase::digitdiv(const Digit a, const Digit b, const Digit c,
                          Digit& q, Digit& r) const
// Algorithm:  n.digitdiv(a, b, c, q, r)
// Input:      n in NumberBase, a,b,c in Digit where a < c, c >= 2^(BETA-1).
// Output:     q,r in Digit such that q*c + r = a*2^BETA + b ||
{
  CONDITION(a < c && c >= (Digit(1) << (BETA-1)));

  Digit d = c >> BETA/2;
  Digit e = c & GAMMA_LOW;
  Digit x = a/d;
  Digit z = a-x*d;
  Digit y = x*e;
  z = (z << BETA/2) | (b >> BETA/2);
  if (z < y) {
    --x; z += c;
    if (z >= c && z < y) { --x; z += c; }
  }
  q = x << BETA/2;
  z -= y;
  x = z/d;
  z -= x*d;
  y = x*e;
  z = (z << BETA/2) | (b & GAMMA_LOW);
  if (z < y) {
    --x; z += c;
    if (z >= c && z < y) { --x; z += c; }
  }
  q |= x; r = z-y;
}

#endif

Digit pow10(size_t a)
// Algorithm:  b := pow10(a)
// Input:      a in size_t where a <= ALPHA_WIDTH.
// Output:     b in Digit such that b = 10^a ||
{
  CONDITION(a <= ALPHA_WIDTH);

  static const Digit c[10] = {1, 10, 100, Digit(1000), Digit(10000),
                              Digit(100000), Digit(1000000),
                              Digit(10000000), Digit(100000000),
                              Digit(1000000000) };

  if (ALPHA_WIDTH < 10 || a < 10) return c[a];
  Digit b = c[9];
  for (a -= 9; a >= 9; a -= 9) b *= c[9];
  return b*c[a];
}

Digit sqrt(Digit a)
// Algorithm:  c := sqrt(a)
// Input:      a in Digit.
// Output:     c in Digit such that c = [sqrt(a)] ||
{
  Digit b = Digit(1) << (BETA-2);
  Digit c = Digit(1) << (BETA-1) | Digit(1) << (BETA-2);

  do {
    if (a >= b) { a -= b; b |= c; }
    c >>= 2;
    b >>= 1;
    b ^= c;
  } while (c != 3);

  if (a >= b) b |= 2;
  return b >>= 1;
}

void sqrt(Digit a, Digit& x, Digit& y)
// Algorithm:  sqrt(a, x, y)
// Input:      a in Digit.
// Output:     x,y in Digit such that x = [sqrt(a)], y = a - x^2 ||
{
  Digit b = Digit(1) << (BETA-2);
  Digit c = Digit(1) << (BETA-1) | Digit(1) << (BETA-2);

  do {
    if (a >= b) { a -= b; b |= c; }
    c >>= 2;
    b >>= 1;
    b ^= c;
  } while (c != 3);

  if (a >= b) { a -= b; b |= 2; }
  y = a; x = b >>= 1;
}

Digit sqrt(Digit a, Digit b, Digit& x, Digit& y)
// Algorithm:  c := sqrt(a, b, x, y)
// Input:      a,b in Digit.
// Output:     c,x,y in Digit such that c = [sqrt(a*2^BETA+b)],
//             x*2^BETA+y = a*2^BETA+b - c^2 ||
{
  const Digit HIBIT = Digit(1) << (BETA-1);
  Digit c = Digit(1) << (BETA-2);
  Digit d = Digit(1) << (BETA-1) | Digit(1) << (BETA-2);

  do {
    if (a >= c) { a -= c; c |= d; }
    d >>= 2;
    c >>= 1;
    c ^= d;
  } while (d != 3);
  if (a >= c) { a -= c; c |= 2; }
  c >>= 1;

  d = Digit(1) << (BETA-1) | Digit(1) << (BETA-2);
  Digit e = Digit(1) << (BETA-2);
  do {
    if (a > c) {
      a -= c;
      if (b < e) --a;
      b -= e;
      e |= d;
    } else if (a == c && b >= e) { a = 0; b -= e; e |= d; }
    d >>= 2;
    e >>= 1;
    if (c&1) e |= HIBIT;
    c >>= 1;
    e ^= d;
  } while (d);
  x = a; y = b;
  return e;
}

Digit gcd(Digit a, Digit b)
// Algorithm:  c := gcd(a, b)
// Input:      a,b in Digit.
// Output:     c in Digit such that c = gcd(a, b) ||
{
  if (a == 0) return b;
  if (b == 0) return a;

  Digit i,j;
  for (i = 0; (a&1) == 0; ++i) a >>= 1;
  for (j = 0; (b&1) == 0; ++j) b >>= 1;
  while (a != b)
    if (a > b) {
      a -= b;
      do a >>= 1; while ((a&1) == 0);
    } else {
      b -= a;
      do b >>= 1; while ((b&1) == 0);
    }
  return a << ((i > j)? j : i);
}

void NumberBase::gcd(Digit a, Digit b, Digit c, Digit d, Digit& x, Digit& y) const
// Algorithm:  n.gcd(a, b, c, d; x, y)
// Input:      n in NumberBase, a,b,c,d in Digit.
// Output:     x,y in Digit such that x*2^BETA+y = gcd(a*2^BETA+b, c*2^BETA+d) ||
{
  if (a == 0 && b == 0) { x = c; y = d; }
  else if (c == 0 && d == 0) { x = a; y = b; }
  else {
    Digit i = 0;
    Digit j = 0;
    if (b == 0) {
      b = a; a = 0;
      for (i = BETA; (b&1) == 0; ++i) b >>= 1;
    } else if ((b&1) == 0) {
      do { ++i; b >>= 1; } while ((b&1) == 0);
      b |= a << (BETA-i); a >>= i;
    }
    if (d == 0) {
      d = c; c = 0;
      for (j = BETA; (d&1) == 0; ++j) d >>= 1;
    } else if ((d&1) == 0) {
      do { ++j; d >>= 1; } while ((d&1) == 0);
      d |= c << (BETA-j); c >>= j;
    }
    if (j < i) i = j;
    while (b != d && a != c)
      if (a > c) {
        a -= c + (b < d); b -= d;
        j = 0;
        do { ++j; b >>= 1; } while ((b&1) == 0);
        b |= a << (BETA-j); a >>= j;
      } else {
        c -= a + (d < b); d -= b;
        j = 0;
        do { ++j; d >>= 1; } while ((d&1) == 0);
        d |= c << (BETA-j); c >>= j;
      }
    if (b != d || a != c) {
      if (a == c) {
        if (b > d) {
          b -= d;
          digitmod(c, d, b, a);
          a = ::gcd(a, b);
        } else {
          d -= b;
          digitmod(a, b, d, c);
          a = ::gcd(c, d);
        }
      } else {
        if (a > c) {
          a -= c;
          digitmod(c, d, a, b);
          a = ::gcd(a, b);
        } else {
          c -= a;
          digitmod(a, b, c, d);
          a = ::gcd(c, d);
        }
      }
      if (i >= BETA) { x = a << (i-BETA); y = 0; }
      else if (i) { x = a >> (BETA-i); y = a << i; }
      else { x = 0; y = a; }
    } else if (i >= BETA) { x = b << (i-BETA); y = 0; }
    else if (i) { x = a << i | b >> (BETA-i); y = b << i; }
    else { x = a; y = b; }
  }
}

