///////////////////////////////
//
// Piologie V1.2.1
// multi-precision arithmetic
// Pi-Calculation
//
// Sebastian Wedeniwski
// 03/18/1998
//

#include "pi.h"

#ifdef _Piologie_Debug_
# define CONDITION(expr)     assert(expr)

# define NATURALCONDITION(a)                            \
  if ((a).root) {                                       \
      CONDITION(*(a).root == 0 &&                       \
                *((a).p-1) == 0 &&                      \
                (a).root < (a).p &&                     \
                (a).size > 0 &&                         \
                ((a).size == 1 || *(a).p != 0));        \
      const Digit* _pE = (a).p;                         \
      for (Digit* _pA = (a).root; _pA != _pE; ++_pA)    \
        CONDITION(*_pA == 0);                           \
  }

# define NATURAL_FOR_CHECK(a, b)                        \
  Natural (a) = (b);

# define INTEGERCONDITION(a)                            \
  CONDITION((a).highest() == 0 && sign(a) == 0 ||       \
            (a).highest() != 0 &&                       \
            (sign(a) == 1 || sign(a) == -1));

# define INTEGER_FOR_CHECK(a, b)                        \
  Integer (a) = (b);

#else
# define CONDITION(__ignore)        ((void)0)
# define NATURALCONDITION(__ignore) ((void)0)
# define INTEGERCONDITION(__ignore) ((void)0)
# define NATURAL_FOR_CHECK(__ignore1, __ignore2) ((void)0)
# define INTEGER_FOR_CHECK(__ignore1, __ignore2) ((void)0)
#endif


void Fixed::output(ostream& out)
// Algorithm:  a.output(o)
// Input:      o in ostream, a in Fixed.
// Output:     o in ostream ||
//
// Note:       puts Fixed a on output stream.
{
  const size_t block_size = 500;
  size_t n = decimals();
  size_t m = precision();
  static Natural c = pow(Natural(ALPHA), Digit(block_size));
  const size_t sB = length();
  enlarge((sB > m)? 3 : 3+m-sB);
  if (sB > m) {
    fast_rshift(m);
    out << value();
    if (n) out << '.';
    value() = 0; fast_append(m); normalize();
  } else out << "0.";
  while (n >= block_size*ALPHA_WIDTH) {
    value() *= c;
    out.width(block_size*ALPHA_WIDTH);
    out.fill('0');
    if (length() > m) {
      fast_rshift(m);
      out << value();
      value() = 0; fast_append(m); normalize();
      if (value() != 0) {
        Digit* pT;
        const size_t sz = trailing_zeros(pT);
        fast_rshift(sz); m -= sz;
      }
    } else out << '0';
    n -= block_size*ALPHA_WIDTH;
  }
  Digit* pE = last();
  Digit* pT = pE-m;
  while (n) {
    value() *= ALPHA;
    if (n <= ALPHA_WIDTH) {
       out.width(n);
       out.fill('0');
       Digit d = highest();
       for (n = ALPHA_WIDTH-n; n; --n) d /= 10;
       out << d;
       break;
    }
    out.width(ALPHA_WIDTH);
    out.fill('0');
    if (length() > m) {
      out << *pT;
      *pT = 0;
      if (*pE == 0) { --pE; fast_rshift(1); --m; }
      normalize();
    } else out << '0';
    n -= ALPHA_WIDTH;
  }
}

void Pi::stoermer(const size_t n, Natural& pi)
// Algorithm:  p.stoermer(n, a)
// Input:      p in Pi, a in Natural, n in size_t
//             where n < 3(GAMMA_LOW+1)/BETA, BETA >= 32.
// Output:     a in Natural such that |a-pi*2^(BETA*n)| < 1 ||
{
  const size_t sz = BETA*n;
  Natural t;
  Natural a = Digit(8*24);
  Natural b = Digit(57*8);
  Natural c = Digit(239*4);
  pi        = 0;
  Digit   k = 0;

  a <<= sz; b <<= sz; c <<= sz;
  do {
    a >>= 12;
    t = a * (63*k+191);
    if (b != 0) {
      b /= Digit(57*57*57*57);
      t += b * Digit(3248*k+9746);
      if (c != 0) {
        c /= Digit(239UL*239*239*239);
        t += c * Digit(57120*k+171362);
      }
    }
    pi += t /= (k+1)*(k+3);
    k += 4;
  } while (t != 0);
}

void Pi::schoenhage(const size_t n, Natural& pi)
// Algorithm:  p.schoenhage(n, a)
// Input:      p in Pi, a in Natural, n in size_t.
// Output:     a in Natural such that |a-pi*2^(BETA*n)| < 1 ||
{
  const size_t m = size_t(log2(BETA*n+BETA)-1.68025927);
  const size_t sz = BETA*n;
 
  Integer s,t,a(1),A(1),b(1),B(1),c(1);
  a <<= sz; A <<= sz; B <<= sz-1; c <<= sz-2;
  for (size_t k = 0; k <= m; ++k) {
    t = A+B; t >>= 2;
    s = B << sz; b = sqrt(s);
    a += b; a >>= 1;
    A = a*a; A >>= sz;
    B = A-t; B <<= 1;
    t = A-B; t <<= k; c -= t;
  }
  A <<= sz;
  div(A, c, t, a);
  pi = abs(t);
}

void Pi::chudnovsky(size_t n, const size_t m, Integer& p, Integer& q, Integer& t) const
// Algorithm:  p.chudnovsky(n, m, p, q, t)
// Input:      p in Pi, p,q,t in Integer, n,m in size_t where n < 2^BETA/6, n < m.
// Output:     p,q,t in Integer ||
{
  CONDITION(n < m && n < ((size_t(1) << (BETA-1))/3));

  const Digit A = Digit(13591409);
  const Digit B = Digit(545140134);
  const Digit C = Digit(640320);
  switch (m-n) {
    case 1: {
      if (n <= GAMMA_LOW/4) p = (6*n-5)*(2*n-1);
      else { p = 6*n-5; p *= 2*n-1; }
      p *= 6*n-1; p = -p;
      if (n <= GAMMA_LOW) q = n*n;
      else { q = n; q *= n; }
      q *= n; q *= 230*C; q *= 116*C;
      t = B; t *= n; t += A; t *= p;
      
      break;
    }
    case 2: {
      Integer s1,s2;
      if (n <= GAMMA_LOW/4) {
        p = (6*n-5)*(2*n-1);
        s1 = p *= 6*n-1; s1 = -s1;
        p *= 6*n+5; p *= (2*n+1)*(6*n+1);
      } else {
        p = 6*n-5; p *= 2*n-1;
        s1 = p *= 6*n-1; s1 = -s1;
        p *= 6*n+5; p *= 2*n+1; p *= 6*n+1;
      }
      if (n < GAMMA_LOW) {
        const size_t k = n+1;
        q = k*k; q *= k; q *= 230*C;
        s2 = q *= 116*C; q *= n*n;
      } else {
        const size_t k = n+1;
        q = k; q *= k; q *= k; q *= 230*C;
        s2 = q *= 116*C; q *= n; q *= n;
      }
      q *= n; q *= 230*C; q *= 116*C;
      t = B; t *= n; t += A;
      const Integer s3 = t;
      t += B;
      const Integer s4 = t*p;
      t = s1*s2*s3;
      t += s4;
      
      break;
    }
    case 3: {
      Integer s1,s2;
      if (n <= GAMMA_LOW/6) {
        p = (6*n+1)*(2*n-1); p *= (6*n-1)*(6*n-5);
        s1 = p *= (2*n+1)*(6*n+5);
        p *= (6*n+7)*(2*n+3);
      } else {
        p = 6*n-5; p *= 2*n-1; p *= 6*n-1;
        p *= 6*n+5; p *= 2*n+1;
        s1 = p *= 6*n+1;
        p *= 6*n+7; p *= 2*n+3;
      }
      p *= 6*n+11; p = -p;
      if (n < GAMMA_LOW-1) {
        const size_t k = n+1;
        q = k*k; q *= k; q *= 230*C; q *= 116*C;
        const size_t k2 = n+2;
        q *= k2*k2; q *= k2; q *= 230*C;
        s2 = q *= 116*C; q *= n*n;
      } else {
        const size_t k = n+1;
        q = k; q *= k; q *= k; q *= 230*C; q *= 116*C;
        const size_t k2 = n+2;
        q *= k2; q *= k2; q *= k2; q *= 230*C;
        s2 = q *= 116*C; q *= n; q *= n;
      }
      q *= n; q *= 230*C; q *= 116*C;
      t = B; t *= n; t += A;
      const Integer s3 = t;
      t += B;
      const Integer s4 = t;
      t += B;
      const Integer s5 = t*p;
      t = s2*s3;
      if (n <= GAMMA_LOW/4) t *= (6*n-5)*(2*n-1);
      else { t *= 6*n-5; t *= 2*n-1; }
      t *= 6*n-1; t = -t;
      s2 = s1*s4;
      n += 2;
      if (n <= GAMMA_LOW) s2 *= n*n;
      else { s2 *= n; s2 *= n; }
      s2 *= n; s2 *= 230*C; s2 *= 116*C;
      t += s2; t += s5;
      
      break;
    }
    default: {
      Integer p1,q1,t1;
      Integer p2,q2,t2;

      const size_t l = (n+m)/2;
      chudnovsky(n, l, p1, q1, t1);
      chudnovsky(l, m, p2, q2, t2);
      t = t1*q2; t += t1 = t2*p1;
      p = p1*p2; q = q1*q2;
    }
  }
}
void Pi::chudnovsky2(const size_t n, const size_t m, Integer& q, Integer& t) const
// Algorithm:  p.chudnovsky(n, m, q, t)
// Input:      p in Pi, q,t in Integer, n,m in size_t where n < m.
// Output:     q,t in Integer ||
{
  CONDITION(n < m);

  Integer p1,q1,t1;
  Integer q2,t2;

  const int l = (n+m)/2;
  chudnovsky(n, l, p1, q1, t1);
  if (m-l <= 3) {
    Integer p2;
    chudnovsky(l, m, p2, q2, t2);
  } else chudnovsky2(l, m, q2, t2);
  t = t1*q2; t += t1 = t2*p1;
  q = q1*q2;
}

Pi::Pi(const size_t n)
// Algorithm:  a := Pi(n)
// Input:      n in size_t.
// Output:     a in Pi such that |a-pi*10^n| < 1 ||
 : pi(n)
{
  const size_t sz = pi.precision();

  if (BETA < 32) schoenhage(sz, pi.value());
  else if (sz < (3*GAMMA_LOW+3)/BETA && n < 10000)
    stoermer(sz, pi.value());
  else if (n >= 100000000 || n >= ((size_t(1) << (BETA-1))/3))
    schoenhage(sz, pi.value());
  else {
    const Digit A = Digit(13591409);
    const Digit C = Digit(640320);

    Integer p1,q1,t1;
    Integer q2,t2;

    const size_t k = size_t(n/14.1643);
    const size_t l = (k+1)/2;
    chudnovsky(1, l, p1, q1, t1);
    chudnovsky2(l, k, q2, t2);
    Integer t = t1*q2;
    t += t1 = t2*p1;
    p1 = q1*q2;
    t += p1*A;
    t2 = Digit(2208*C); t2 *= Digit(290*C); t2 <<= BETA*sz*2;
    t1 = sqrt(t2); t2 = p1*t1;
    pi.value() = abs(t2) / abs(t);
    pi.value() /= 12;
  }
}

void Zeta3::series(size_t n, const size_t m, Integer& p, Integer& q, Integer& t) const
// Algorithm:  z.series(n, m, p, q, t)
// Input:      z in Zeta3, p,q,t in Integer, n,m in size_t where n < 2^BETA/64-32, n < m.
// Output:     p,q,t in Integer ||
{
  CONDITION(n < m && n < ((size_t(1) << (BETA-1))/64)-32);

  if (n+1 == m) {
    if (n <= GAMMA_LOW) { p = n*n; p *= n*n; }
    else { p = n; p *= n; p *= n; p *= n; }
    p *= n; p = -p;
    if (n < GAMMA/207) t = 205*n+250;
    else { t = n; t *= 205; t += 250; }
    t *= n; t += 77; t *= p;
    n = 2*n+1;
    if (n <= GAMMA_LOW) { q = n*n; q *= n*n; }
    else { q = n; q *= n; q *= n; q *= n; q *= n; }
    q *= 32*n;
  } else {
    Integer p1,q1,t1;
    Integer p2,q2,t2;

    const int l = (n+m)/2;
    series(n, l, p1, q1, t1);
    series(l, m, p2, q2, t2);
    t = t1*q2; t += t1 = t2*p1;
    p = p1*p2; q = q1*q2;
  }
}
void Zeta3::series2(const size_t n, const size_t m, Integer& q, Integer& t) const
// Algorithm:  z.series2(n, m, q, t)
// Input:      z in Zeta3, q,t in Integer, n,m in size_t where n < m.
// Output:     q,t in Integer ||
{
  CONDITION(n < m);

  Integer p1,q1,t1;
  Integer q2,t2;

  const int l = (n+m)/2;
  series(n, l, p1, q1, t1);
  if (m-l <= 1) {
    Integer p2;
    series(l, m, p2, q2, t2);
  } else series2(l, m, q2, t2);
  t = t1*q2; t += t1 = t2*p1;
  q = q1*q2;
}

void Zeta3::linear(const size_t n, Natural& z) const
// Algorithm:  b.linear(n, a)
// Input:      b in Zeta3, a in Natural, n in size_t where b.decimals()/574 <= GAMMA.
// Output:     a in Natural such that |a-Zeta(3)*2^(BETA*n)| < 1 ||
{
  CONDITION((zeta.decimals()/1.4)/410 <= GAMMA);

  Natural a = 1;
  Natural c = 77;
  Natural t;

  a <<= n*BETA-5;
  z = 0;
  Digit k = 1;
  do {
    t = a * c;
    if (k&1) z += t;
    else z -= t;

    c += 410*k+45;
    if (k <= GAMMA_LOW) { a *= k*k; a *= k*k; }
    else { a *= k; a *= k; a *= k; a *= k; }
    a *= k;
    const Digit i = 2*k+1;
    if (i <= GAMMA_LOW) { a /= i*i; a /= i*i; }
    else { a /= i; a /= i; a /= i; a /= i; }
    a /= 32*i;
    ++k;
  } while (a != 0);
  z >>= 1;
}

Zeta3::Zeta3(const size_t n)
// Algorithm:  a := Zeta3(n)
// Input:      n in size_t.
// Output:     a in Zeta3 such that |a-zeta(3)*10^n| < 1 ||
 : zeta(n)
{
  const size_t sz = zeta.precision();

  if (n <= 4000 && n/574 <= GAMMA) linear(sz, zeta.value());
  else {
    Integer p1,q1,t1;
    Integer q2,t2;

    const size_t k = (n >= 100000)? size_t(n/3.0102348) : n/3;
    const size_t l = (k+1)/2;
    series(1, l, p1, q1, t1);
    series2(l, k, q2, t2);
    Integer t = t1*q2;
    t += t1 = t2*p1;
    t2 = q1*q2;
    t += t2*77; t2 <<= 5;
    t <<= BETA*sz-1;
    zeta.value() = abs(t) / abs(t2);
  }
}

