///////////////////////////////
//
// Piologie V1.2.1
// multi-precision arithmetic
// Rational
//
// Sebastian Wedeniwski
// 03/18/1998
//


#ifndef _Include_Rational_H_
#define _Include_Rational_H_

#include "integer.h"

struct Rational_plus_tag {};
struct Rational_negate_tag {};
struct Rational_minus_tag {};
struct Rational_multiplies_tag {};
struct Rational_divides_tag {};
struct Rational_modulus_tag {};
struct Rational_square_root_tag {};
struct Rational_lshift_tag {};
struct Rational_rshift_tag {};

//#ifdef _Old_STD_
//template <class I, class N>
//#else
//template <class I, class N = I>
//#endif
class Rational {
public:
  struct rep {
    const Integer& num;
    const Natural& den;
    rep(const Integer& a, const Natural& b)
      : num(a), den(b) {}
  };
  

private:
  Integer num;
  Natural den;

  void add(const Rational&, const Rational&);
  void sub(const Rational&, const Rational&);
  void mul(const Rational&, const Rational&);
  void sqr(const Rational&);
  void div(const Rational&, const Rational&);
  void lshift(const Rational&, size_t);
  void rshift(const Rational&, size_t);

public:
  Rational(const Digit = 0);
  Rational(const Integer&);
  Rational(const Integer&, const Natural&);
  Rational(const Rational&);
  ~Rational();

  const Integer&   numerator() const;
  const Natural&   denominator() const;

  Digit            operator=(const Digit);
  Rational&        operator=(const Integer&);
  Rational&        operator=(const Rational&);
  Rational&        operator+=(const Rational&);
  Rational&        operator-=(const Rational&);
  Rational&        operator*=(const Rational&);
  Rational&        operator/=(const Rational&);
  Rational&        operator<<=(const size_t);
  Rational&        operator>>=(const size_t);
  
  const Rational&  operator++();
  const Rational&  operator--();
  Rational         operator++(int);
  Rational         operator--(int);


  Rational(const binder_arguments<Rational, Rational, Rational_negate_tag>&);
  Rational& operator=(const binder_arguments<Rational, Rational, Rational_negate_tag>&);

  Rational(const binder_arguments<Rational, Rational, Rational_plus_tag>&);
  Rational& operator=(const binder_arguments<Rational, Rational, Rational_plus_tag>&);

  Rational(const binder_arguments<Rational, Rational, Rational_minus_tag>&);
  Rational& operator=(const binder_arguments<Rational, Rational, Rational_minus_tag>&);

  Rational(const binder_arguments<Rational, Rational, Rational_multiplies_tag>&);
  Rational& operator=(const binder_arguments<Rational, Rational, Rational_multiplies_tag>&);

  Rational(const binder_arguments<Rational, Rational, Rational_divides_tag>&);
  Rational& operator=(const binder_arguments<Rational, Rational, Rational_divides_tag>&);

  Rational(const binder_arguments<Rational, size_t, Rational_lshift_tag>&);
  Rational& operator=(const binder_arguments<Rational, size_t, Rational_lshift_tag>&);

  Rational(const binder_arguments<Rational, size_t, Rational_rshift_tag>&);
  Rational& operator=(const binder_arguments<Rational, size_t, Rational_rshift_tag>&);

  void rand(const size_t);
  bool scan(istream&);

  inline friend void  swap(Rational&, Rational&);
  friend istream&     operator>>(istream&, Rational&);
  friend Rational     inv(const Rational&);
};

inline binder_arguments<Rational, Rational, Rational_negate_tag>
 operator-(const Rational&);
inline binder_arguments<Rational, Rational, Rational_plus_tag>
 operator+(const Rational&, const Rational&);
inline binder_arguments<Rational, Rational, Rational_minus_tag>
 operator-(const Rational&, const Rational&);
inline binder_arguments<Rational, Rational, Rational_multiplies_tag>
 operator*(const Rational&, const Rational&);
inline binder_arguments<Rational, Rational, Rational_divides_tag>
 operator/(const Rational&, const Rational&);
inline binder_arguments<Rational, size_t, Rational_lshift_tag>
 operator<<(const Rational&, const size_t&);
inline binder_arguments<Rational, size_t, Rational_rshift_tag>
 operator>>(const Rational&, const size_t&);


inline bool operator==(const Rational&, const Rational&);
inline bool operator!=(const Rational&, const Rational&);
inline bool operator<(const Rational&, const Rational&);
inline bool operator<=(const Rational&, const Rational&);
inline bool operator>(const Rational&, const Rational&);
inline bool operator>=(const Rational&, const Rational&);

inline bool operator==(const Rational&, const Digit);
inline bool operator!=(const Rational&, const Digit);
inline bool operator<(const Rational&, const Digit);
inline bool operator<=(const Rational&, const Digit);
inline bool operator>(const Rational&, const Digit);
inline bool operator>=(const Rational&, const Digit);

inline const Integer& numerator(const Rational&);
inline const Natural& denominator(const Rational&);
inline Rational       abs(const Rational&);
inline int            sign(const Rational&);
inline Rational::rep  print(const Rational&);
inline ostream&       operator<<(ostream&, const Rational::rep&);
inline ostream&       operator<<(ostream&, const Rational&);
Integer               ceil(const Rational&);
Integer               floor(const Rational&);
inline Integer        round(const Rational&);
Integer               trunc(const Rational&);
inline Rational       pow(const Rational&, const Digit);
Rational              pow(const Rational&, const Integer&);


// Unfortunatelly not:
//typedef Rational<Integer, Natural> Rational;



///////////////////////// Inline-Implementation ////////////////////

inline void Rational::sqr(const Rational& a)
// Algorithm:  b.sqr(a)
// Input:      a in Rational.
// Output:     b in Rational such that b = a^2 ||
{
  num = a.num*a.num;
  den = a.den*a.den;
}

inline Rational::Rational(const Digit a)
 : num(a), den(1)
// Algorithm:  c := Rational(a)
// Input:      a in Digit.
// Output:     c in Rational such that c = a/1 ||
{
}

inline Rational::Rational(const Integer& a)
 : num(a), den(1)
// Algorithm:  c := Rational(a)
// Input:      a in I.
// Output:     c in Rational such that c = a/1 ||
{
}

inline Rational::Rational(const Rational& a)
 : num(a.num), den(a.den)
// Algorithm:  c := Rational(a)
// Input:      a in Rational.
// Output:     c in Rational such that c = a ||
{
}

inline Rational::~Rational()
{
}

inline const Integer& Rational::numerator() const
// Algorithm:  c := a.numerator()
// Input:      a in Rational.
// Output:     c in Integer such that c = a_1 where a_1/a_2 = a ||
{
  return num;
}

inline const Natural& Rational::denominator() const
// Algorithm:  c := a.denominator()
// Input:      a in Rational.
// Output:     c in Natural such that c = a_2 where a_1/a_2 = a ||
{
  return den;
}

inline Digit Rational::operator=(const Digit a)
// Algorithm:  c := b = a
// Input:      a in Digit, b in Rational.
// Output:     b in Rational, c in Digit such that b = a/1, c = a ||
{
  num = a; den = 1;
  return a;
}

inline Rational& Rational::operator=(const Integer& a)
// Algorithm:  c := c = a
// Input:      a in Integer, c in Rational.
// Output:     c in Rational such that c = a/1 ||
{
  num = a; den = 1;
  return *this;
}

inline Rational& Rational::operator=(const Rational& a)
// Algorithm:  c := c = a
// Input:      a,c in Rational.
// Output:     c in Rational such that c = a ||
{
  num = a.num; den = a.den;
  return *this;
}

inline Rational& Rational::operator+=(const Rational& a)
// Algorithm:  c := c += a
// Input:      a,c in Rational.
// Output:     c in Rational such that c := c+a ||
{
  add(*this, a);
  return *this;
}

inline Rational& Rational::operator-=(const Rational& a)
// Algorithm:  c := c -= a
// Input:      a,c in Rational.
// Output:     c in Rational such that c := c-a ||
{
  sub(*this, a);
  return *this;
}

inline Rational& Rational::operator*=(const Rational& a)
// Algorithm:  c := c *= a
// Input:      a,c in Rational.
// Output:     c in Rational such that c := c*a ||
{
  if (this == &a) sqr(*this);
  else mul(*this, a);
  return *this;
}

inline Rational& Rational::operator/=(const Rational& a)
// Algorithm:  c := c /= a
// Input:      a,c in Rational where not a = 0.
// Output:     c in Rational such that c := c/a ||
{
  div(*this, a);
  return *this;
}

inline Rational& Rational::operator<<=(const size_t a)
// Algorithm:  c := c <<= a
// Input:      a in size_t, c in Rational.
// Output:     c in Rational such that c := c * 2^a ||
{
  lshift(*this, a);
  return *this;
}

inline Rational& Rational::operator>>=(const size_t a)
// Algorithm:  c := c >>= a
// Input:      a in size_t, c in Rational.
// Output:     c in Rational such that c := c / 2^a ||
{
  rshift(*this, a);
  return *this;
}

inline const Rational& Rational::operator++()
// Algorithm:  c := ++a
// Input:      a in Rational.
// Output:     a,c in Rational such that a := a+1, c := a ||
{
  num += den;
  return *this;
}

inline const Rational& Rational::operator--()
// Algorithm:  c := --a
// Input:      a in Rational.
// Output:     a,c in Rational such that a := a-1, c := a ||
{
  num -= den;
  return *this;
}

inline Rational Rational::operator++(int)
// Algorithm:  c := a++
// Input:      a in Rational.
// Output:     a,c in Rational such that c := a, a := a+1 ||
{
  Rational a(*this);
  ++(*this);
  return a;
}

inline Rational Rational::operator--(int)
// Algorithm:  c := a--
// Input:      a in Rational.
// Output:     a,c in Rational such that c := a, a := a-1 ||
{
  Rational a(*this);
  --(*this);
  return a;
}

inline Rational::Rational(const binder_arguments<Rational, Rational,
                                                   Rational_negate_tag>& a)
 : num(-a.x.num), den(a.x.den)
{
}

inline Rational& Rational::operator=(const binder_arguments<Rational, Rational,
                                                            Rational_negate_tag>& a)
{
  num = -a.x.num;
  if (this != &a.x) den = a.x.den;
  return *this;
}

inline binder_arguments<Rational, Rational, Rational_negate_tag>
 operator-(const Rational& a)
// Algorithm:  c := -a
// Input:      a in Rational.
// Output:     c in Rational such that c = -a ||
{
  return binder_arguments<Rational, Rational, Rational_negate_tag>(a, a);
}

inline Rational::Rational(const binder_arguments<Rational, Rational,
                                                   Rational_plus_tag>& a)
{
  add(a.x, a.y);
}

inline Rational& Rational::operator=(const binder_arguments<Rational, Rational,
                                                            Rational_plus_tag>& a)
{
  add(a.x, a.y);
  return *this;
}

inline binder_arguments<Rational, Rational, Rational_plus_tag>
 operator+(const Rational& a, const Rational& b)
// Algorithm:  c := a+b
// Input:      a,b in Rational.
// Output:     c in Rational such that c = a+b ||
{
  return binder_arguments<Rational, Rational, Rational_plus_tag>(a, b);
}

inline Rational::Rational(const binder_arguments<Rational, Rational,
                                                   Rational_minus_tag>& a)
{
  sub(a.x, a.y);
}

inline Rational& Rational::operator=(const binder_arguments<Rational, Rational,
                                                            Rational_minus_tag>& a)
{
  sub(a.x, a.y);
  return *this;
}

inline binder_arguments<Rational, Rational, Rational_minus_tag>
 operator-(const Rational& a, const Rational& b)
// Algorithm:  c := a-b
// Input:      a,b in Rational.
// Output:     c in Rational such that c = a-b ||
{
  return binder_arguments<Rational, Rational, Rational_minus_tag>(a, b);
}

inline Rational::Rational(const binder_arguments<Rational, Rational,
                                                 Rational_multiplies_tag>& a)
{
  if (&a.x == &a.y) sqr(a.x);
  else mul(a.x, a.y);
}

inline Rational& Rational::operator=(const binder_arguments<Rational, Rational,
                                                            Rational_multiplies_tag>& a)
{
  if (&a.x == &a.y) sqr(a.x);
  else mul(a.x, a.y);
  return *this;
}

inline binder_arguments<Rational, Rational, Rational_multiplies_tag>
 operator*(const Rational& a, const Rational& b)
// Algorithm:  c := a*b
// Input:      a,b in Rational.
// Output:     c in Rational such that c = a*b ||
{
  return binder_arguments<Rational, Rational, Rational_multiplies_tag>(a, b);
}

inline Rational::Rational(const binder_arguments<Rational, Rational,
                                                 Rational_divides_tag>& a)
{
  div(a.x, a.y);
}

inline Rational& Rational::operator=(const binder_arguments<Rational, Rational,
                                                            Rational_divides_tag>& a)
{
  div(a.x, a.y);
  return *this;
}

inline binder_arguments<Rational, Rational, Rational_divides_tag>
 operator/(const Rational& a, const Rational& b)
// Algorithm:  c := a/b
// Input:      a,b in Rational.
// Output:     c in Rational such that c = a/b ||
{
  return binder_arguments<Rational, Rational, Rational_divides_tag>(a, b);
}

inline Rational::Rational(const binder_arguments<Rational, size_t,
                                                   Rational_lshift_tag>& a)
{
  lshift(a.x, a.y);
}

inline Rational& Rational::operator=(const binder_arguments<Rational, size_t,
                                                            Rational_lshift_tag>& a)
{
  lshift(a.x, a.y);
  return *this;
}

inline binder_arguments<Rational, size_t, Rational_lshift_tag>
 operator<<(const Rational& a, const size_t& b)
// Algorithm:  c := a << b
// Input:      a in Rational, b in size_t.
// Output:     c in Rational such that c = a * 2^b ||
{
  return binder_arguments<Rational, size_t, Rational_lshift_tag>(a, b);
}

inline Rational::Rational(const binder_arguments<Rational, size_t,
                                                 Rational_rshift_tag>& a)
{
  rshift(a.x, a.y);
}

inline Rational& Rational::operator=(const binder_arguments<Rational, size_t,
                                                            Rational_rshift_tag>& a)
{
  rshift(a.x, a.y);
  return *this;
}

inline binder_arguments<Rational, size_t, Rational_rshift_tag>
 operator>>(const Rational& a, const size_t& b)
// Algorithm:  c := a >> b
// Input:      a in Rational, b in size_t.
// Output:     c in Rational such that c = a / 2^b ||
{
  return binder_arguments<Rational, size_t, Rational_rshift_tag>(a, b);
}

inline void Rational::rand(const size_t n)
// Algorithm:  a.rand(n)
// Input:      n in size_t.
// Output:     a in Rational
//             such that |a_1| < 2^n, a_2 <= 2^n where a_1/a_2 = a (random number) ||
{
  num.rand(n);
  den.rand(n); ++den;
  const Natural t = gcd(abs(num), den);
  num /= t; den /= t;
}

inline void swap(Rational& a, Rational& b)
// Algorithm:  swap(a, b)
// Input:      a,b in Rational.
// Output:     a,b in Rational such that t := a, a := b, b := t
//             where t in Rational ||
{
  swap(a.num, b.num);
  swap(a.den, b.den);
}

inline bool operator==(const Rational& a, const Rational& b)
// Algorithm:  c := a == b
// Input:      a,b in Rational.
// Output:     c in bool such that if a = b then c = true else c = false ||
{
  return (a.numerator() == b.numerator() && a.denominator() == b.denominator());
}


inline bool operator!=(const Rational& a, const Rational& b)
// Algorithm:  c := a != b
// Input:      a,b in Rational.
// Output:     c in bool such that if a = b then c = false else c = true ||
{
  return (a.numerator() != b.numerator() || a.denominator() != b.denominator());
}

inline bool operator<(const Rational& a, const Rational& b)
// Algorithm:  c := a < b
// Input:      a,b in Rational.
// Output:     c in bool such that if a < b then c = true else c = false ||
{
  return (a.numerator()*b.denominator() < a.denominator()*b.numerator());
}

inline bool operator<=(const Rational& a, const Rational& b)
// Algorithm:  c := a <= b
// Input:      a,b in Rational.
// Output:     c in bool such that if a <= b then c = true else c = false ||
{
  return (a.numerator()*b.denominator() <= a.denominator()*b.numerator());
}

inline bool operator>(const Rational& a, const Rational& b)
// Algorithm:  c := a > b
// Input:      a,b in Rational.
// Output:     c in bool such that if a > b then c = true else c = false ||
{
  return (a.numerator()*b.denominator() > a.denominator()*b.numerator());
}

inline bool operator>=(const Rational& a, const Rational& b)
// Algorithm:  c := a >= b
// Input:      a,b in Rational.
// Output:     c in bool such that if a >= b then c = true else c = false ||
{
  return (a.numerator()*b.denominator() >= a.denominator()*b.numerator());
}

inline bool operator==(const Rational& a, const Digit b)
// Algorithm:  c := a == b
// Input:      a in Rational, b in Digit.
// Output:     c in bool such that if a = b then c = true else c = false ||
{
  return (a.numerator() == b && a.denominator() == 1);
}

inline bool operator!=(const Rational& a, const Digit b)
// Algorithm:  c := a != b
// Input:      a in Rational, b in Digit.
// Output:     c in bool such that if a = b then c = false else c = true ||
{
  return (a.numerator() != b || a.denominator() != 1);
}

inline bool operator<(const Rational& a, const Digit b)
// Algorithm:  c := a < b
// Input:      a in Rational, b in Digit.
// Output:     c in bool such that if a < b then c = true else c = false ||
{
  return (a.numerator() < a.denominator()*b);
}

inline bool operator<=(const Rational& a, const Digit b)
// Algorithm:  c := a <= b
// Input:      a in Rational, b in Digit.
// Output:     c in bool such that if a <= b then c = true else c = false ||
{
  return (a.numerator() <= a.denominator()*b);
}

inline bool operator>(const Rational& a, const Digit b)
// Algorithm:  c := a > b
// Input:      a in Rational, b in Digit.
// Output:     c in bool such that if a > b then c = true else c = false ||
{
  return (a.numerator() > a.denominator()*b);
}

inline bool operator>=(const Rational& a, const Digit b)
// Algorithm:  c := a >= b
// Input:      a in Rational, b in Digit.
// Output:     c in bool such that if a >= b then c = true else c = false ||
{
  return (a.numerator() >= a.denominator()*b);
}

inline const Integer& numerator(const Rational& a)
// Algorithm:  c := numerator(a)
// Input:      a in Rational.
// Output:     c in Integer such that c = a_1 where a_1/a_2 = a ||
{
  return a.numerator();
}

inline const Natural& denominator(const Rational& a)
// Algorithm:  c := denominator(a)
// Input:      a in Rational.
// Output:     c in Natural such that c = a_2 where a_1/a_2 = a ||
{
  return a.denominator();
}

inline Rational abs(const Rational& a)
// Algorithm:  c := abs(a)
// Input:      a in Rational.
// Output:     c in Rational such that c = |a| ||
{
  return Rational(abs(a.numerator()), a.denominator());
}

inline int sign(const Rational& a)
// Algorithm:  c := sign(a)
// Input:      a in Rational.
// Output:     c in int such that if a = 0 then c = 0
//             else if a > 0 then c = 1 else c = -1 ||
{
  return sign(a.numerator());
}

inline Rational pow(const Rational& a, const Digit b)
// Algorithm:  c := pow(a, b)
// Input:      a in Rational, b in Digit.
// Output:     c in Rational such that c = a^b ||
{
  return Rational(pow(a.numerator(), b), pow(a.denominator(), b));
}

inline Rational::rep print(const Rational& a)
// Algorithm:  o := o << print(a)
// Input:      o in ostream, a in Rational.
// Output:     o in ostream ||
//
// Note:       puts internal representation of Rational a on an output stream.
{
  return Rational::rep(a.numerator(), a.denominator());
}

inline ostream& operator<<(ostream& out, const Rational::rep& a)
// puts internal representation of Rational a on output stream.
{
  return out << print(a.num) << '/' << print(a.den);
}

inline ostream& operator<<(ostream& out, const Rational& a)
// Algorithm:  o := o << a
// Input:      o in ostream, a in Rational.
// Output:     o in ostream ||
//
// Note:       puts Rational a on output stream.
{
  return out << a.numerator() << '/' << a.denominator();
}

inline Integer round(const Rational& a)
// Algorithm:  c := round(a)
// Input:      a in Rational.
// Output:     c in Integer such that c = floor(a+1/2) ||
{
  return floor(a + Rational(1, 2));
}


#endif
