
#include <assert.h>
#include <math.h>
#include <iostream.h>

#include "hfdata.h"
#include "auxid.h"
#include "mybuiltin.h"


// for debug:
#define PR(x)
#define PRCD(x)
#define PRL(x)
#define OVD(x)     // overwrite data before delete


double hfdata::nbytes = 0;
double hfdata::nbytesmax = 0;
long   hfdata::ndata = 0;
double hfdata::fxtw = 0;


int    hfdata::checkmult = 1;

// meaningless values to make sure initialisation works:
ulong  hfdata::rx = 0;       // 10000;
ulong  hfdata::ldrx = 0;     // 12
ulong  hfdata::rxbits = 0;   // 13
double hfdata::lg10rx = -1.0; // 4
int    hfdata::rx2pw = 0;     // 0
ulong  hfdata::rxdig = 0;     // 4
char*  hfdata::rxfmt = "";    // "%04d";
ulong  hfdata::rxbase = 0;    // 10
ulong  hfdata::rxbspw = 0;    // 4
void (* hfdata::mulcnvl)(double *, double *, ulong) = NULL;
void (* hfdata::sqrcnvl)(double *, ulong) = NULL;




hfdata::hfdata(ulong n)
//
// allocate data (n LIMBs) for this hfdata
//
{
    PRCD( cout<<"  "<< (void *)this<<"=hfdata("<< n <<") "<<endl; );
    dtdigit = NULL;
    dtsize = 0;

    dtmydat = 1;
    dtlinks = 0;

    size(n);

    ndata++;
}
//--------------


hfdata::hfdata(LIMB *d, ulong n)
//
// use allocated data (n LIMBs) for this hfdata
//
{
    PRCD( cout<<"  "<< (void *)this<<"=data(d,n) "<<endl; );

    assert( d );
    assert( n );
    dtdigit = d;
    dtsize  = n;

    dtmydat = 0;
    dtlinks = 0;

    ndata++;
}
//--------------


hfdata::~hfdata()
{
    PRCD( cout<<"  "<< (void *)this<<": ~data "<<endl; );

    assert( 0==dtlinks );

    if ( mydata() )  // delete old data
    {
        size(0);
    }
    else
    {
        PRL( cout<<"\n"<< (void *)this<<": forget [] "<<dtdigit<<endl; );
    }

    ndata--;
}
//--------------


void
hfdata::addlink()
//
// tell this hfdata that one more hfguts is looking
//
{
    dtlinks++;
}
//--------------


long
hfdata::sublink()
//
// tell this hfdata that one less hfguts is looking
//
    ////
    //// if 0 is returned then this hfdata should be deleted by the caller
    ////
{
    assert( dtlinks>0 );

    dtlinks--;

    return dtlinks;
}
//--------------


ulong
hfdata::size() const
{
    return dtsize;
}
//--------------


void
hfdata::size(ulong n)
{
    assert( mydata() );

    LIMB *oldd = dig();
    LIMB *newd;
    ulong osz = size();
    // if ( on<=n )  skip_it...

    if ( n==0 )
    {
        //        cout << "\n size(0) \n";
        newd = 0;

        if ( oldd )
        {
            PRL( cout<<"\n"<< (void *)this<<": delete [] "<<oldd<<endl; );
            delete [] oldd;
            nbytes -= osz*sizeof(LIMB);
        }
    }
    else
    {
        newd = new LIMB [n];
        PRL( cout<<"\n"<< (void *)this<<": new ["<<n<<"] "<<newd<<endl; );
        assert( newd );
        nbytes += n*sizeof(LIMB);

        if ( oldd )
        {
            i_copy(oldd,osz,newd,n);
            PRL( cout<<"\n"<< (void *)this<<": delete [] "<<oldd<<endl; );
            delete [] oldd;
            nbytes -= osz*sizeof(LIMB);
        }
        else
        {
            i_null(newd,n);
        }
    }

    dtdigit = newd;
    dtsize = n;

    if ( nbytes>nbytesmax )  nbytesmax=nbytes;
}
//--------------



void
hfdata::copy(const hfdata &h, ulong p)
{
    assert( dtlinks<=1 );  // strange if more than one is looking here
    assert( p<=dtsize );
    i_copy(h.dig(),h.size(),dtdigit,p);
}
//--------------


void
hfdata::shift_left(ulong n, ulong s)
{
    i_shift_left(dig(),n,s);
}
//--------------


void
hfdata::shift_right(ulong n, ulong s)
{
    i_shift_right(dig(),n,s);
}
//--------------


void
hfdata::rand(ulong n1, ulong n2)
{
    LIMB *d = dig();
    assert( d );

    ulong mi = MIN(n1,n2);
    mi = MIN(dtsize,mi);

    ulong ma = MAX(n1,n2);
    ma = MIN(dtsize,ma);

    i_rand(d+mi,ma-mi,radix()-1);
}
//--------------


void
hfdata::fill(ulong n1, ulong n2, LIMB f)
//
// fill LIMBs from n1...n2
// take care that 0 <= min(n1,n2) <= max(n1,n2) < dtsise
//
{
    LIMB *d = dig();
    assert( d );

    ulong mi = MIN(n1,n2);
    mi = MIN(dtsize,mi);

    ulong ma = MAX(n1,n2);
    ma = MIN(dtsize,ma);

    i_fill(d+mi,ma-mi+1,f);
}
//--------------


void
hfdata::fill(LIMB f)
{
    LIMB *d = dig();
    assert( d );
    i_fill(d,dtsize,f);
}
//--------------


LIMB *
hfdata::dig() const
{
    return dtdigit;
}
//--------------


void
hfdata::dump()
{
    cout<< "\n DIGIT[]= "<< dig()
        << "   size()="<< size()
	<< "   dtlink="<< dtlinks
	<< "   radix()="<< radix()
	<< endl;
}
//--------------


int
hfdata::radix() // const
{
    return rx;
}
//--------------


char *
get_radix_format(int r)
{
    static char fmt[22];
    int p;
    strcpy(fmt,"%05d,");

    fmt[2]='0'+(char)floor(log(r)/log(10)+1);

    if ( 0!=(p=ispowof(r,10)) )
    {
        fmt[2]=(char)('0'+p);
	fmt[4]=0;
    }

    if ( 0!=(p=ispowof(r,2)) )
    {
        fmt[3]='x';
	fmt[2]='0'+(char)floor(log(r)/log(16)+1);

	if ( 0!=(p=ispowof(r,16)) )
	{
	    fmt[2]='0'+(char)p;
	    fmt[4]=0;
	}
    }

    return fmt;
}
//--------------



void
hfdata::radix(int r, char *rfmt/*=0*/, int rdig/*=0*/)
{
    assert( (r>=2)
            *(int)" hfdata::setradix(): radix must be >=2 " );

    assert( (r<=(1<<(8*sizeof(LIMB))))
            *(int)" hfdata::setradix(): radix doesn't fit into LIMB " );
    rx=r;

    if ( !rfmt ) rxfmt=get_radix_format(rx);
    else         rxfmt=rfmt;

    if ( !rdig ) rxdig=5;
    else         rxdig=rdig;

    ldrx=ld(rx);
    lg10rx=LOG10(rx);

    rxbase = rx;
    rxbspw = 1;
    int p;

    p = ispowof(rx,10);
    if ( p )
    {
        rxbase = 10;
        rxbspw = p;
    }

    p = ispowof(rx,16);
    if ( p )
    {
        rxbase = 16;
        rxbspw = p;
    }


    if ( rx==((unsigned)1<<ldrx) )
    {
        rx2pw = 1;
	rxbits = ldrx;
    }
    else
    {
        rx2pw = 0;
	rxbits = ldrx+1;
    }
}
//--------------


int
hfdata::mydata() const
{
    return dtmydat;
}
//--------------


int
hfdata::check() const
{
    if ( mydata() )
    {
        assert( dtdigit );
        assert( dtsize );
    }

    if ( dtsize )  assert( dtdigit );
    else           assert( !dtdigit );


/*
    LIMB *d=dig();
    for (ulong k=0; k<size(); k++)  // performance killer
    {
        if (d[k]>=RADIX)
	{
	  cerr<<"\n hfdata::check(): limb["<<k<<"]="
              <<d[k]<<" >=RADIX !!!"<<endl;
          assert(0);
        }
    }
*/

    assert( dtlinks>=0 );

    assert( ndata>=0 );

    assert( (rx>0) && (rx<=65536) );

    return 0;
}
//--------------



void
hfdata::statistics(ulong n)
{
    cout << endl;

    cout << "bytes currently allocated: " << hfdata::nbytes
         << endl
         << "max bytes allocated:" << hfdata::nbytesmax
        //	 << "    ndata=" << hfdata::ndata
         << " (plus workspace)"
         << endl;


    cout << "work was=" 
         << hfdata::fxtw / (1024.0*1024.0)
         << " times length 2^20 real FFTs"
         << endl;


    if ( n )
    {
        cout << "or " 
             <<  (hfdata::fxtw) / (3*2*n*log(2*n)/log(2.0)) 
             << " full prec mults "
             << endl;
    }


    if ( return_elapsed_time() > 0 )
    {
        cout << "bogo_Mfxtstone="
	     <<  (hfdata::fxtw) / (return_elapsed_time()*1024.0*1024.0) 
	     <<  endl;
    }

}
//--------------
