
#include <math.h>
#include <assert.h>

#include "fxtaux.h"
#include "fxt.h"

double *wr, *wi;  // workspace

// always call fft_ndim(), don't call e.g. fft_3dim() directly !

void fft_2dim(double *fr, double *fi, ulong d1, ulong d2, int is);
void fft_3dim(double *fr, double *fi, ulong d1, ulong d2, ulong d3, int is);
void fft_4dim(double *fr, double *fi, ulong d1, ulong d2, ulong d3, ulong d4, int is);
void fft_5dim(double *fr, double *fi, ulong d1, ulong d2, ulong d3, ulong d4, ulong d5, int is);

#define FFT(re,im,n,is)  fht_fft(re,im,ld(n),is)


void
fft_ndim(double *fr, double *fi, ulong ndim, ulong *ldn, int is)
//
// ndim must be 1,2,3,4 or 5
// ldn[] contains base 2 logarithms of dimensions
//
{
    if( (ndim<1) || (ndim>5) )
    {
        assert(0*(int)" invalid ndim in ndim_fft ");
    }

    if(ndim==1)  // 1-dim fft
    {
        FFT(fr,fi,1<<ldn[0],is);
        return;
    }

    // find max(d1,d2,...,dn):
    ulong nw=0;
    for(ulong k=0; k<ndim; ++k)
    {
        if( ldn[k]>nw )  nw=ldn[k];
    }
    nw=(1<<nw);

    wr=new double[nw];
    wi=new double[nw];

    ulong d1=(1<<ldn[0]);
    ulong d2=(1<<ldn[1]);
    if(ndim==2)
    {
        fft_2dim(fr,fi,d1,d2,is);
    }
    else
    {
        ulong d3=(1<<ldn[2]);
        if(ndim==3)
        {
            fft_3dim(fr,fi,d1,d2,d3,is);
        }
        else
        {
            ulong d4=(1<<ldn[3]);
            if(ndim==4)
            {
                fft_4dim(fr,fi,d1,d2,d3,d4,is);
            }
            else
            {
                ulong d5=(1<<ldn[4]);
                if(ndim==5)
                {
                    fft_5dim(fr,fi,d1,d2,d3,d4,d5,is);
                }
            }
        }
    }

    delete [] wr;
    delete [] wi;
}
// =============== end FFT_NDIM =========== 


void
fft_2dim(double *fr, double *fi, ulong d1, ulong d2, int is)
{
    ulong n=d1*d2;

    ulong k;
    for(k=0; k<n; k+=d1)      // rows
    {
        FFT(fr+k,fi+k,d1,is);
    }

    for(k=0; k<d1; k++)       // columns
    {
        wide_fft(fr+k,fi+k,d2,d1,wr,wi,is);
    }
}
// =============== end FFT_2DIM =========== 



void
fft_3dim(double *fr, double *fi, ulong d1, ulong d2, ulong d3, int is)
{
    ulong d12=d1*d2;
    ulong n=d12*d3;

    ulong k;
    for(k=0; k<n; k+=d12)
    {
        fft_2dim(fr+k,fi+k,d1,d2,is);
    }

    for(k=0; k<d12; k++)
    {
        wide_fft(fr+k,fi+k,d3,d12,wr,wi,is);
    }
}
// =============== end FFT_3DIM =========== 



void
fft_4dim(double *fr, double *fi, ulong d1, ulong d2, ulong d3, ulong d4, int is)
{
    ulong d123=d1*d2*d3;
    ulong n=d123*d4;

    ulong k;
    for(k=0; k<n; k+=d123)
    {
        fft_3dim(fr+k,fi+k,d1,d2,d3,is);
    }

    for(k=0; k<d123; k++)
    {
        wide_fft(fr+k,fi+k,d4,d123,wr,wi,is);
    }
}
// =============== end FFT_4DIM =========== 



void
fft_5dim(double *fr, double *fi, ulong d1, ulong d2, ulong d3, ulong d4, ulong d5, int is)
{
    ulong d1234=d1*d2*d3*d4;
    ulong n=d1234*d5;

    ulong k;

    for(k=0; k<n; k+=d1234)
    {
        fft_4dim(fr+k,fi+k,d1,d2,d3,d4,is);
    }

    for(k=0; k<d1234; k++)
    {
        wide_fft(fr+k,fi+k,d5,d1234,wr,wi,is);
    }
}
// =============== end FFT_5DIM =========== 

