
#include <math.h>
#include <assert.h>
#include <stdlib.h>

#include "hfloatfu.h"
#include "hfverbosity.h"


hfloat
sqrt(const hfloat &h)
{
    hfloat h3;
    sqrt(h,h3);
    return h3;
}
// -----------------


hfloat
isqrt(const hfloat &h)
{
    hfloat h3;
    isqrt(h,h3);
    return h3;
}
// -----------------


hfloat
root(const hfloat &h, long r)
{
    hfloat h3;
    root(h,r,h3);
    return h3;
}
// -----------------


hfloat
iroot(const hfloat &h, ulong r)
{
    hfloat h3;
    iroot(h,r,h3);
    return h3;
}
// -----------------


hfloat
cbrt(const hfloat &h)
{
    hfloat h3;
    cbrt(h,h3);
    return h3;
}
// -----------------




static hfloat xi(64);  // for the int-variants


void
poly_root(hfloat &x, ulong deg, const long *num, const long *den)
{
    poly_root_iteration(x,deg,num,den,0);
}
//===================== end POLY_ROOT ========================


void
root(const hfloat &a, long e, hfloat &c, ulong startprec) // =0
{
    if (e>=0) 
    {
        if (e>3)  rootn(a,e,c,startprec);

        if (e==3)
        {
	    if (startprec!=0)  rootn(a,e,c,startprec);
	    else               cbrt(a,c);
	}

	if (e==2)  sqrt(a,c,startprec);

        if (e==1)  copy(a,c);
    
        if (e==0)  i2hfloat(1,c);
    }
    else  // --- negative => inv root
    {
	if (e==-1)  inv(a,c,startprec);

	if (e==-2)  isqrt(a,c,startprec);

        if (e<=-3)  iroot(a,-e,c,startprec);
    }
}
//======================= end ROOT =============================


void
root(long i, long r, hfloat &c, ulong startprec) // =0
//
// uses static xi
//
{
    i2hfloat_prec_adj(i,xi);
    root(xi,r,c,startprec);
}
//===================== end ROOT ============================



void
isqrt(const hfloat &d, hfloat &c, ulong startprec) // =0
//
// startprec here for the inv sqrt
// (which has to be in c)
//
{
    //    isqrt_iteration(d,c,startprec);
    iroot_iteration(d,2,c,startprec);
}
//===================== end ISQRT ============================


void
isqrt(long i, hfloat &c, ulong startprec) // =0
//
// uses static xi
//
{
    i2hfloat_prec_adj(i,xi);
    isqrt(xi,c,startprec);
}
//===================== end ISQRT ============================



void
sqrt(const hfloat &d, hfloat &c, ulong startprec) // =0
//
// startprec here for the _inverse_ sqrt !
// (which has to be in c)
//
{
    if ( !same_mantissa(c,d) )
    {  
        //        isqrt_iteration(d,c,startprec);
        iroot_iteration(d,2,c,startprec);
        mul(c,d,c);
    }
    else // inplace operation ...
    {
        hfloat t(c.prec());
        //        isqrt_iteration(d,t,startprec);
        iroot_iteration(d,2,t,startprec);
        mul(d,t,c);
    }
}
//===================== end SQRT ================================


void
sqrt(long i, hfloat &c, ulong startprec) // =0
//
// uses static xi
//
{
    i2hfloat_prec_adj(i,xi);
    sqrt(xi,c,startprec);
}
//===================== end SQRT ============================



void
iroot(const hfloat &d, ulong r, hfloat &c, ulong startprec) // =0
//
// startprec here for the inv root
// (which has to be in c)
//
{
    iroot_iteration(d,r,c,startprec);
}
//===================== end IROOT ============================


void
iroot(long i, ulong r, hfloat &c, ulong startprec) // =0
//
// uses static xi
//
{
    i2hfloat_prec_adj(i,xi);
    iroot(xi,r,c,startprec);
}
//===================== end IROOT ============================



void
rootn(const hfloat &d, long r, hfloat &c, ulong startprec) // =0
//
// startprec here for the inv root !
// (which has to be in c)
//
// for any r that factors nontrivial the inversion can be avoided
// e.g.: root(d,6)==invroot(invroot(d,2),3)
// i.e.: d^(1/6)==(d^(-(1/2))^(-1/3)     
//
//
{
    iroot_iteration(d,r,c,startprec);
    inv(c,c);
}
//===================== end ROOTN ================================



void
cbrt(const hfloat &d, hfloat &c) 
//
// 
// d^((r-1)/r) is computed then a 
// final multiplication with d gives d^(1/r)
//
{
    hfloat cl(c.prec());
    hfloat dp(c.prec());
    sqr(d,dp);
    iroot_iteration(dp,3,cl,0);
    mul(cl,d,c);
}
//===================== end CBRT ================================


void
cbrt(long i, hfloat &c)
//
// uses static xi
//
{
    i2hfloat_prec_adj(i,xi);
    cbrt(xi,c);
}
//===================== end IROOT ============================


void
root_x(const hfloat &d, ulong r, hfloat &c) 
//
// alternative n-th root routine: 
// may be good if 
// one wants the hi prec root of a few-digits number
// or generally for the third root (see above)
// 
// d^((r-1)/r) is computed then a 
// final multiplication with d gives d^(1/r)
//
// LITTLE TESTED !
{
    hfloat cl(c.prec());
    hfloat dp(c.prec()); 
    pow(d,r-1,dp);
    iroot_iteration(dp,r,cl,0);
    mul(cl,d,c);
}
//===================== end ROOT_X ================================


void
root_x(long i, ulong r, hfloat &c)
//
// uses static xi
//
{
    i2hfloat_prec_adj(i,xi);
    root_x(xi,r,c);
}
//===================== end IROOT ============================

