#if !defined __HFLOATFU_H
#define      __HFLOATFU_H

#include <iostream.h>

#include "hfloat.h"  // class hfloat

// whether forbid unnormalized in/output in some functions:
#define FORBID_UNNORMALIZED  1  // 1 (default) or 0


// ----------------------------------------------
// hfloatop.cc: (part I, efficient stuff)
// x,y,z denote hfloats,
// i denotes a (long) integer
// d debotes a double
hfloat & operator += (hfloat &, const hfloat &);   // x += y
hfloat & operator -= (hfloat &, const hfloat &);   // x -= y
hfloat & operator *= (hfloat &, const hfloat &);   // x *= y
hfloat & operator /= (hfloat &, const hfloat &);   // x /= y

hfloat & operator += (hfloat &, long);  // x += i
hfloat & operator -= (hfloat &, long);  // x -= i
hfloat & operator *= (hfloat &, long);  // x *= i
hfloat & operator /= (hfloat &, long);  // x /= i

int operator == (const hfloat &, const hfloat &);  // x == y
int operator == (long, const hfloat &);            // i == x
int operator == (const hfloat &, long);            // x == i

int operator != (const hfloat &, const hfloat &);  // x != y
int operator != (long, const hfloat &);            // i != x
int operator != (const hfloat &, long);            // x != i

int operator >  (const hfloat &, const hfloat &);  // x > y
int operator >  (long, const hfloat &);            // i > x
int operator >  (const hfloat &, long);            // x > i
int operator >  (double, const hfloat &);          // d > x
int operator >  (const hfloat &, double);          // x > d

int operator >= (const hfloat &, const hfloat &);  // x >= y
int operator >= (long, const hfloat &);            // i >= x
int operator >= (const hfloat &, long);            // x >= i

int operator <  (const hfloat &, const hfloat &);  // x < y
int operator <  (long, const hfloat &);            // i < x
int operator <  (const hfloat &, long);            // x < i
int operator <  (double, const hfloat &);          // d < x
int operator <  (const hfloat &, double);          // x < d

int operator <= (const hfloat &, const hfloat &);  // x <= y
int operator <= (long, const hfloat &);            // i <= x
int operator <= (const hfloat &, long);            // x <= i

hfloat operator + (const hfloat &);  // +x
hfloat operator - (const hfloat &);  // -x

istream&  operator >> (istream& is, hfloat& x);        // is >> x
ostream&  operator << (ostream& os, const hfloat& x);  // os << x

// ======================= inefficient stuff
//  (temporary hfloats are created, see the docs)
// instead of 'x=a*b;' use 'x=a; x*=b;' or use 'mul(a,b,x)' ! 

// ----------------------------------------------
// hfloatop.cc: (part II, inefficient stuff)

// x,y,z denote hfloats,  i denotes a (long) integer
hfloat operator + (const hfloat &, const hfloat &);  // x = y+z
hfloat operator - (const hfloat &, const hfloat &);  // x = y-z
hfloat operator * (const hfloat &, const hfloat &);  // x = y*z
hfloat operator / (const hfloat &, const hfloat &);  // x = y/z
hfloat operator + (const hfloat &, long i);  // x = y+i
hfloat operator + (long i, const hfloat &);  // x = i+y
hfloat operator - (const hfloat &, long i);  // x = y-i
hfloat operator - (long i, const hfloat &);  // x = i-y
hfloat operator * (const hfloat &, long i);  // x = y*i
hfloat operator * (long i, const hfloat &);  // x = i*y
hfloat operator / (const hfloat &, long i);  // x = y/i
hfloat operator / (long i, const hfloat &);  // x = i/y

// ======================= (end inefficient stuff)


// ----------------------------------------------
// init.cc:
// constructor and destructor for hfloat_init class

// ----------------------------------------------
// inform.cc:
int inform_user(ulong n, int fakeq); // print some info


// ----------------------------------------------
// print.cc:

// hae: all void returns replaced by int

// print n LIMBs of r:
int print(const hfloat &r, ulong n=0);

// print what[] and n LIMBs of r:
int print(const char *what, const hfloat &a, ulong n=0);

// print what[] and the n last LIMBs of r:
int print_last(const char *what, const hfloat &a, ulong n);

// save to os n LIMBs of r:
int save(ostream &os, const hfloat &r, ulong n=0);

// save to os what[] and n LIMBs of r:
int save(ostream &os, const char *what, const hfloat &a, ulong n=0);

// save to os what[] and the n last LIMBs of r:
int save_last(ostream &os, const char *what, const hfloat &a, ulong n);

// save to file named fname[] r:
int save(const char* fname, const hfloat &r);


// ----------------------------------------------
// binsave.cc:
// binary save to file named fname[] the LIMBs of r:
int binsave_limbs(const char *fname, const hfloat &a, long n=0, long s=0);


// ----------------------------------------------
//convert.cc
int i2hfloat(long i, hfloat &x);            // x = i, use the '=' operator
int u2hfloat(ulong i, hfloat &x);           // x = i, use the '=' operator
void i2hfloat_prec_adj(long i, hfloat &xi); // x = i, sets precision 
int hfloat2i(const hfloat &x, long &i);     // i = x
int d2hfloat(double d, hfloat &x);          // x = d, use the '=' operator
int hfloat2d(const hfloat &x, double &d);   // d = x
int st2hfloat(const char *str, hfloat &x);  // x = str, use the '=' operator
// ... and of course there is (hfloat)x = (hfloat)y 


// ----------------------------------------------
//compare.cc:
long cmp(const hfloat &a, const hfloat &b);        // ? a == b
long cmp1(const hfloat &a, ulong startdig=0);      // ? a == 1
long cmp_limbs(const hfloat &a, const hfloat &b);  // ? how many LIMBs agree

// whether a==d within relative error relerr
int approximate(const hfloat &a, double d, double relerr=1.0e-12);


// ----------------------------------------------
//funcsadd.cc:
int add(const hfloat &a, const hfloat &b, hfloat &c);  // c = a+b
int add(const hfloat &a, long i, hfloat &c);           // c = a+i
int sub(const hfloat &a, const hfloat &b, hfloat &c);  // c = a-b
int sub(const hfloat &a, long i, hfloat &c);           // c = a-i
int sub(long i, const hfloat &a, hfloat &c);           // c = i-a


// ----------------------------------------------
//funcsmul.cc:
hfloat sqr(const hfloat &a);                             //  = a^2
hfloat mul(const hfloat &h1, const hfloat &h2);          //  = a*b
void sqr(const hfloat &a, hfloat &c, ulong pa=0, ulong pc=0);  // c = a^2
void mul(const hfloat &a, const hfloat &b, hfloat &c,
	ulong pa=0, ulong pb=0, ulong pc=0);                  // c = a*b
void mul(const hfloat &a, long i, hfloat &c);                  // c = a*i
void mul_by_radix_pow(const hfloat &x, int ex, hfloat &c);     // c = x*radix^ex


// ----------------------------------------------
//funcsdiv.cc:
hfloat inv(const hfloat &a);                                        //  = 1/a
void inv(const hfloat &a, hfloat &c, ulong stpr=0);                 // c = 1/a
void inv(long i, hfloat &c, ulong stpr=0);                          // c = 1/i
void div(const hfloat &a, const hfloat &b, hfloat &c, ulong stpr=0);// c = a/b
void div(const hfloat &a, long i, hfloat &c, int *rem=0);           // c = a/i, rem = a%i
void fmod(const hfloat &a, const hfloat &b, hfloat &c);             // c = a%b
void fmod(const hfloat &a, const hfloat &b, hfloat &c, hfloat &d);  // c = a/b, d = a%b
void div_by_radix_pow(const hfloat &x, int ex, hfloat &a);          // c = x*radix^(-ex)


// ----------------------------------------------
//funcspow.cc:
hfloat pow(const hfloat &a, long  ex);                //  = a^ex
void pow(const hfloat &a, long x, hfloat &c);         // c = a^x, x in Z
void pow_geq_3(const hfloat &a, ulong x, hfloat &c);  // c = a^x, x>=+3


// ----------------------------------------------
//funcsrt.cc:
hfloat sqrt(const hfloat &a);           //  = a^(1/2)
hfloat isqrt(const hfloat &a);          //  = a^(-1/2)
hfloat root(const hfloat &a, long r);   //  = a^(1/r)
hfloat iroot(const hfloat &a, long r);  //  = a^(-1/r)
hfloat cbrt(const hfloat &d);           //  = a^(1/3)

void sqrt(const hfloat &d, hfloat &c, ulong stpr=0);  // c = d^(1/2)
void sqrt(long i, hfloat &c, ulong stpr=0);           // c = i^(1/2)
void isqrt(const hfloat &d, hfloat &c, ulong stpr=0); // c = d^(-1/2)
void isqrt(long i, hfloat &c, ulong stpr=0);          // c = i^(1/2)
void root(const hfloat &d, long r, hfloat &c, ulong stpr=0);   // c = d^(1/r), r in Z
void root(long i, long r, hfloat &c, ulong stpr=0);            // c = i^(1/r), r in Z
void iroot(const hfloat &d, ulong r, hfloat &c, ulong stpr=0); // c = d^(-1/r), r in N
void iroot(long i, ulong r, hfloat &c, ulong stpr=0);          // c = i^(-1/r), r in N
void cbrt(const hfloat &d, hfloat &c);                  // c = d^(1/3)
void cbrt(long i, hfloat &c);                           // c = i^(1/3)
void root_x(const hfloat &d, ulong r, hfloat &c);       // c = d^(1/r), r in N
void root_x(long i, ulong r, hfloat &c);                // c = i^(1/r), r in N
void rootn(const hfloat &d, long r, hfloat &c, ulong stpr=0); // c = d^(1/r), r in Z

// x = root of poly \sum_{k=0}^{deg}{num[k]/den[k]*x^k}:
void poly_root(hfloat &x, ulong deg, const long *num, const long *den);


// ----------------------------------------------
//round.cc:
int integerq(const hfloat &a);          // ? a = floor(a)
int round(const hfloat &a, hfloat &b);  // b = floor(a+0.5)

//round0.cc:
int round0(const hfloat &a, hfloat &b); // round towards zero
int frac0(const hfloat &a, hfloat &b);  // b = a-round0(a)

//floor.cc:
int floor(const hfloat &a, hfloat &b);  // b = floor(a)
int frac(const hfloat &a, hfloat &b);   // b = a-floor(a)



// ----------------------------------------------
//contfrac.cc:

// cf[0..n-1]= n terms of the  simple continued fraction of x:
int hfloat_to_contfrac(const hfloat &x, ulong n, long cf[]);

// ratio from first n terms of continued fraction cf[]: 
int contfrac_to_ratio(const long cf[], ulong n, hfloat &p, hfloat &q);

// x= n-th partial fraction of the simple continued fraction cf[]:
int contfrac_to_hfloat(const long cf[], ulong n, hfloat &x);


// ----------------------------------------------
//funcs2pow.cc:
hfloat ldexp(const hfloat &x, long ex);            //  = a*2^ex
void ldexp(const hfloat &x, long ex, hfloat &a);   // a = x*(2^ex)
void mul2(const hfloat &x, hfloat &a);             // a = x*2
void mul2pow(const hfloat &x, long ex, hfloat &a); // a = x*(2^ex)
void div2(const hfloat &x, hfloat &a);             // a = x/2
void div2pow(const hfloat &x, long ex, hfloat &a); // a = x/(2^ex)


// ----------------------------------------------
// mulpow.cc:
void mulpow(const hfloat &x, long b, long ex, hfloat &a); // a = x*(b^ex)



// ----------------------------------------------
//polynom.cc:

// p = \sum_{k=0}^{deg}{num[k]/den[k]*x^k}:
void polynom(const hfloat &x, ulong deg, const long *num, const long *den, hfloat &p);

// dnmu[], dden[] = coefficients of derived poly with coefficients num[]/den[]
void poly_derive(ulong deg, const long *num, const long *den, long *dnum, long *dden);

void print_poly(const char *bla, ulong deg, const long *num, const long *den);


// ----------------------------------------------
// sumalt.cc:

// s = \sum_{k=0}^{\infty}{termfct(k)}:
void sumalt( void (*termfct)(ulong, hfloat &), hfloat &s);


// ----------------------------------------------
// tz/funcstrz.cc:
hfloat log(const hfloat &a);            //  = log(a)
hfloat exp(const hfloat &a);            //  = exp(a)
void log(const hfloat &a, const hfloat &b, hfloat &c); // c = log_b(a)
void log(const hfloat &a, hfloat &b);              // b = log(a)
void acosh(const hfloat &a, hfloat &b);            // b = acosh(a)
void asinh(const hfloat &a, hfloat &b);            // b = asinh(a)
void exp(const hfloat &a, hfloat &b, ulong stpr=0);// b = exp(a)
void cosh(const hfloat &a, hfloat &b);             // b = cosh(a)
void sinh(const hfloat &a, hfloat &b);             // b = sinh(a)

// ----------------------------------------------
// tz/mean.cc:

// x = arithmetic_geometric_mean(a,b):
int agm(const hfloat &a, const hfloat &b, hfloat &x);

// x = harmonic_geometric_mean(a,b):
int hgm(const hfloat &a, const hfloat &b, hfloat &x);

// x = j_mean(a,b):
int j_mean(const hfloat &a, const hfloat &b, hfloat &x);

// ----------------------------------------------
// tz/power.cc:

hfloat pow(const hfloat &a, const hfloat &ex);             //  = a^ex
void   pow(const hfloat &a, const hfloat &ex, hfloat &t);  //  t = a^ex


// ==============================================
// semi internal stuff, might be occasionally useful:

// tz/hfconst.cc:
const hfloat & constant_pi(ulong n);     // pi
const hfloat & constant_logrx(ulong n);  // log(radix)
const hfloat & constant_log2(ulong n);   // log(2)

// pi/pi4th.c:
int pi_4th_order(hfloat &pi, int var=0);

// pi/pi5th.c:
int pi_5th_order(hfloat &pi);

// pi/piagm.cc:
int pi_agm(hfloat &pi, int var=0);

// pi/piagm3.cc:
int pi_agm3(hfloat &pi, int var=4);

// pi/pi2nd.cc:
int pi_2nd_order(hfloat &p);

// pi/pideriv.cc:
int pi_derived_agm(hfloat &pi);

// pi/piarctan.cc:
int pi_arctan(hfloat &pi, int fno);

// pi/pi3rd.cc:
int pi_3rd_order(hfloat &p);

// pi/pi9th.cc:
int pi_9th_order(hfloat &p);

// pi/picubagm.cc:
int pi_cubic_agm(hfloat &p);



// ==============================================
// internal stuff, better don't use:

// ----------------------------------------------
// hfaux.cc:
void copy(const hfloat &src, hfloat &dst);              // dst = src
int  same_mantissa(const hfloat &a, const hfloat &b);   // ? same mantissa
void rand(hfloat &a);                                   // a = (not so very) random
void dump(const char *bla, const hfloat &a, long prn);  // (for debug)



// ----------------------------------------------
// tz/logagm.cc:
int log_agm(const hfloat &a, hfloat &z);  // z = log(a)


// ----------------------------------------------
// tz/itexp.cc:
int exp_iteration(const hfloat &x, hfloat &b, ulong stpr);


// ----------------------------------------------
// tz/rprm.cc:
int r_prime(const hfloat &b0, hfloat &s, ulong nn, hfloat *m=0, int sum0q=0, long vq=0);


// ----------------------------------------------
// tz/rprm4th.cc:
int r_prime_4th(const hfloat &b0, hfloat &s, ulong nn, hfloat *m=0, long vq=0);


// ----------------------------------------------
//itinv.cc:
int inv_iteration(const hfloat &d, hfloat &c, ulong stpr);


// ----------------------------------------------
//itinv.cc:
int isqrt_iteration(const hfloat &d, hfloat &c, ulong stpr);


// ----------------------------------------------
//itnroot.cc:
int iroot_iteration(const hfloat &d, ulong r, hfloat &c, ulong stpr);


// ----------------------------------------------
//itpoly.cc:
int poly_root_iteration(hfloat &x, ulong deg, const long *num, const long *den, ulong stpr);


// ----------------------------------------------
// pilogq.cc:
void pi_by_log_q(long n, hfloat &x, ulong q);  // pi/log(q)
void theta3_qinv(ulong q, hfloat &th3);        // theta3(1/q)
void theta2_qinv(ulong q, hfloat &th2);        // theta2(1/q)



#endif // !defined __HFLOATFU_H
