/*
 *	File: ad2Zipf.c
 *
 *      (C) IWTS
 *          KU Nijmegen
 *          The Netherlands
 *
 *      Author: R. Harald Baayen
 *
 *      History:
 *
 *      - feb 1997, version 1.0
 *
 *      Description:
 *
 *      
 */

#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <errno.h>
#include <malloc.h>
#include <string.h>
#include "lex_cons.h"


/* EXTERN FUNCTIONS */

/* functions for numerical procedures */

extern double	expV ();
extern double	expVm ();
extern double	funcVm ();
extern double	getZ2 ();
extern double	getlexprop ();
extern double	adjustV ();
extern double	adjustVm ();
extern double	qromb ();

/* argument reading, file manipulation, and help function */

extern int	leesgetal ();
extern double	leesdouble ();
extern void	change_extension ();
extern void	help ();


/* GLOBAL VARIABLES */

double    N, V, n1, n2,              /* number of tokens, types, hapaxes, disleg */
         E,                         /* extrapolation sample size */
         pstar,                     /* maximum relative frequency */
         mmax,                      /* highest frequency rank */
         mrank,                     /* word frequency, used in expVm() */
         SPECTRUM[MAXM1][3],        /* frequency spectrum m Vm EVm */
         Z, VZ,                     /* Zipf size, and V(Z) */
         Nzero, Vzero,              /* original sample size N0 */
         eV, eV2N, S,               /* E[V(N)], E[V(2N)], S */
         OBSERVED[MAXCHUNKS3+1][6], /* V and V1..5 */
         lexprop,                   /* proportion specialized words */
         mse,                       /* MSE for specialization fit */
         CHUNKS[MAXCHUNKS3],        /* the chunk sizes */
         chunksize, remainDer,      /* chunk variables */
         x, x1, y, y2, y3, y4, y5,
	 u_bound;		    /* upper integration intervals */

FILE     *fpspectrum,               /* fpspectrum: datafile: m, Vm */
         *fpexpspect,               /* expected spectrum */
         *fpexpspect2N,             /* spectrum at 2N */
         *fpprofile,                /* observed developmental profile */
         *fpVN,                     /* file with E[V(N)] and E[V(2N)] */
         *fpsum,                    /* file with summary of model */
         *fpint,                    /* interpolation results */
         *fpext,                    /* extrapolation results */
         *fpE;                      /* spectrum at sample size E */

int      nranks,                    /* number of different ranks in spectrum */
         maxrank,                   /* largest rank for fit, default 15 */
         i, j,                      /* counter */
         header,                    /* boolean for presence of header */
         k,                         /* variable for chunks */
         nchunks,                   /* number of chunks for interpolation */
         enchunks,                  /* number of chunks for extrapolation */
         token, type,               /* auxiliary variables for scanf */
         aantal;                    /* for command line options */

char     woord[MAXWORDLENGTH],      /* variable for skipping header in fscanf */
         new_name[MAXWORDLENGTH],   /* variables for extension handling    */
         base_name[MAXWORDLENGTH],
         *fs;                       /* variable for scanning options */


/* MAIN () */

int main (argc, argv)
int     argc;
char    *argv[];

{ 
   /* DEFAULTS */

   maxrank = DEF_MAXRANK;
   nchunks = DEF_CHUNKS;
   enchunks = DEF_CHUNKS;
   header = 1;
   E = NULL_F;
   Z = NULL_F;
   u_bound = U_BOUND;

   /* COMMAND LINE OPTIONS */

   while ((--argc > 0) && ((*++argv)[0] == '-')) {
        for (fs = argv[0] + 1; *fs != '\0'; fs++) {
            switch (*fs) {
            case 'h':
                help();
                break;
            case 'Z':
                Z =  leesdouble (fs, &aantal);
                for (; aantal > 0; aantal--){
                   fs++;
                }
                break;
            case 'E':
                i =  leesgetal (fs, &aantal);
                E = (double) i;
                for (; aantal > 0; aantal--){
                   fs++;
                }
                break;
            case 'k':
                nchunks = leesgetal (fs, &aantal);
                for (; aantal > 0; aantal--){
                   fs++;
                }
                break;
            case 'K':
                enchunks = leesgetal (fs, &aantal);
                for (; aantal > 0; aantal--){
                   fs++;
                }
                break;
            case 'i':
                u_bound = (double) leesgetal (fs, &aantal);
                for (; aantal > 0; aantal--){
                   fs++;
                }
                break;
            case 'm':
                maxrank = leesgetal (fs, &aantal);
                for (; aantal > 0; aantal--){
                   fs++;
                }
                break;
            case 'H':      /* input files without headers! */
                header = 0;
                break;
            default:
                fprintf(stderr, "ad2Zipf: illegal option %c\n", *fs);
                exit(1);
                break;
            }
        }
   } /* of while */

   /* FILE HANDLING */

   if (argc == 0) {
     help();
   }

   /* load input spectrum, should have .spc extension */

   if ((fpspectrum = fopen(*argv, "r")) == NULL) {
       fprintf(stderr, "ad2Zipf: can't open %s\n", *argv);
       exit(1);
   }

   strncpy(base_name, *argv, strlen(*argv) - 4);

   change_extension (base_name, new_name, ".obs");
   if ((fpprofile = fopen(new_name, "r")) == NULL) {
       fprintf(stderr, "ad2Zipf: can't open profile file %s\n", new_name);
       exit(1);
   }

   /* file name handling output files */

   change_extension (base_name, new_name, "_bZ.spc");
   if ((fpexpspect = fopen(new_name, "w")) == NULL){
      fprintf(stderr, "ad2Zipf: can't open output file %s\n", new_name);
      exit(1);
   }
   change_extension (base_name, new_name, "_bZ.sp2");
   if ((fpexpspect2N = fopen(new_name, "w")) == NULL){
      fprintf(stderr, "ad2Zipf: can't open output file %s\n", new_name);
      exit(1);
   }
   change_extension (base_name, new_name, "_bZ.ev2");
   if ((fpVN = fopen(new_name, "w")) == NULL){
      fprintf(stderr, "ad2Zipf: can't open output file %s\n", new_name);
      exit(1);
   }
   change_extension (base_name, new_name, "_bZ.sum");
   if ((fpsum = fopen(new_name, "w")) == NULL){
      fprintf(stderr, "ad2Zipf: can't open output file %s\n", new_name);
      exit(1);
   }
   change_extension (base_name, new_name, "_bZ.int");
   if ((fpint = fopen(new_name, "w")) == NULL){
      fprintf(stderr, "ad2Zipf: can't open output file %s\n", new_name);
      exit(1);
   }
   change_extension (base_name, new_name, "_bZ.ext");
   if ((fpext = fopen(new_name, "w")) == NULL){
      fprintf(stderr, "ad2Zipf: can't open output file %s\n", new_name);
      exit(1);
   }
   if (E > NULL_F){
     change_extension (base_name, new_name, "_bZ.sp3");
     if ((fpE = fopen(new_name, "w")) == NULL){
        fprintf(stderr, "ad2Zipf: can't open output file %s\n", new_name);
        exit(1);
     }
   }

   /* LOAD SPECTRUM FILE */

   nranks = 0; n1 = 0; n2 = 0;
   if (header){
      fscanf(fpspectrum, "%s ", woord);  /* m */
      fscanf(fpspectrum, "%s ", woord);  /* Vm */
   }

   while (fscanf(fpspectrum, "%d %d", &token, &type) != EOF)  {
        nranks++;
        SPECTRUM[nranks][0] = (double) token;
        SPECTRUM[nranks][1] = (double) type;
        if (token == 1) n1 = type;
        if (token == 2) n2 = type;
        N+= (double) token * (double) type;
        V+= (double) type;
   }
   mmax = SPECTRUM[nranks][0];
   pstar = mmax/N;
   Nzero = N; Vzero = V;

   /* DETERMINE THE PARAMETER OF THE MODEL */

   if (Z == NULL_F) {
     Z = getZ2 (pstar, Nzero, Vzero, N);
   }

   VZ = Z / log (pstar * Z);

   /* read observed profile */

   fprintf(stdout, "reading profile\n");
   fflush(stdout);
   if (header){
       for (i = 1; i <= NMEASURES+1; i++){
          if (fscanf(fpprofile, "%s ", woord) == EOF){
            fprintf(stderr, "incomplete header for %s.obs\n", base_name);
          }
       }
   }

  
   for (i = 1; i <= nchunks; i++){
       if (fscanf(fpprofile, "%d ", &type) != EOF){
           ;
       }
       else{
           fprintf(stderr, "%s.obs is incompatible with required number of chunks\n", base_name);
           exit(1);
       }
       for (j = 2; j <= NMEASURES+1; j++){
           if (fscanf(fpprofile, "%lf ", &x) != EOF){
               switch (j){
                 case 4:
                   OBSERVED[i][0] = x;
                   break;
                 case 5:
                   OBSERVED[i][1] = x;
                   break;
                 case 6:
                   OBSERVED[i][2] = x;
                   break;
                 case 7:
                   OBSERVED[i][3] = x;
                   break;
                 case 8:
                   OBSERVED[i][4] = x;
                   break;
                 case 9:
                   OBSERVED[i][5] = x;
                   break;
                 default:
                   break;
              }
           }
           else{
               fprintf(stderr,
                "%s.obs is incompatible with required number of chunks\n",
                base_name);
               exit(1);
           }
       }
   }

   fprintf(stdout, "starting estimation of p\n");
   fflush(stdout);

   /* determine optimal specialization proportion */

   lexprop = getlexprop ();

   /* AND CALCULATE E[V(N)] AND E[V(m,N)] ;  NOTE: S IS INFINITE */

   fprintf(stdout, "computing expected spectrum at N\n");
   fflush(stdout);
   eV = expV(Nzero);
   S = NULL_F;
   for (i = 1; i <= maxrank; i++) {
     fprintf(stdout, "[%d]\n", i);
     fflush(stdout);
     SPECTRUM[i][2] = expVm((double) i, Nzero);
     /*     if (i <= 5) VMZERO[i] = SPECTRUM[i][2]; */
   }
   fprintf(stdout, "\n");
   fflush(stdout);

   /* PRINT SUMMARY */

   fprintf(fpsum, "Partition-adjusted Extended Zipf's Law for %s\n", *argv);
   fprintf(fpsum, "N         = %12d\n", (int) N);
   fprintf(fpsum, "V(N)      = %12d\n", (int) V);
   fprintf(fpsum, "E[V(N)]   = %12.4f\n", eV);
   fprintf(fpsum, "V(1,N)    = %12d\n", (int) n1);
   fprintf(fpsum, "E[V(1,N)] = %12.4f\n", SPECTRUM[1][2]);
   fprintf(fpsum, "S         =     infinite\n");
   fprintf(fpsum, "Z         = %12.5f\n",  Z);
   fprintf(fpsum, "beta      = 1.0\n");
   fprintf(fpsum, "VZ        = %12.5f\n",  VZ);
   fprintf(fpsum, "p         = %12.8f\n", lexprop);
   fprintf(fpsum, "MSE       = %12.8f\n", mse);
   fclose(fpsum);

   /* PRINT SPECTRUM */

   fprintf(fpexpspect, "         m         Vm        EVm     alphaM    EalphaM\n");
   for (i = 1; i <= maxrank; i++) {
    fprintf(fpexpspect, "%10d %10d ",(int) SPECTRUM[i][0],(int) SPECTRUM[i][1]);
    fprintf(fpexpspect, "%10.4f %10.4f %10.4f\n", SPECTRUM[i][2],
           SPECTRUM[i][1]/Vzero, SPECTRUM[i][2]/eV);
   }
   fclose(fpexpspect);

   /* PRINT SPECTRUM AT 2N */

   fprintf(stdout, "computing expected spectrum at 2N\n");
   fflush(stdout);
   N = 2 * Nzero;
   eV2N = adjustV (expV (N), N, lexprop, Nzero, Vzero);
   fprintf(fpexpspect2N, "         m      EVm2N\n");
   for (i = 1; i <= 2 * maxrank; i++){
     fprintf(stdout, "[%d]\n", i);
     fflush(stdout);
     fprintf(fpexpspect2N, "%10d %10.2f\n", i, 
            adjustVm (expVm ((double) i, N), N, lexprop, Nzero, SPECTRUM[i][2])
            );
   }
   fprintf(stdout, "\n");
   fflush(stdout);
   fclose(fpexpspect2N);

   /* PRINT VOCABULARY SIZES */

   fprintf(fpVN, "       V       EV     EV2N\n");
   fprintf(fpVN, "%8.2f %8.2f %8.2f\n", V, eV, adjustV (eV2N, 2 * Nzero, lexprop, Nzero, Vzero));
   fclose(fpVN);

   /* INTERPOLATION */

   if (nchunks > 0){

     /* CALCULATE THE TEXT CHUNKS */

     chunksize = floor(Nzero/(nchunks*1.0));
     remainDer = Nzero - ((nchunks*1.0) * chunksize);
     for (k = 1; k <= nchunks; k++)   CHUNKS[k] = chunksize;
     for (k = 1; k <= remainDer; k++) CHUNKS[k]++;
     for (k = 2; k <= nchunks; k++)   CHUNKS[k] += CHUNKS[k-1];

     /* AND PRINT THE CORRESPONDING STATISTICS */

     fprintf(stdout, "computing interpolation statistics\n");
     fprintf(fpint, "       N       EV   Alpha1      EV1      EV2      EV3      EV4      EV5       GV\n");
     for (k = 1; k <= nchunks; k++){
        fprintf(stdout, "[%d]\n", k);
        fflush(stdout);
        x = adjustV (expV (CHUNKS[k]), CHUNKS[k], lexprop, Nzero, Vzero);
        x1 = adjustV (expV (CHUNKS[k]+1.0), CHUNKS[k], lexprop, Nzero, Vzero);
        y =  adjustVm (expVm (1.0, CHUNKS[k]), CHUNKS[k], lexprop, Nzero, SPECTRUM[1][2]);
        y2 = adjustVm (expVm (2.0, CHUNKS[k]), CHUNKS[k], lexprop, Nzero, SPECTRUM[2][2]);
        y3 = adjustVm (expVm (3.0, CHUNKS[k]), CHUNKS[k], lexprop, Nzero, SPECTRUM[3][2]);
        y4 = adjustVm (expVm (4.0, CHUNKS[k]), CHUNKS[k], lexprop, Nzero, SPECTRUM[4][2]);
        y5 = adjustVm (expVm (5.0, CHUNKS[k]), CHUNKS[k], lexprop, Nzero, SPECTRUM[5][2]);
        fprintf(fpint, 
           "%8.2f %8.2f %8.4f %8.2f %8.2f %8.2f %8.2f %8.2f %8.4f\n", 
           CHUNKS[k], x, y/x, y, y2, y3, y4, y5, x1-x);
     }
     fprintf(stdout, "\n");
     fflush(stdout);
   }

   /* EXTRAPOLATION */
  
   if (E == NULL_F) {  /* extrapolate to 2N */
     fprintf(stdout, "computing extrapolation statistics to 2N\n");
     fflush(stdout);
     fprintf(fpext, "         N         EV        EV1      EV2      EV3      EV4      EV5       GV\n");

     for (k = 1; k <= nchunks; k++){
        fprintf(stdout, "[%d]\n", k);
        fflush(stdout);
        x = adjustV (expV (Nzero+CHUNKS[k]), Nzero+CHUNKS[k], lexprop, Nzero, Vzero);
        x1 = adjustV (expV (Nzero+CHUNKS[k]+1.0), Nzero+CHUNKS[k]+1.0, lexprop, Nzero, Vzero);
        y = adjustVm (expVm (1.0, Nzero+CHUNKS[k]), Nzero+CHUNKS[k], lexprop, Nzero, SPECTRUM[1][2]);
        y2 = adjustVm (expVm (2.0, Nzero+CHUNKS[k]), Nzero+CHUNKS[k], lexprop, Nzero, SPECTRUM[2][2]);
        y3 = adjustVm (expVm (3.0, Nzero+CHUNKS[k]), Nzero+CHUNKS[k], lexprop, Nzero, SPECTRUM[3][2]);
        y4 = adjustVm (expVm (4.0, Nzero+CHUNKS[k]), Nzero+CHUNKS[k], lexprop, Nzero, SPECTRUM[4][2]);
        y5 = adjustVm (expVm (5.0, Nzero+CHUNKS[k]), Nzero+CHUNKS[k], lexprop, Nzero, SPECTRUM[5][2]);
        fprintf(fpext, "%10.2f %10.2f %10.4f ", Nzero+CHUNKS[k],  x, y);
        fprintf(fpext, "%8.2f %8.2f %8.2f %8.2f %8.4f\n", y2, y3, y4, y5,x1-x);
     }
     fprintf(stdout, "\n");
     fflush(stdout);
   }
   else{
     fprintf(stdout, "warning: extrapolation to arbitrary E not implemented\n");
     exit(1);

     /* FIND NEW CHUNKSIZES */

     fprintf(stdout, "computing extrapolation statistics to E\n");
     fflush(stdout);
     chunksize = floor((E-Nzero)/(enchunks*1.0));
     remainDer = (E-Nzero) - ((enchunks*1.0) * chunksize);
     for (k = 1; k <= enchunks; k++)   CHUNKS[k] = chunksize;
     for (k = 1; k <= remainDer; k++)  CHUNKS[k]++;
     for (k = 2; k <= enchunks; k++)   CHUNKS[k] += CHUNKS[k-1];

     /* PRINT THE GROWTH CURVE */

	 fprintf(stdout, "computing extrapolation statistics to 2N\n");
     fflush(stdout);
     fprintf(fpext, "         N         EV        EV1      EV2      EV3      EV4      EV5       GV\n");

     for (k = 1; k <= enchunks; k++){
        fprintf(stdout, "[%d]\n", k);
        fflush(stdout);
        x = adjustV (expV (Nzero+CHUNKS[k]), Nzero+CHUNKS[k], lexprop, Nzero, Vzero);
        x1 = adjustV (expV (Nzero+CHUNKS[k]+1.0), Nzero+CHUNKS[k]+1.0, lexprop, Nzero, Vzero);
        y = adjustVm (expVm (1.0, Nzero+CHUNKS[k]), Nzero+CHUNKS[k], lexprop, Nzero, SPECTRUM[1][2]);
        y2 = adjustVm (expVm (2.0, Nzero+CHUNKS[k]), Nzero+CHUNKS[k], lexprop, Nzero, SPECTRUM[2][2]);
        y3 = adjustVm (expVm (3.0, Nzero+CHUNKS[k]), Nzero+CHUNKS[k], lexprop, Nzero, SPECTRUM[3][2]);
        y4 = adjustVm (expVm (4.0, Nzero+CHUNKS[k]), Nzero+CHUNKS[k], lexprop, Nzero, SPECTRUM[4][2]);
        y5 = adjustVm (expVm (5.0, Nzero+CHUNKS[k]), Nzero+CHUNKS[k], lexprop, Nzero, SPECTRUM[5][2]);
        fprintf(fpext, "%10.2f %10.2f %10.4f ", Nzero+CHUNKS[k],  x, y);
        fprintf(fpext, "%8.2f %8.2f %8.2f %8.2f %8.4f\n", y2, y3, y4, y5,x1-x);
     }
     fprintf(stdout, "\n");
     fflush(stdout);


     /* AND SHOW THE SPECTRUM AT E */

     eV2N = expV(E);
     fprintf(fpE, "         m      EVmXN\n");
     for (i = 1; i <= maxrank; i++){
       fprintf(fpE, "%10d %10.2f\n", i, expVm((double) i, E));
     }
   }

   return (0);
} /* end of main */


double expV (n)
double	n;

{
  N = n;
  return ((Z * N * log (N / Z)) / ((N - Z) * log (pstar * Z)));
}


double expVm (m, n)
double	m, n;  

{
  mrank = m;
  N = n;
  return (qromb (funcVm, L_BOUND, u_bound) * VZ * N / Z); 
}


double funcVm (x)
double	x;

{
 return (x / (exp ((mrank + 1.0) * log (1.0 + x)) * (1.0 + (x * N / Z))));
}


void help ()
{
  fprintf (stderr, "ad2Zipf text.spc\n");
  fprintf (stderr, "OPTIONS:\n");
  fprintf (stderr, "     -h: display help\n");
  fprintf (stderr, "     -m: number of ranks in fit (default: 15)\n");
  fprintf (stderr, "     -k: number of chunks for interpolation (default: 20)\n");
  fprintf (stderr, "     -K: number of chunks for extrapolation (default: 20)\n");
  fprintf (stderr, "     -E: extrapolation sample size (default: 2N)\n");
  fprintf (stderr, "     -H: input files lack header (default: with header)\n");
  fprintf (stderr, "INPUT:\n");
  fprintf (stderr, "     text.spc:  m Vm\n");
  fprintf (stderr, "     text.obs should be available for reading \n");
  fprintf (stderr, "OUTPUT:\n");
  fprintf (stderr, "     text_bZ.spc:      expected spectrum\n");
  fprintf (stderr, "     text_bZ.sp2:      expected spectrum at 2N\n");
  fprintf (stderr, "     text_bZ.ev2:      E[V(N)] and E[V(2N)]\n");
  fprintf (stderr, "     text_bZ.sum:      summary, fitted parameters\n");
  fprintf (stderr, "     text_bZ.int:      interpolation results\n");
  fprintf (stderr, "     text_bZ.ext:      extrapolation results\n");
  exit (1);
}


double getlexprop ()
{
     double expvoc, adjvoc, optp, p, t, diff;
     int i, k;

     /* determine chunksizes */

     chunksize = floor(Nzero/(nchunks*1.0));
     remainDer = Nzero - ((nchunks*1.0) * chunksize);
     for (k = 1; k <= nchunks; k++)   CHUNKS[k] = chunksize;
     for (k = 1; k <= remainDer; k++) CHUNKS[k]++;
     for (k = 2; k <= nchunks; k++)   CHUNKS[k] += CHUNKS[k-1];

     /* use vocabulary growth curve for fit */

     mse = MAXX;
     optp = NULL_F;
     for (p = 0.01; p <= 1.0; p += 0.01){
       x = NULL_F;
       for (i = 1; i <= nchunks; i++){
          N = CHUNKS[i]; t = N/CHUNKS[nchunks];
          expvoc = expV(N);
          adjvoc = (p * t * Vzero) + (1.0-p)*expvoc;
          diff = (adjvoc-OBSERVED[i][0]);
          x += (diff*diff);
       }
       x = x / nchunks;
       if (x < mse) {
            mse = x;
            optp = p;
       }
     }
     return (optp);
}
