(*
 -----------------------------------------------------------------------------------------
 -
 -         SCIENTIFIC COMPUTING WITH MATHEMATICA:
 -                    MATHEMATICAL PROBLEMS FOR ORDINARY DIFFERENTIAL EQUATIONS
 -
 -                     A. Marasco & A. Romano
 -
 -
 -         THE PACKAGE ODE.m
 -
 -             Title: ODE.m
 -
 -             Authors: A. Marasco & A. Romano
 -
 -             Copyright: Copyright 2000-2001 by Birkhauser. All rights reserved.
 -
 -             Mathematica Versions: 3.0
 -
 ---------------------------------------------------------------------------------------
*)


BeginPackage["ODE`"]


(* = = = = = = = = = = = = =  S T A N D A R D  P A C K A G E S  = = = = = = = = = = = = = *)


Needs["Graphics`Arrow`"]
Needs["Graphics`ImplicitPlot`"]
Needs["Graphics`PlotField`"]


(* = = = = = = = = = = = = = = = = = =  U S A G E  = = = = = = = = = = = = = = = = = = = *)


HelpSysn::usage:="Help for Sysn"
UsageSysn::usage:="Use Sysn"
Sysn::usage:="The program Sysn"

HelpPhase2D::usage:="Help for Phase2D"
UsagePhase2D::usage:="Use of Phase2D"
Phase2D::usage:="The program Phase2D"

HelpPolarPhase::usage:="Help for PolarPhase"
UsagePolarPhase::usage:="Use of PolarPhase"
PolarPhase::usage:="The program PolarPhase"

HelpPhase3D::usage:="Help for Phase3D"
UsagePhase3D::usage:="Use of Phase3D"
Phase3D::usage:="The program Phase3D"

HelpLinSys::usage:="Help for LinSys"
UsageLinSys::usage:="Use of LinSys"
LinSys::usage:="The program LinSys"

HelpSerSol::usage:="Help for SerSol"
UsageSerSol::usage:="Use SerSol"
SerSol::usage:="The program SerSol"

HelpFrobenius::usage:="Help for Frobenius"
UsageFrobenius::usage:="Use Frobenius"
Frobenius::usage:="The program Frobenius"

HelpPoincare::usage:="Help for Poincar"
UsagePoincare::usage:="Use Poincar"
Poincare::usage:="The program Poincar"

HelpLiapunov::usage:="Help for Liapunov"
UsageLiapunov::usage:="Use Liapunov"
Liapunov::usage:="The program Liapunov"

HelpLStability::usage:="Help for LStability"
UsageLStability::usage:="Use LStability"
LStability::usage:="The program LStability"

HelpCriticalEqN::usage:="Help for CriticalEqN"
UsageCriticalEqN::usage:="Use CriticalEqN"
CriticalEqN::usage:="The program CriticalEqN"
CriticalEqN1::usage:="Subprogram of CriticalEqN"
CriticalEqN2::usage:="Subprogram of CriticalEqN to
overcome an error of Mathematica"

HelpCriticalEqS::usage:="Help for CriticalEqS"
UsageCriticalEqS::usage:="Use CriticalEqS"
CriticalEqS::usage:="The program CriticalEqS"
CriticalEqS1::usage:="Subprogram of CriticalEqS"
CriticalEqS2::usage:="Subprogram of CriticalEqS to
overcome an error of Mathematica"

HelpCManifold::usage:="Help for CManifold"
UsageCManifold::usage:="Use CManifold"
CManifold::usage:="The program CManifold"
TaylorCM::usage="The program is a subroutine of CManifold"

HelpBif1::usage:="Help for Bif1"
UsageBif1::usage:="Use Bif1"
Bif1::usage:="The program Bif1"

HelpBif1G::usage:="Help for Bif1G"
UsageBif1G::usage:="Use Bif1G"
Bif1G::usage:="The program Bif1G"

HelpBif2::usage:="Help for Bif2"
UsageBif2::usage:="Use Bif2"
Bif2::usage:="The program Bif2"

HelpBif2G::usage:="Help for Bif2G"
UsageBif2G::usage:="Use Bif2G"
Bif2G::usage:="The program Bif2G"

HelpHopfBif::usage:="Help for HopfBif"
UsageHopfBif::usage:="Use HopfBif"
HopfBif::usage:="Analysis of Hopf's bifurcation"

HelpLindPoinc::usage:="Help for LindPoinc"
UsageLindPoinc::usage:="Use LindPoinc"
LindPoinc::usage:="The program LindPoinc"

HelpGLindPoinc::usage:="Help for GLindPoinc"
UsageGLindPoinc::usage:="Use GLindPoinc"
GLindPoinc::usage:="The program GLindPoinc"

HelpNBoundary::usage:="Help for NBoundary"
UsageNBoundary::usage:="Use NBoundary"
NBoundary::usage:="The program NBoundary"

HelpNBoundary1::usage:="Help for NBoundary1"
UsageNBoundary1::usage:="Use NBoundary1"
NBoundary1::usage:="The program NBoundary1"

HelpNBoundary2::usage:="Help for NBoundary2"
UsageNBoundary2::usage:="Use NBoundary2"
NBoundary2::usage:="The program NBoundary2"

HelpNormalForm::usage="Help for NormalForm"
UsageNormalForm::usage="Use NormalForm"
NormalForm::usage="The program NormalForm"

HelpZeros::usage:="Help for Zeros"
UsageZeros::usage:="Use Zeros"
Zeros::usage="The program Zeros"

HelpTaylor::usage:="Help for Taylor"
UsageTaylor::usage:="Use Taylor"
Taylor::usage="The program Taylor"

HelpPoinsot::usage:="Help for Poinsot"
UsagePoinsot::usage:="Use Poinsot"
Poinsot::usage="The program Poinsot"

HelpSolid::usage:="Help for Solid"
UsageSolid::usage:="Use Solid"
Solid::usage="The program Solid"



(* = = = = = = = = = = = = = =  B E G I N  P A C K A G E  = = = = = = = = = = = = = = = = *)


Begin["`Private`"]



(* = = = = = = = = = =  C O N T E N T S  O F  P A C K A G E  O D E.m  = = = = = = = = = = *)


StylePrint["The user must digit HelpProgram[] or UsageProgram[], where Program is one of the following names, to enter the use or the contents of Program, respectively.", "Output",FontFamily->"Times-Bold",FontSize->12,CellDingbat->"\[FilledSquare]"];
StylePrint["Sysn:  Numerical solutions of n differential equations with their plots.","Output",FontFamily->"Times-Bold",CellDingbat->"\[FilledDiamond]"];
StylePrint["Phase2D:  Phase portrait 2D in Cartesian coordinates.", "Output",FontFamily->"Times-Bold",CellDingbat->"\[FilledDiamond]", CellDingbat->"\[FilledDiamond]"];
StylePrint["PolarPhase:  Phase portrait 2D in polar coordinates.","Output",FontFamily->"Times-Bold",CellDingbat->"\[FilledDiamond]"];
StylePrint["Phase3D:  Phase portrait 3D in Cartesian coordinates.","Output",FontFamily->"Times-Bold",CellDingbat->"\[FilledDiamond]"];
StylePrint["LinSys:  General solution of a linear system with constant coefficients.","Output",FontFamily->"Times-Bold",CellDingbat->"\[FilledDiamond]"];
StylePrint["SerSol:  Power series solution of ODEs.","Output",FontFamily->"Times-Bold",CellDingbat->"\[FilledDiamond]"];
StylePrint["Frobenius:  Series expansions at a regular singular point of the solution of a linear second order ODE (Frobenius' method).","Output",FontFamily->"Times-Bold",CellDingbat->"\[FilledDiamond]"];
StylePrint["Poincare:  Series solution of ODEs containing a parameter (Poincar's method).","Output",FontFamily->"Times-Bold",CellDingbat->"\[FilledDiamond]"];
StylePrint["Liapunov:  Equilibrium stability and polynomial Liapunov functions.","Output",FontFamily->"Times-Bold",CellDingbat->"\[FilledDiamond]"];
StylePrint["LStability:  Equilibrium stability and linear methods.","Output",FontFamily->"Times-Bold",CellDingbat->"\[FilledDiamond]"];
StylePrint["CriticalEqN:  Analysis of the equilibrium in the critical case by Poincar's method (numerical coefficients).","Output",FontFamily->"Times-Bold",CellDingbat->"\[FilledDiamond]"];
StylePrint["CriticalEqS:  Analysis of the equilibrium in the critical case by Poincar's method (symbolic coefficients).","Output",FontFamily->"Times-Bold",CellDingbat->"\[FilledDiamond]"];
StylePrint["CManifold:  Analysis of the equilibrium in the critical case and the center manifold.","Output",FontFamily->"Times-Bold",CellDingbat->"\[FilledDiamond]"];
StylePrint["Bif1:  Bifurcation for an equation containing one parameter.","Output",FontFamily->"Times-Bold",CellDingbat->"\[FilledDiamond]"];
StylePrint["Bif1G:  Graphical analysis of bifurcation for an equation containing one parameter.","Output",FontFamily->"Times-Bold",CellDingbat->"\[FilledDiamond]"];
StylePrint["Bif2:  Bifurcation for an equation containing two parameters.","Output",FontFamily->"Times-Bold",CellDingbat->"\[FilledDiamond]"];
StylePrint["Bif2G:  Graphical analysis of bifurcation for an equation containing two parameters.","Output",FontFamily->"Times-Bold",CellDingbat->"\[FilledDiamond]"];
StylePrint["HopfBif:  Hopf's bifurcation.","Output",FontFamily->"Times-Bold",CellDingbat->"\[FilledDiamond]"];
StylePrint["LindPoinc:  Approximate periodic solutions of ODEs containing a parameter (Lindstedt-Poincar's method).","Output",FontFamily->"Times-Bold",CellDingbat->"\[FilledDiamond]"];
StylePrint["GLindPoinc:  Comparison between the numerical and approximate solutions obtained by LindPoinc.","Output",FontFamily->"Times-Bold",CellDingbat->"\[FilledDiamond]"]
StylePrint["NBoundary:  Numerical solution by the shooting method of mixed boundary-value problems for the equation y'' - F(x, y, y') = 0.","Output",FontFamily->"Times-Bold",CellDingbat->"\[FilledDiamond]"];
StylePrint["NBoundary1:  Numerical solution by the finite difference method of mixed boundary-value problems for the linear equation y'' + a(x)y' + b(x)y + f(x) = 0.","Output",FontFamily->"Times-Bold",CellDingbat->"\[FilledDiamond]"];
StylePrint["NBoundary2:  Numerical solution by the finite difference method of mixed boundary-value problems for the nonlinear equation y'' - F(x, y, y') = 0.","Output",FontFamily->"Times-Bold",CellDingbat->"\[FilledDiamond]"];
StylePrint["Poinsot:  Free rotations of a solid with a fixed point.","Output",FontFamily->"Times-Bold",CellDingbat->"\[FilledDiamond]"];
StylePrint["Solid:  General motion of a solid with a fixed point.","Output",FontFamily->"Times-Bold",CellDingbat->"\[FilledDiamond]"];
StylePrint["NormalForm:  Normal form of a planar differential system whose linear part has immaginary eigenvalues.","Output",FontFamily->"Times-Bold",CellDingbat->"\[FilledDiamond]"];
StylePrint["Zeros:  Equilibrium solutions of a nonlinear system of 1, 2 or 3 differential equations.","Output",FontFamily->"Times-Bold",CellDingbat->"\[FilledDiamond]"];
StylePrint["Taylor:  Taylor's expansion of a function depending on one or more variables.","Output",FontFamily->"Times-Bold",CellDingbat->"\[FilledDiamond]"];


(* = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = *)


Off[General::spell];
Off[General::spell1];
Off[Solve::svars];
Off[Plot::plnr];
Off[Part::partd];
Off[Part::partw];
Off[Part::pspec];
Off[FindRoot::regex];
Off[FindRoot::cvnwt];
Off[InterpolatingFunction::dmval];


(* = = = = = = = = = = = = = = =  T H E   P R O G R A M S  = = = = = = = = = = = = = = = *)


(*
 -----------------------------------------------------------------------------------------
 -
 -   T H E   P R O G R A M  Sysn:
 -
 -                 UsageSysn[]
 -                 HelpSysn[]
 -                 SysnSysn[sys_, unk_, var_, data_, T1_, index_,steps_]
 -
 -----------------------------------------------------------------------------------------
*)


UsageSysn[] :=
  Module[{},
  StylePrint["Aims of the program Sysn", "Output", FontFamily -> "Times-Bold", FontSize -> 12];
  StylePrint["The program evaluates and represents the numerical solutions  corresponding to fixed sets of initial data and gives the plots of their components \!\(x\_i\)(t), for i = 1,...,n,", "Output", FontFamily -> "Times-Plain", FontSize -> 10]; Print[DisplayForm[\(\(x\^,\)\_i\)], " = ", DisplayForm[\(f\_i\)], "(t, \!\(x\_1\),..., \!\(x\_n\)),", "   i=1,...,n."];
  StylePrint["Moreover, it also supplies the plots of the trajectories corresponding to any fixed pair (\!\(x\_r\)(t), \!\(x\_s\)(t)) of the solution components.", "Output", FontFamily -> "Times-Plain", FontSize -> 10];
  StylePrint["The raw command is", "Output", FontFamily -> "Times-Plain", FontSize -> 10];
  StylePrint["Sysn[sys, unk, var, data, T1, index, steps],", "Output", FontFamily -> "Times-Plain", FontSize -> 10];
  StylePrint["where sys is the system, unk are the unknowns, var denotes the independent variable, data is the set of initial data, T1 is the right exstremum of the time interval, index is the pair of unknowns to which refers the phase portrait, steps is the steps of numerical integration.", "Output", FontFamily -> "Times-Plain", FontSize -> 10];
  ]


HelpSysn[]:=
Module[{},
StylePrint["How to use the program Sysn", "Output", FontFamily -> "Times-Bold", FontSize -> 12];
StylePrint["To make active the program type the following input data:", "Output", FontFamily -> "Times-Plain", FontSize -> 10];
StylePrint["sys = the system (f.e. {x' == x - y - z(\!\(x\^2\) + \!\(y\^2\)), y' == x + y - y(\!\(z\^2\) + \!\(y\^2\)), z' == x - y};)", "Output", FontFamily -> "Times-Plain", FontSize -> 10];
StylePrint["unk = the unknowns (f.e.{x, y, z};)", "Output", FontFamily -> "Times-Plain", FontSize -> 10];
StylePrint["var = the independent variable (f.e. t;)", "Output", FontFamily -> "Times-Plain", FontSize -> 10];
StylePrint["data = the set of initial data (f.e. {{0, 1, -1}, {0.5, 0, 1}};)", "Output", FontFamily -> "Times-Plain", FontSize -> 10];
StylePrint["T1 = the right extremum of time interval (f.e. 10;)", "Output", FontFamily -> "Times-Plain", FontSize -> 10];
StylePrint["index = the pair of unknowns to which the phase portrait refers (f.e. {1, 3};)", "Output", FontFamily -> "Times-Plain", FontSize -> 10];
StylePrint["steps = the steps of numerical integration (f.e. 1000;)", "Output", FontFamily -> "Times-Plain", FontSize -> 10];
StylePrint["Sysn[sys, unk, var, data, T1, index, steps]", "Output", FontFamily -> "Times-Plain", FontSize -> 10];
]


Sysn[sys_, unk_, var_, data_, T1_, index_,steps_] :=
 Module[{n2,m2,sostn,in,sys2n,sysfi2n,sol2n,plpar,phase,phase1,phase2},
 n2 = Length[sys];
 m2 = Length[data];
 sostn = Flatten[Table[{unk[[i]] -> unk[[i]][var], Derivative[1][unk[[i]]] ->
         Derivative[1][unk[[i]]][var]}, {i, 1, n2}]];
 in[i_] := Flatten[Table[{unk[[j]][data[[i,1]]] == data[[i,j + 1]]}, {j, 1, n2}]];
 sys2n = sys /. sostn;
 sysfi2n[i_] := Join[sys2n, in[i]];
 sol2n[i_] :=NDSolve[sysfi2n[i],unk,{var,data[[i,1]],T1},MaxSteps -> steps];
 plpar[j_] := Table[Plot[Evaluate[unk[[j]][var] /. sol2n[i]], {var, data[[i,1]], T1},
       DisplayFunction -> Identity], {i, 1, m2}];
 StylePrint["The components of solution versus time are","Output", FontFamily->"Times-Plain",FontSize->12];
 Table[Show[plpar[j],AxesLabel -> {StyleForm[var, FontSlant -> "Italic"],
 StyleForm[unk[[j]],FontSlant -> "Italic"]}, DisplayFunction -> $DisplayFunction], {j, 1, n2}];
 If[n2 == 1, Goto[2], Goto[1]];
 Label[1];
 phase = Table[ParametricPlot[Evaluate[{unk[[index[[1]]]][var], unk[[index[[2]]]][var]} /. sol2n[i]],
       {var, data[[i,1]], T1}, AxesLabel -> {unk[[index[[1]]]], unk[[index[[2]]]]},
       AspectRatio -> Automatic, DisplayFunction -> Identity, PlotRange -> All], {i, 1, m2}];
 StylePrint["The orbits corresponding to the data are","Output", FontFamily->"Times-Plain",FontSize->12];

 Show[phase, AxesLabel -> {StyleForm[unk[[index[[1]]]], FontSlant -> "Italic"],
       StyleForm[unk[[index[[2]]]], FontSlant -> "Italic"]}, DisplayFunction -> $DisplayFunction];
 phase1 = Table[ParametricPlot3D[Evaluate[{unk[[index[[1]]]][var], unk[[index[[2]]]][var], var}/. sol2n[i]],
           {var, data[[i,1]], T1}, BoxRatios -> {1, 1, 1}, PlotPoints -> 300,
       DisplayFunction -> Identity, PlotRange -> All], {i, 1, m2}];
 phase2 = Table[ParametricPlot3D[Evaluate[{unk[[index[[1]]]][var], unk[[index[[2]]]][var], 0} /.
         sol2n[i]], {var, data[[i,1]], T1}, BoxRatios -> {1, 1, 1}, DisplayFunction -> Identity,
       PlotRange -> All, PlotPoints -> 50], {i, 1, m2}];
 StylePrint["The trajectories corresponding to the data are","Output", FontFamily->"Times-Plain",FontSize->12];

 Show[phase1, phase2, AxesLabel -> {StyleForm[unk[[index[[1]]]], FontSlant -> "Italic"],
       StyleForm[unk[[index[[2]]]], FontSlant -> "Italic"], StyleForm[var, FontSlant -> "Italic"]},
     DisplayFunction -> $DisplayFunction];
 Label[2];
 ]


(*
 ------------------------------------------------------------------------------------------
 -
 -   T H E   P R O G R A M  Phase2D:
 -
 -                 UsagePhase2D[]
 -                 HelpPhase2D[]
 -                 Phase2D[sys_,unk_,unk0_,h_,n_,{T1_,T2_},steps_,unkvar_,graph_]
 -
 ------------------------------------------------------------------------------------------
*)


HelpPhase2D[]:=
Module[{},
StylePrint["How to use the program Phase2D ","Output",FontFamily->"Times-Bold",FontSize->12];
StylePrint["To make active the program type in the following input data","Output",FontFamily->"Times-Plain",FontSize->10];
StylePrint["eq1 = first equation (f.e. x' == y;)","Output",FontFamily->"Times-Plain",FontSize->10];
StylePrint["eq2 = second equation (f.e. y' == - x;)","Output",FontFamily->"Times-Plain",FontSize->10];
StylePrint["sys = {eq1, eq2};","Output",FontFamily->"Times-Plain",FontSize->10];
StylePrint["unk = list of unknowns (f.e. {x, y};)","Output",FontFamily->"Times-Plain",FontSize->10];
StylePrint["unk0 = coordinates of central point of phase portrait (f.e. {0, 0};)","Output",FontFamily->"Times-Plain",FontSize->10];
StylePrint["h = step lengths along the axes (f.e. {0.2, 0.3};)","Output",FontFamily->"Times-Plain",FontSize->10];
StylePrint["n = points along the axes (f.e. {3, 2};)","Output",FontFamily->"Times-Plain",FontSize->10];
StylePrint["{T1, T2} = time interval of numerical integration (f.e. {0, 10};)","Output",FontFamily->"Times-Plain",FontSize->10];
StylePrint["steps = step number of numerical integration (f.e. 2000;)","Output",FontFamily->"Times-Plain",FontSize->10];
StylePrint["unkvar = plot view window (f.e. {{-1, 1}, {-1, 2}};)","Output",FontFamily->"Times-Plain",FontSize->10];
StylePrint["graph = option on the plot (f.e. 0;)","Output",FontFamily->"Times-Plain",FontSize->10];
StylePrint["Phase2D[sys, unk, unk0, h, n, {T1, T2}, steps, unkvar, graph]","Output",FontFamily->"Times-Plain",FontSize->10];
StylePrint["The option graph can assume the values 0, 1, 2, 3.","Output",FontFamily->"Times-Plain",FontSize->10];
StylePrint["graph = 0, gives the simple phase portrait (suggested choice);", "Output",FontFamily->"Times-Plain",FontSize->10];
StylePrint["graph = 1, gives the phase portrait with the arrows on the orbits directed along the increasing time values;","Output",FontFamily->"Times-Plain",FontSize->10];
StylePrint["graph = 2, gives the phase portrait and the vector field plot in two distinct figures;","Output",FontFamily->"Times-Plain",FontSize->10];
StylePrint["graph = 3, gives the phase portrait and the vector field plot in a single figure.","Output",FontFamily->"Times-Plain",FontSize->10];
]

UsagePhase2D[]:=
Module[{},
StylePrint["Aims of the program Phase2D ","Output",FontFamily->"Times-Bold",FontSize->12];
StylePrint["The program draws the phase portrait of the system (sys)","Output",FontFamily->"Times-Plain",FontSize->10];
Print[TableForm[{"x' = X(x, y),", "y' = Y(x, y),"},TableDepth->2]];
StylePrint[" in the unknowns unk = {x(t), y(t)} in an arbitrary neighbourhood of the point unk0 = {x0, y0}. The chosen orbits start from the points (x0 + i*h1, y0 +j*h2), where i and j are integer belonging to (-n1, n1) and (-n2, n2) respectively. Finally, steps denotes the number of steps of numerical integration and unkvar = {{a, b}, {c, d}} is the plot range window. The command line writes","Output",FontFamily->"Times-Plain",FontSize->10];
StylePrint["Phase2D[sys, unk, unk0, h, n, {T1, T2}, steps, unkvar, graph],","Output",FontFamily->"Times-Plain",FontSize->10];
StylePrint["where h = {h1, h2}, n = {n1, n2}, {T1, T2} is the time interval and graph is 0, 1, 2 or 3.","Output",FontFamily->"Times-Plain",FontSize->10];
]


Phase2D[sys_,unk_,unk0_,h_,n_,{T1_,T2_},steps_,unkvar_,graph_]:=
Module[{a,b,i,j,sost,systi,s,z,z1,z2,t},
If[Head[graph]===Integer&&graph==0||Head[graph]===Integer&&graph==1||
   Head[graph]===Integer&&graph==2||Head[graph]===Integer&&graph==3,Goto[0],Goto[6]];
Label[0];

(*Initial Data*)
a=Table[unk0[[1]]+i h[[1]],{i,-n[[1]],n[[1]]}];
b=Table[unk0[[2]]+j h[[2]],{j,-n[[2]],n[[2]]}];

(*Solutions corresponding to the previous initial data*)
sost=Table[{unk[[i]]->unk[[i]][t],unk[[i]]'->unk[[i]]'[t]},{i,1,2}]//Flatten;
systi=sys/.sost;
s[i_,j_]:=NDSolve[{systi[[1]],systi[[2]],unk[[1]][0]==a[[i]],
unk[[2]][0]==b[[j]]},unk,{t,T1,T2},MaxSteps->steps]//Flatten;
f[i_,j_,t_]:={unk[[1]][t],unk[[2]][t]}/.s[i,j];

(*Orbits*)
orbits[i_,j_]:=ParametricPlot[Evaluate[f[i,j,t]], {t,T1,T2},AxesOrigin->{0,0},Compiled->False,
AxesLabel->{StyleForm[unk[[1]],FontSlant->"Italic"],StyleForm[unk[[2]],
FontSlant->"Italic"]},PlotRange->unkvar,DisplayFunction->Identity];
z=Table[orbits[i,j],{i,1,2 n[[1]]+1},{j,1,2 n[[2]]+1}];

(*Arrows*)
Which[graph==1,Goto[1],graph==2,Goto[2],graph==3,Goto[3],graph==0,Goto[4]];
Label[1];
arr[i_,j_]:= Graphics[Arrow[f[i,j,(T2+T1)/2],f[i,j,((T2+T1)/2)+0.1],HeadCenter->0.8,
HeadWidth->0.2]];
z1=Table[arr[i,j],{i,1,2 n[[1]]+1},{j,1,2 n[[2]]+1}];
Show[z,z1,DisplayFunction->$DisplayFunction, AspectRatio->Automatic]; Goto[5]; Label[2];
Show[z,DisplayFunction->$DisplayFunction, AspectRatio->Automatic];
z2=PlotVectorField[{sys[[1,2]],sys[[2,2]]},{unk[[1]],unkvar[[1,1]],
unkvar[[1,2]]},{unk[[2]],unkvar[[2,1]],unkvar[[2,2]]}];
Goto[5];
Label[3];
z2=PlotVectorField[{sys[[1,2]],sys[[2,2]]},{unk[[1]],unkvar[[1,1]],
unkvar[[1,2]]},{unk[[2]],unkvar[[2,1]],unkvar[[2,2]]}, DisplayFunction->Identity];
Show[z,z2,DisplayFunction->$DisplayFunction,AspectRatio->Automatic]; Goto[5]; Label[4];
Show[z,DisplayFunction->$DisplayFunction, AspectRatio->Automatic];
Goto[5];
Label[6];

StylePrint["Wrong value of graph","Output",FontFamily->"Times-Plain",FontSize->12];
Label[5];
]


(*
 ---------------------------------------------------------------------------------------------
 -
 -   T H E   P R O G R A M  PolarPhase:
 -
 -                 UsagePolarPhase[]
 -                 HelpPolarPhase[]
 -                 PolarPhase[sys_, unk_, unk0_, h_, n_, {T1_, T2_}, steps_, unkvar_, graph_]
 -
 ---------------------------------------------------------------------------------------------
*)


UsagePolarPhase[]:=
Module[{},
StylePrint["Aims of the program PolarPhase ","Output",FontFamily->"Times-Bold",FontSize->12];
StylePrint["The program draws the phase portrait of the system (sys)","Output",FontFamily->"Times-Plain",FontSize->10];
Print[ TableForm[{"r' = X(r, \[CurlyPhi]),","\[CurlyPhi]' = Y(r, \[CurlyPhi]),"},TableDepth->2]];
StylePrint["in the unknowns unk = {r(t), \[CurlyPhi](t)} in an arbitrary neighbourhood of the point unk0 = {r0, \[CurlyPhi]0). The drawn orbits start from the points (r0 + i*h1,  \[CurlyPhi]0 + j*h2) where i is an integer belonging to (- n1, n1) and j is an other integer belonging to (- n2, n2). Finally, steps denotes the number of steps of numerical integration. The command line writes","Output",FontFamily->"Times-Plain",FontSize->10];
StylePrint["PolarPhase[sys, unk, unk0, h, n, {T1, T2}, steps, unkvar, graph],","Output",FontFamily->"Times-Plain",FontSize->10];
StylePrint["where h = {h1, h2}, n = {n1, n2}, {T1, T2} is the time interval and graph is 0, 1, 2 or 3.","Output",FontFamily->"Times-Plain",FontSize->10];
]

HelpPolarPhase[]:= Module[{},
StylePrint["How to use the program PolarPhase ","Output",FontFamily->"Times-Bold",FontSize->12];
StylePrint["To make active the program type in the following input data","Output",FontFamily->"Times-Plain",FontSize->10];
StylePrint["eq1 = first equation (f.e. r'== \[CurlyPhi];)","Output",FontFamily->"Times-Plain",FontSize->10];
StylePrint["eq2 = second equation (f.e. \[CurlyPhi]'== - r;)","Output",FontFamily->"Times-Plain",FontSize->10];
StylePrint["sys = {eq1,eq2};","Output",FontFamily->"Times-Plain",FontSize->10];
StylePrint["unk = list of unknowns (f.e. {r, \[CurlyPhi]};)","Output",FontFamily->"Times-Plain",FontSize->10];
StylePrint["unk0 = coordinates of the central point of phase portrait (f.e. {0,0};)","Output",FontFamily->"Times-Plain",FontSize->10];
StylePrint["h = step lengths along the polar axis and the angle (f.e. {0.2,0.3};)","Output",FontFamily->"Times-Plain",FontSize->10];
StylePrint["n = points along the polar axis and the angle (f.e. {3,2};)","Output",FontFamily->"Times-Plain",FontSize->10];
StylePrint["{T1,T2} = time interval of numerical integration (f.e. {0,10};)","Output",FontFamily->"Times-Plain",FontSize->10];
StylePrint["steps = step number of numerical integration (f.e. 2000;)","Output",FontFamily->"Times-Plain",FontSize->10];
StylePrint["unkvar = plot view window (f.e. {{-1, 1},{-1, 2}};)","Output",FontFamily->"Times-Plain",FontSize->10];
StylePrint["graph = option on the plot (f.e.0;)","Output",FontFamily->"Times-Plain",FontSize->10];
StylePrint["PolarPhase[sys, unk, unk0, h, n, {T1, T2}, steps, unkvar, graph]","Output",FontFamily->"Times-Plain",FontSize->10];
StylePrint["The option graph can assume the values 0, 1, 2, 3.","Output",FontFamily->"Times-Plain",FontSize->10];
StylePrint["graph = 0, gives the simple phase portrait (adviced choice);","Output",FontFamily->"Times-Plain",FontSize->10];
StylePrint["graph = 1, gives the phase portrait with the arrows on the orbits directed along the increasing time values;","Output",FontFamily->"Times-Plain",FontSize->10];
StylePrint["graph = 2, gives the phase portrait and the vector field plot in two distinct figures;","Output",FontFamily->"Times-Plain",FontSize->10];
StylePrint["graph = 3, gives the phase portrait and the vector field plot in a single figure.","Output",FontFamily->"Times-Plain",FontSize->10];
]


PolarPhase[sys_, unk_, unk0_, h_, n_, {T1_, T2_}, steps_, unkvar_, graph_] :=
Module[{(*a2, b2, i, j, sostp, systip, sp, cunk, zp, z1p, z2p, t*)},

If[Head[graph]===Integer&&graph==0||Head[graph]===Integer&&graph==1||
  Head[graph]===Integer&&graph==2||Head[graph]===Integer&&graph==3,Goto[0],Goto[6]];

Label[0];

a2 = Table[unk0[[1]] + i*h[[1]], {i, 0, n[[1]]}];
b2 = Table[unk0[[2]] + j*h[[2]], {j, 0, n[[2]]}];
sostp = Flatten[Table[{unk[[i]] -> unk[[i]][t],Derivative[1][unk[[i]]] ->
Derivative[1][unk[[i]]][t]}, {i, 1, 2}]];
systip = sys /. sostp;
sp[i_, j_] := NDSolve[{systip[[1]], systip[[2]], unk[[1]][0] == a2[[i]],
unk[[2]][0] == b2[[j]]}, unk, {t, T1, T2}, MaxSteps -> steps];
cunk1p[i_, j_] := (unk[[1]][t] /. sp[i, j][[1,1]])*Cos[unk[[2]][t] /. sp[i, j][[1,2]]];
cunk2p[i_, j_] := (unk[[1]][t] /. sp[i, j][[1,1]])*Sin[unk[[2]][t] /. sp[i, j][[1,2]]];
orbitsp[i_, j_] := ParametricPlot[Evaluate[{cunk1p[i, j], cunk2p[i, j]}],
      {t, T1, T2}, AxesLabel -> {StyleForm[Global`x, FontSlant -> "Italic"],
        StyleForm[Global`y, FontSlant -> "Italic"]}, PlotRange -> unkvar, DisplayFunction -> Identity];
zp = Table[orbitsp[i, j], {i, 1, n[[1]] + 1}, {j, 1, n[[2]] + 1}];
Which[graph == 1, Goto[1], graph == 2, Goto[2], graph == 3, Goto[3], graph == 0, Goto[4]];
Label[1];
arrp[i_, j_] := Graphics[Arrow[{cunk1p[i, j] /. t -> (T2 + T1)/2,
        cunk2p[i, j] /. t -> (T2 + T1)/2}, {cunk1p[i, j] /. t -> (T2 + T1)/2 + 0.1,
        cunk2p[i, j] /. t -> (T2 + T1)/2 + 0.1}, HeadCenter -> 0.8, HeadWidth -> 0.2]];
z1p = Table[arrp[i, j], {i, 1, n[[1]] + 1}, {j, 1, n[[2]] + 1}];
Show[zp, z1p, DisplayFunction -> $DisplayFunction, AspectRatio -> Automatic]; Goto[5]; Label[2];
derr = (cx*Derivative[1][cx] + cy*Derivative[1][cy])/Sqrt[cx^2 + cy^2];
derphi = (Derivative[1][cy]*cx - cy*Derivative[1][cx])/(cx^2 + cy^2);
sostcart = {unk[[1]] -> Sqrt[cx^2 + cy^2], unk[[2]] -> ArcTan[cy/cx],
      Derivative[1][unk[[1]]] -> derr, Derivative[1][unk[[2]]] -> derphi};
syscart = sys /. sostcart;
syscartfi = Simplify[Flatten[Solve[syscart, {Derivative[1][cx], Derivative[1][cy]}]]];
cartfield = {syscartfi[[1,2]], syscartfi[[2,2]]};
Show[zp, DisplayFunction -> $DisplayFunction, AspectRatio -> Automatic];
z2p = PlotVectorField[cartfield, {cx, unkvar[[1,1]], unkvar[[1,2]]},
      {cy, unkvar[[2,1]], unkvar[[2,2]]}];
Goto[5];
Label[3];
derr =(cx*Derivative[1][cx] + cy*Derivative[1][cy])/Sqrt[cx^2 + cy^2];
derphi = (Derivative[1][cy]*cx - cy*Derivative[1][cx])/(cx^2 + cy^2);
sostcart = {unk[[1]] -> Sqrt[cx^2 + cy^2], unk[[2]] -> ArcTan[cy/cx],
      Derivative[1][unk[[1]]] -> derr, Derivative[1][unk[[2]]] -> derphi};
syscart = sys /. sostcart;
syscartfi = Simplify[Flatten[Solve[syscart, {Derivative[1][cx], Derivative[1][cy]}]]];
cartfield = {syscartfi[[1,2]], syscartfi[[2,2]]};
z2p = PlotVectorField[cartfield, {cx, unkvar[[1,1]], unkvar[[1,2]]},
      {cy, unkvar[[2,1]], unkvar[[2,2]]}, DisplayFunction -> Identity];
Show[zp, z2p, DisplayFunction -> $DisplayFunction, AspectRatio -> Automatic];
Goto[5];
Label[4];
Show[zp, DisplayFunction -> $DisplayFunction, AspectRatio -> Automatic];
Goto[5];
Label[6];
StylePrint["Wrong value of graph","Output",FontFamily->"Times-Plain",FontSize->12];
Label[5];
]


(*
 --------------------------------------------------------------------------------------------
 -
 -   T H E   P R O G R A M  Phase3D:
 -
 -                 UsagePhase3D[]
 -                 HelpPhase3D[]
 -                 Phase3D[sys3_,unk3_,unk03_,h3_,n3_,{T31_,T32_},{k1_,k2_},steps3_,unkvar3_]
 -
 --------------------------------------------------------------------------------------------
*)


UsagePhase3D[]:=
Module[{},
StylePrint["Aims of the program Phase3D ","Output",FontFamily->"Times-Bold",FontSize->12];
StylePrint["The program supplies the phase portrait in Cartesian coordinates of the system (sys)","Output",FontFamily->"Times-Plain",FontSize->10];
Print[TableForm[{"x' = X(x, y, z),","y' = Y(x, y, z),", "z' = Z(x, y, z),"}, TableDepth->2]];
StylePrint["in the unknowns unk = {x(t), y(t), z(t)} in an arbitrary neighbourood of the point unk0 = {x0, y0, z0}. The drawn orbits start from the points {x0 + i*h1, y0 + j*h2, z0 + r*h3}, where i is an integer belonging to (-n1, n1), j is an other integer belonging to (-n2, n2) and r an integer belonging to (-n3, n3). {T1, T2} is the integration time interval and {\[Tau]1, \[Tau]2} the time interval to which the graphic representation refers. Finally, the other symbols have the same meanings of Phase2D. The command line writes", "Output",FontFamily->"Times-Plain",FontSize->10];
StylePrint["Phase3D[sys, unk, unk0, h, n, {T1, T2}, {\[Tau]1, \[Tau]2}, steps, unkvar],","Output",FontFamily->"Times-Plain",FontSize->10];
StylePrint["where h = {h1, h2, h3}, n = {n1, n2, n3}, {T1, T2} is the time interval of the numerical integration, {\[Tau]1, \[Tau]2} the time interval of representation.","Output",FontFamily->"Times-Plain",FontSize->10];
]

HelpPhase3D[]:= Module[{},
StylePrint["How to use the program Phase3D","Output",FontFamily->"Times-Bold",FontSize->12];
StylePrint["To make active the program type in the following input data","Output",FontFamily->"Times-Plain",FontSize->10];
StylePrint["eq1 = first equation (f.e. x' == y - z;)","Output",FontFamily->"Times-Plain",FontSize->10];
StylePrint["eq2 = second equation (f.e. y' == - x;)","Output",FontFamily->"Times-Plain",FontSize->10];
StylePrint["eq3 = third equation (f.e. z' == x + z;)","Output",FontFamily->"Times-Plain",FontSize->10];
StylePrint["sys = {eq1, eq2, eq3};","Output",FontFamily->"Times-Plain",FontSize->10];
StylePrint["unk = list of unknowns (f.e. {x, y, z};)","Output",FontFamily->"Times-Plain",FontSize->10];
StylePrint["unk0 = coordinates of central point of phase portrait (f.e. {0, 0, 0};)","Output",FontFamily->"Times-Plain",FontSize->10];
StylePrint["h = step lengths along the axes (f.e. {0.2, 0.3, 0.1};)","Output",FontFamily->"Times-Plain",FontSize->10];
StylePrint["n = points along the axes (f.e.{3, 2, 4};)","Output",FontFamily->"Times-Plain",FontSize->10];
StylePrint["{T1, T2} = time interval of numerical integration (f.e. {0, 10};)","Output",FontFamily->"Times-Plain",FontSize->10];
StylePrint["{\[Tau]1, \[Tau]2} = time interval of representation (f.e. {0, 8};)","Output",FontFamily->"Times-Plain",FontSize->10];
StylePrint["steps = step number of numerical integration (f.e. 2000;)","Output",FontFamily->"Times-Plain",FontSize->10];
StylePrint["unkvar = view window (f.e. {{- 1, 1}, {- 1, 2}, {- 1, 1}};)","Output",FontFamily->"Times-Plain",FontSize->10];
StylePrint["Phase3D[sys, unk, unk0, h, n, {T1, T2}, {\[Tau]1, \[Tau]2}, steps, unkvar]","Output",FontFamily->"Times-Plain",FontSize->10];
]


Phase3D[sys3_,unk3_,unk03_,h3_,n3_,{T31_,T32_},{k1_,k2_},steps3_,unkvar3_]:=
Module[{i3,a3,b3,c3,sost3,systi3,s3,u3,orbits3,t},

(*Initial Data*)
a3=Table[unk03[[1]]+i3*h3[[1]],{i3,-n3[[1]],n3[[2]]}];
b3=Table[unk03[[2]]+i3*h3[[2]],{i3,-n3[[2]],n3[[2]]}];
c3=Table[unk03[[3]]+i3*h3[[3]],{i3,-n3[[3]],n3[[3]]}];

sost3=Table[{unk3[[i3]]->unk3[[i3]][t],unk3[[i3]]'->unk3[[i3]]'[t]},{i3,1,3}]//Flatten;
systi3=sys3/.sost3;

(*Solutions corresponding to the previous initial data*)
s3[i3_,j_,r_]:=NDSolve[{systi3[[1]],systi3[[2]],systi3[[3]],unk3[[1]][0]==a3[[i3]],
unk3[[2]][0]==b3[[j]],unk3[[3]][0]==c3[[r]]},unk3,{t,T31,T32},MaxSteps->steps3];

(*Orbits*)
orbits3[i3_,j_,r_]:=ParametricPlot3D[Evaluate[{unk3[[1]][t],unk3[[2]][t],unk3[[3]][t]}/.s3[i3,j,r]],
{t,k1,k2},Compiled->False,AxesLabel->{StyleForm[unk3[[1]],FontSlant->"Italic"],StyleForm[unk3[[2]],FontSlant->"Italic"],
StyleForm[unk3[[3]],FontSlant->"Italic"]},PlotPoints->400,PlotRange->unkvar3,DisplayFunction->Identity];
u3=Table[orbits3[i3,j,r],{i3,1,2 n3[[1]]+1},{j,1,2 n3[[2]]+1},{r,1,2*n3[[3]]+1}];
Show[u3,DisplayFunction->$DisplayFunction,AspectRatio->Automatic];
ClearAll[]
]



(*
 --------------------------------------------------------------------------------------------
 -
 -   T H E   P R O G R A M  LinSys:
 -
 -                 UsageLinSys[]
 -                 HelpLinSys[]
 -                 LinSys[A_,b_,unk_,unk0_]
 -
 --------------------------------------------------------------------------------------------
*)


UsageLinSys[]:=
Module[{},
StylePrint["Aims of the program LinSys","Output",FontFamily->"Times-Bold",FontSize->12];
StylePrint["The program supplies the general solution of a linear nonhomogeneous system with constant coefficients","Output",FontFamily->"Times-Plain",FontSize->10];
Print[DisplayForm[SubscriptBox[SuperscriptBox["x", ","], "i"]]," = ", DisplayForm[UnderoverscriptBox["\[Sum]","j=1","n"]], DisplayForm[SubscriptBox["a", "ij"]],DisplayForm[SubscriptBox["x", "j"]]," + ",DisplayForm[SubscriptBox["b", "i"]],"(","t)"];
StylePrint["The command line writes","Output",FontFamily->"Times-Plain",FontSize->10];
StylePrint["LinSys[A, b, unk, unk0],","Output",FontFamily->"Times-Plain",FontSize->10];
StylePrint["where","Output",FontFamily->"Times-Plain",FontSize->10];
Print["A = ",MatrixForm[{{"\!\(a\_11\)", "...", "\!\(a\_\(1  n\)\)"}, {" ", "...", " "}, {"\!\(a\_n1\)", "...", "\!\(a\_nn\)"}}],"     ""b = ",MatrixForm[{{"\!\(b\_1\)(t)"}, {"..."}, {"\!\(b\_n\)(t)"}}]];
StylePrint["are respectively the matrix of coefficients and known terms, unk is the list of unknowns and unk0 the list of initial data.","Output",FontFamily->"Times-Plain",FontSize->10];
]

HelpLinSys[]:= Module[{},
StylePrint["How to use the program LinSys","Output",FontFamily->"Times-Bold",FontSize->12];
StylePrint["To launch the program type in the following input data","Output",FontFamily->"Times-Plain",FontSize->10];
StylePrint["A = coefficient matrix (f.e. {{0, 1}, {-1, -1}};)","Output",FontFamily->"Times-Plain",FontSize->10];
StylePrint["b = list of know terms (f.e.{0, 0};)","Output",FontFamily->"Times-Plain",FontSize->10];
StylePrint["unk = list of unknowns (f.e. {x1, x2};)","Output",FontFamily->"Times-Plain",FontSize->10];
StylePrint["unk0 = list of general initial data (f.e. {x01, x02};)","Output",FontFamily->"Times-Plain",FontSize->10];
StylePrint["LinSys[A, b, unk, unk0]","Output",FontFamily->"Times-Plain",FontSize->10];
]


LinSys[A_,b_,unk_,unk0_]:=
Module[{i,e,n,inc,equaz,inda,sys,soluz,u,t},
n=Length[A];

(*Unknowns, equations and initial data*)
inc=Table[unk[[i]][t],{i,1,n}];
e[i_]:=unk[[i]]'[t]==Sum[A[[i,j]]*unk[[j]][t],{j,1,n}]+b[[i]];
equaz=Table[e[i],{i,1,n}];
inda=Table[unk[[i]][0]==unk0[[i]],{i,1,n}];
sys=Join[equaz,inda];

(*Analitic solution*)
soluz=DSolve[sys,inc,t];
u[i_]:=(soluz[[1,i,2]]//ComplexExpand//Simplify)/.t->Global`t;
StylePrint["The analytical solution is","Output",FontFamily->"Times-Plain",FontSize->12];
Table[Print[unk[[i]]," = ",u[i]],{i,1,n}];
]



(*
 --------------------------------------------------------------------------------------------
 -
 -   T H E   P R O G R A M  SerSol:
 -
 -                 UsageSerSol[]
 -                 HelpSerSol[]
 -                 SerSol[sys_,unk_,var_,unk0_,r_,T1_,T2_,steps_]
 -
 --------------------------------------------------------------------------------------------
*)


UsageSerSol[]:=
Module[{},
StylePrint["Aims of the program SerSol ","Output",FontFamily->"Times-Bold",FontSize->12];
StylePrint["The program supplies the solution of the Cauchy problem","Output",FontFamily->"Times-Plain",FontSize->10];
Print[DisplayForm[SubscriptBox[SuperscriptBox["x", ","], "i"]]," = ",DisplayForm[SubscriptBox["F", "i"]],"(",DisplayForm[SubscriptBox["x", "1"]],",..., ",DisplayForm[SubscriptBox["x", "n"]],", t), "];
Print[DisplayForm[SubscriptBox["x", "i"]],"(T1) = ",DisplayForm[SubscriptBox[SuperscriptBox["x", "0"], "i"]],",      i = 1, ..., n"];
StylePrint["as a power series  up to the order r of the independent variable t.  The obtained solution is graphically compared with the numerical one in the interval [T1, T2]. The command line writes","Output",FontFamily->"Times-Plain",FontSize->10];
StylePrint["SerSol[sys, unk, unk0, var, r, T1, T2, steps].","Output",FontFamily->"Times-Plain",FontSize->10];
StylePrint["Here sys is the system and steps denotes the number of  steps in the numerical integration","Output",FontFamily->"Times-Plain",FontSize->10];
]

HelpSerSol[]:=
Module[{},
StylePrint["How to use the program SerSol","Output",FontFamily->"Times-Bold",FontSize->12];
StylePrint["To make active the program type in the following input data","Output",FontFamily->"Times-Plain",FontSize->10];
StylePrint["sys = list of the equations (f.e.{x' == y, y' == - x};)","Output",FontFamily->"Times-Plain",FontSize->10];
StylePrint["unk  = list of unknowns (f.e. {x, y};)","Output",FontFamily->"Times-Plain",FontSize->10];
StylePrint["var  = independent variable (f.e. t;)","Output",FontFamily->"Times-Plain",FontSize->10];
StylePrint["unk0 = list of the initial data (f.e. {0, 0};)","Output",FontFamily->"Times-Plain",FontSize->10];
StylePrint["r = order of the expansion (f.e. 5;)","Output",FontFamily->"Times-Plain",FontSize->10];
StylePrint["T1 = left bound of time interval (T1, T2) (f.e. 1;)","Output",FontFamily->"Times-Plain",FontSize->10];
StylePrint["T2 = right bound of time interval (T1, T2) (f.e. 2;)","Output",FontFamily->"Times-Plain",FontSize->10];
StylePrint["steps = number of steps in the numerical integration (f.e. 1000;)","Output",FontFamily->"Times-Plain",FontSize->10];
StylePrint["SerSol[sys, unk, var, unk0, r, T1, T2, steps]","Output",FontFamily->"Times-Plain",FontSize->10];
]


SerSol[sys_,unk_,var_,unk0_,r_,T1_,T2_,steps_]:=
Module[{xse,n,a,e,c,sys1,m,coeff,sol,h,ss},

(*The solution is represented as a power series in the variable t verifying the initial conditions*)
n=Length[sys];
xse[h_]:=SeriesData[var,T1,Table[a[h,i],{i,0,r}]]/.a[h,0]->unk0[[h]];

(*Introduction of the series into the system*)
sost=Table[unk[[h]]->xse[h],{h,1,n}];
e=Table[Normal[D[xse[h],var]-sys[[h,2]]/.sost],{h,1,n}];
c[h_,j_]:=Coefficient[e[[h]],var-T1,j];
sys1=Flatten[Table[c[h,j]==0,{h,1,n},{j,0,r-1}]];
m=Flatten[Table[a[h,i],{h,1,n},{i,1,r}]];

(*Determination of coefficients of power series*)
coeff=Solve[sys1,m]; sol=Flatten[Table[xse[h],{h,1,n}]/.coeff];
ta1=Table[If[Head[unk0[[i]]]===Real||Head[unk0[[i]]]===Rational||
Head[unk0[[i]]]===Integer,1,0],{i,1,Length[unk0]}];
If[ta1===Table[0,{i,Length[unk0]}],Goto[2],Goto[1]];
Label[1];

(*Numerical solutions*)
Do[Print[unk[[h]]," =",N[sol[[h]]//Normal,3]," + o[",r+1,"]"],{h,1,n}];
sost1=Table[unk[[h]]->unk[[h]][var],{h,1,n}];
ss=NDSolve[Table[{unk[[h]]'[var]==(sys[[h,2]]/.sost1),unk[[h]][T1]==unk0[[h]]},
{h,1,n}]//Flatten,unk,{var,T1,T2},MaxSteps->steps];

(*Graphics of power solutions and numerical solutions*)
pls[h_]:=Plot[Evaluate[Normal[sol[[h]]]],{var,T1,T2},AspectRatio->1,DisplayFunction->Identity];
pln[h_]:=Plot[Evaluate[unk[[h]][var]/.ss],{var,T1,T2},
PlotStyle->{Dashing[{0.015,0.015}]},AspectRatio->1,DisplayFunction->Identity];
Do[Show[pls[h],pln[h],AxesLabel->{StyleForm[var,FontSlant->"Italic"],
StyleForm[unk[[h]],FontSlant->"Italic"]},DisplayFunction->$DisplayFunction], {h,1,n}];
StylePrint["The dashed lines refer to the numerical solution and the continuous lines refer to the power expansions.","Output",FontFamily->"Times-Plain",FontSize->12];
Goto[3];
Label[2];
Do[Print[unk[[h]]," = ",(sol[[h]]//Normal)," +o[",r+1,"]"],{h,1,n}];
Label[3];
]



(*
 --------------------------------------------------------------------------------------------
 -
 -   T H E   P R O G R A M  Frobenius:
 -
 -                 UsageFrobenius[]
 -                 HelpFrobenius[]
 -                 Frobenius[eq_,unk_, var_,order_,data1_,data2_,steps_]
 -
 --------------------------------------------------------------------------------------------
*)


UsageFrobenius[]:=
Module[{},
StylePrint["Aims of the program Frobenius","Output",FontFamily->"Times-Bold",FontSize->12];
StylePrint["The program supplies the Frobenius power series solutions of a second order linear differential equation","Output",FontFamily->"Times-Plain",FontSize->10];
Print[DisplayForm[SuperscriptBox["x", "2"]]," y'' + x P(x) y' + Q(x) y = 0"];
StylePrint["with a regular singularity at the origin. Moreover, it compares the numerical and power series solutions relative to a given Cauchy problem.","Output",FontFamily->"Times-Plain",FontSize->10];
StylePrint["About the hypotheses satisfied by the function P and Q, see Chapter 3.","Output",FontFamily->"Times-Plain",FontSize->10];
StylePrint["The command line writes","Output",FontFamily->"Times-Plain",FontSize->10];
StylePrint["Frobenius[eq, unk, var, order, data1, data2, steps]","Output",FontFamily->"Times-Plain",FontSize->10];
]

HelpFrobenius[]:=
Module[{},
StylePrint["How to use the program Frobenius","Output",FontFamily->"Times-Bold",FontSize->12];
StylePrint["To make active the program type in the following input data","Output",FontFamily->"Times-Plain",FontSize->10];
StylePrint["eq = linear second order equation with a regular singularity at the origin only in the form ","Output",FontFamily->"Times-Plain",FontSize->10];
StylePrint["\!\(x\^2\) y''+ x P(x) y' + Q(x) y = 0   (f.e. {\!\(x\^2\) y'' + x y' + 2y == 0 };)","Output",FontFamily->"Times-Plain",FontSize->10];
StylePrint["unk  = unknown (f.e. y;)","Output",FontFamily->"Times-Plain",FontSize->10];
StylePrint["var  = independent variable (f.e. x;)","Output",FontFamily->"Times-Plain",FontSize->10];
StylePrint["order = order of the Frobenius series (f.e. 5;)","Output",FontFamily->"Times-Plain",FontSize->10];
StylePrint["data1 = interval of the independent variable, where the left bound is not zero (f.e. {0.3, 2};)","Output",FontFamily->"Times-Plain",FontSize->10];
StylePrint["data2 = initial data (f.e.{y[0.3] == 0, y'[0.3] == 1};)","Output",FontFamily->"Times-Plain",FontSize->10];
StylePrint["steps = number of steps in the numerical integration (f.e. 1000;)","Output",FontFamily->"Times-Plain",FontSize->10];
StylePrint["Frobenius[eq, unk, var, order, data1, data2, steps]","Output",FontFamily->"Times-Plain",FontSize->10];
]


Frobenius[eq_,unk_, var_,order_,data1_,data2_,steps_] :=
Module[{sost,P,Q,p0,q0,indpol,polroot,rad1,rad2,serP,serQ,coeffP,coeffQ,a,
sol1,b,sol2,c,tab1,tab2,solpar,com,tabcom,sol2fi},

(*unk is expressed as a function of the indipendent variable*)
sost={unk->unk[var],unk'->unk'[var],unk''->unk''[var]};
eq1=eq/.sost;
P=Coefficient[eq[[1]],unk']/var;
Q=Coefficient[eq[[1]],unk];
p0=P/.var->0;
q0=Q/.var->0;

(*Verify that the origin is a regular singular point*)
If[Head[Limit[P,var->0]]===DirectedInfinity||Head[Limit[Q,var->0]]===
DirectedInfinity,StylePrint["The origin is not a regular singular point","Output",FontFamily->"Times-Plain",FontSize->12];Goto[end]];

(*Roots of the indicial polynomial*)
indpol[root_]:=(root^2)+(p0-1)root+q0;
polroot=Expand[{(-(p0-1)-Sqrt[(p0-1)^2-4*q0])/2,(-(p0-1)+Sqrt[(p0-1)^2-4*q0])/2}];

(*The case of complex roots*)
If[Im[polroot[[1]]]!=0, StylePrint["The roots of indicial polynomial are conjugate complex.","Output",FontFamily->"Times-Plain",FontSize->12];
StylePrint["The program is not applicable","Output",FontFamily->"Times-Plain",FontSize->12];
Goto[end], Goto[2]];

(*The case of real roots*)
Label[2];
rad1=Max[polroot];
rad2=Min[polroot];


StylePrint["The roots of the indicial polynomial are","Output",FontFamily->"Times-Plain",FontSize->12];
Print["r1 = ",rad1,"  r2 = ",rad2];

(*Taylor's series of P and Q*)
serP=Normal[Series[P,{var,0,order}]];
serQ=Normal[Series[Q,{var,0,order}]];
coeffP[0]=p0;
coeffQ[0]=q0;
Do[
coeffP[i]=Coefficient[serP,var^i];
coeffQ[i]=Coefficient[serQ,var^i],
{i,1,order}];

(*The first solution: for all the roots of the indicial polynomial*)
a[0]=1 (* not zero, for instance = 1*);
Do[
a[i]=-(1/(indpol[rad1+i]))Sum[((k+rad1)coeffP[i-k]+coeffQ[i-k])a[k],{k,0,i-1}],
{i,1,order}];

sol1=(var^rad1)Sum[a[k] var^k, {k,0,order}];
StylePrint["The first solution is","Output",FontFamily->"Times-Plain",FontSize->12];
Print[unk,"[1] = ",Expand[sol1]];
StylePrint["or in numerical form","Output",FontFamily->"Times-Plain",FontSize->12];
Print[unk,"[1] = ",N[sol1,3]];

(*Second solution: the different possibilities*)
Which[Head[rad1-rad2]===Integer&&rad1!=rad2,Goto[3],rad1===rad2,Goto[4],Im[rad1]==0,Goto[5]];
Label[3];
rad1=Max[polroot];
rad2=Min[polroot];

(*The difference rad1-rad2 is a positive integer*)
sol2=A sol1 Log[var]+(var^rad2)(Sum[b[i] var^i,{i,0,order}]);

(*Here A could be zero but b[0]=1*)
equaz=Expand[eq[[1]]/.{unk->sol2,unk'->D[sol2,var],unk''->D[sol2,{var,2}]}];
b[0]=1;
c[0]=Coefficient[equaz,var,0]/.var->0;
c[i_]:=Coefficient[equaz,var^i];
tab1=Table[c[i]==0,{i,0,order}];
tab2=Join[{A},Table[b[i],{i,1,order}]]//Flatten;
solpar=Solve[tab1,tab2]//Flatten;

solpar1=Table[solpar[[i,1]],{i,1,Length[solpar]}];
com=Complement[tab2,solpar1];
tabcom=Table[com[[i]]->1,{i,1,Length[com]}];
If[Length[tabcom]!=0,solpar2=Join[solpar/.tabcom,tabcom],solpar2=solpar];

sol2fi=(sol2/.solpar2);
StylePrint["The second solution when the roots of indicial polynomial are real, not equal, and differ for a positive integer is","Output",FontFamily->"Times-Plain",FontSize->12];
Goto[print];
Label[4];
rad1=Max[polroot];
rad2=Min[polroot];

(*rad1 and rad2 are real and the difference rad1-rad2= 0*)
sol2=sol1*Log[var]+(var^rad1)(1+Sum[b[i]var^i,{i,1,order}]);
equaz=Expand[eq[[1]]/.{unk->sol2,unk'->D[sol2,var],unk''->D[sol2,{var,2}]}];
c[i_]:=If[i+rad2!=0,Coefficient[equaz,var^(i+rad2)],Coefficient[equaz,var,0]];
tab1=Table[c[i]==0,{i,1,order}];
tab2=Table[b[i],{i,1,order}];
solpar=Solve[tab1,tab2]//Flatten;

solpar1=Table[solpar[[i,1]],{i,1,Length[solpar]}];
com=Complement[tab2,solpar1];
tabcom=Table[com[[i]]->1,{i,1,Length[com]}];
If[Length[tabcom]!=0,solpar2=Join[solpar/.tabcom,tabcom],solpar2=solpar];

sol2fi=Expand[(sol2/.solpar2)];
StylePrint["The second solution when the roots of indicial polynomial are real and equal is","Output",FontFamily->"Times-Plain",FontSize->12];
Goto[print];
Label[5];
rad1=Max[polroot];
rad2=Min[polroot];

(*second solution when the difference rad1-rad2 is not a positive integer*)
b[0]= 1;(* not zero, for instance = 1*)
Do[
b[i]=-(1/(indpol[rad2+i]))Sum[((k+rad2)coeffP[i-k]+coeffQ[i-k])b[k],{k,0,i-1}],
{i,1,order}];

sol2fi=Expand[(var^rad2)Sum[b[k] var^k, {k,0,order}]];
StylePrint["The second solution when the difference of the roots of indicial polynomial is not equal to an integer is","Output",FontFamily->"Times-Plain",FontSize->12];
Goto[print];
Label[print];
Print[unk,"[2] = ",Expand[sol2fi]];
StylePrint["or in numerical form","Output",FontFamily->"Times-Plain",FontSize->12];
Print[unk,"[2] = ",N[sol2fi,3]];

If[Length[data1]==0&&Length[data2]==0,Goto[end],Goto[simul]];
Label[simul];
solgen=B1 sol1+B2 sol2fi;
cond1=(solgen/.var->data1[[1]])==data2[[1,2]];
cond2=(D[solgen,var]/.var->data1[[1]])==data2[[2,2]];
sysgen=Solve[{cond1,cond2},{B1,B2}];
solgenfi=solgen/.sysgen;
solnum=NDSolve[{eq1,data2[[1]],data2[[2]]},unk,{var,data1[[1]],data1[[2]]},MaxSteps->steps];

StylePrint["The following plot compares the numerical solution (continuous line) with the series solution (dashed line)","Output",FontFamily->"Times-Plain",FontSize->12];
plfi=Plot[{solgenfi,Evaluate[unk[var]/.solnum]},{var,data1[[1]],data1[[2]]},
AxesLabel->{StyleForm[var,FontSlant->"Italic"],StyleForm[unk,FontSlant->"Italic"]},
PlotStyle->{Dashing[{0.03,0.03}],Thickness[0.01]}];
Label[end];
]



(*
 --------------------------------------------------------------------------------------------
 -
 -   T H E   P R O G R A M  Poincare:
 -
 -                 UsagePoincare[]
 -                 HelpPoincare[]
 -                 Poincare[sys_,unk_,var_,unk0_,par_,m_,r_,T1_,T2_,steps_,pref_]
 -
 --------------------------------------------------------------------------------------------
*)


UsagePoincare[]:=
Module[{},
StylePrint["Aims of the program Poincare","Output",FontFamily->"Times-Bold",FontSize->12];
StylePrint["The program supplies the solution of the Cauchy problem ","Output",FontFamily->"Times-Plain",FontSize->10];
Print[DisplayForm[SubscriptBox[SuperscriptBox["x", ","], "i"]]," = ",DisplayForm[UnderoverscriptBox["\[Sum]","j=1","n"]],DisplayForm[SubscriptBox["A", "i,j"]],DisplayForm[SubscriptBox["x", "j"]]," + \[Mu] ", DisplayForm[SubscriptBox["F", "i"]],"(",DisplayForm[SubscriptBox["x", "1"]],",..., ",DisplayForm[SubscriptBox["x", "n"]],", t), "];
Print[DisplayForm[SubscriptBox["x", "i"]],"(T1) = ",DisplayForm[SubscriptBox[SuperscriptBox["x", "0"], "i"]],",      i = 1, ..., n"];
StylePrint["as a power series in the parameter \[Mu] of order r and makes the graphical comparison between the approximate solution and the numerical one. The program supplies the phase portrait when the number of the unknowns is two. The command line writes  ","Output",FontFamily->"Times-Plain",FontSize->10];
StylePrint["Poincare[sys, unk, var, unk0, par, m, r, T1, T2, steps, pref].","Output",FontFamily->"Times-Plain",FontSize->10];
StylePrint["Here steps denotes the number of steps in the numerical integration","Output",FontFamily->"Times-Plain",FontSize->10];
]

HelpPoincare[]:=
Module[{},
StylePrint["How to use the program Poincare","Output",FontFamily->"Times-Bold",FontSize->12];
StylePrint["To make active the program type in the following input data","Output",FontFamily->"Times-Plain",FontSize->10];
StylePrint["sys = list of the equations (f.e. {x' == y, y' == - x - \[Mu]*x^3};)","Output",FontFamily->"Times-Plain",FontSize->10];
StylePrint["unk = list of the unknowns (f.e. {x, y};)","Output",FontFamily->"Times-Plain",FontSize->10];
StylePrint["var=  independent variable (f.e t;)","Output",FontFamily->"Times-Plain",FontSize->10];
StylePrint["unk0 = list of the initial data (f.e. {0, 0};)","Output",FontFamily->"Times-Plain",FontSize->10];
StylePrint["par = parameter appearing in the system (f.e. \[Mu];)","Output",FontFamily->"Times-Plain",FontSize->10];
StylePrint["m = parameter value (f.e. 0.1;)","Output",FontFamily->"Times-Plain",FontSize->10];
StylePrint["r = order of the expansion (f.e. 2;)","Output",FontFamily->"Times-Plain",FontSize->10];
StylePrint["T1 = left bound of time interval (T1, T2) (f.e.1;)","Output",FontFamily->"Times-Plain",FontSize->10];
StylePrint["T2 = right bound of time interval (T1, T2) (f.e.2;)","Output",FontFamily->"Times-Plain",FontSize->10];
StylePrint["steps = number of steps in the numerical integration (f.e. 1000;)","Output",FontFamily->"Times-Plain",FontSize->10];
StylePrint["pref = fixes the numerical form of the coefficients of the expansion (f.e. frac;)","Output",FontFamily->"Times-Plain",FontSize->10];
StylePrint["Poincare[sys, unk, var, unk0, par, m, r, T1, T2, steps, pref]","Output",FontFamily->"Times-Plain",FontSize->10];
StylePrint["If pref = frac, the coefficients of the expansion, have the same form of the input data. On the contrary, if pref is different from frac, the coefficients are expressed as decimal numbers with three digits.","Output",FontFamily->"Times-Plain",FontSize->10];
]


Poincare[sys_,unk_,var_,unk0_,par_,m_,r_,T1_,T2_,steps_,pref_]:=
Module[{n,h,sost,xs,u,eq,s,inc,system,sol,y,p,q,ap,eseq,iniz,pl,pp},

(*The system*)
n=Length[sys];
Do[
xs[h]=SeriesData[par,T1,Table[u[h,k][var],{k,0,m}]],
{h,1,n}];
sost=Table[{unk[[h]]->xs[h],unk[[h]]'->D[xs[h],var]},{h,1,n}]//Flatten;
Do[
eq[h]=sys[[h]]/.sost,
{h,1,n}];

(*s[h] includes all the approximations u[h,k] of the unknown x[h]*)
Do[
s[h]=LogicalExpand[eq[h]],
{h,1,n}];
inc[k_]:=Table[u[h,k][var],{h,1,n}];

(*sol[0] represents the solution of zero order; sol[k] represents the kth-order solution*)
Do[
system[0]=Flatten[Table[{s[h][[1]],u[h,0][0]==unk0[[h]]},
{h,1,n}]];

sol[0]=Flatten[DSolve[system[0],inc[0],var]];
system[k]=Flatten[{Table[s[i][[k+1]],{i,1,n}]/.Flatten[Table[sol[j],{j,0,k-1}]],Table[u[i,k][0]==0,{i,1,n}]}];
sol[k]=Flatten[DSolve[system[k],inc[k],var]], {k,1,m}];

Do[
y1[h]=Collect[ComplexExpand[sol[0][[h,2]]+Sum[(par^k)*sol[k][[h,2]],{k,1,m}]]//Simplify,par],
{h,1,n}];

If[pref===Global`frac,Do[Print[unk[[h]]," = ",y1[h]/.Arg[x_]->0],{h,1,n}],
Do[Print[unk[[h]]," = ",N[Chop[y1[h]/.Arg[x_]->0],3]],{h,1,n}]];

(*Numerical integration of the complete equation and its representation*)
Clear[xs];
sost1=Table[{unk[[h]]->unk[[h]][var],unk[[h]]'->unk[[h]]'[var],par->r},{h,1,n}]//Flatten;
sys1=Flatten[Table[{sys[[h]]/.sost1,unk[[h]][T1]==unk0[[h]]},{h,1,n}]];
sx=NDSolve[sys1,unk,{var,T1,T2},MaxSteps->steps];
ap[i_]:=Plot[y1[i]/.par->r,{var,T1,T2},AxesLabel->{StyleForm[var,FontSlant->"Italic"],StyleForm[unk[[i]],FontSlant->"Italic"]},DisplayFunction->Identity];
pl[i_]:=Plot[Evaluate[unk[[i]][var]/.sx],{var,T1,T2},DisplayFunction->Identity,PlotStyle->{Dashing[{0.03,0.03}]}];

Do[
Show[{ap[i],pl[i]},DisplayFunction->$DisplayFunction],
{i,1,n}];

If[n==2,Goto[1],Goto[2]];
Label[1];

(*Phase portrait*)
aparp=ParametricPlot[Evaluate[{y1[1]/.par->r,y1[2]/.par->r}],{var,T1,T2},AxesLabel->{unk[[1]],unk[[2]]},DisplayFunction->Identity];
pp=ParametricPlot[Evaluate[{unk[[1]][var],unk[[2]][var]}/.sx],{var,T1,T2},DisplayFunction->Identity,PlotStyle->{Dashing[{0.03,0.03}]}];
Show[aparp,pp,DisplayFunction->$DisplayFunction,AxesLabel->{StyleForm[unk[[1]],FontSlant->"Italic"],
StyleForm[unk[[2]],FontSlant->"Italic"]}];
Label[2];
StylePrint["The dashed lines refer to the numerical solution and the continuous lines refer to the approximate solution.","Output",FontFamily->"Times-Plain",FontSize->12];
]


(*
 --------------------------------------------------------------------------------------------
 -
 -   T H E   P R O G R A M  Liapunov:
 -
 -                 UsageLiapunov[]
 -                 HelpLiapunov[]
 -                 Liapunov[sys_, unk_, W_, m_, viewx_, viewy_, valV_, valV1_,points_]
 -
 --------------------------------------------------------------------------------------------
*)


UsageLiapunov[]:=
Module[{},
StylePrint["Aims of the program Liapunov","Output",FontFamily->"Times-Bold",FontSize->12];
StylePrint["The direct Liapunov method is applied to evaluate the origin stability of a planar system by a homogeneuos polynomial W. The even degree m of W is fixed starting from two; then the signes of W and W' along the solutions are graphically determined.","Output",FontFamily->"Times-Plain",FontSize->10];
StylePrint["The command is","Output",FontFamily->"Times-Plain",FontSize->10];
StylePrint["Liapunov[sys, unk, W, m, viewx, viewy, valV, valV1, options]","Output",FontFamily->"Times-Plain",FontSize->10];
]

HelpLiapunov[]:=
Module[{},
StylePrint["How to use the program Liapunov","Output",FontFamily->"Times-Bold",FontSize->12];
StylePrint["It searches for polynomial Liapunov functions for a given planar system ","Output",FontFamily->"Times-Plain",FontSize->10];
StylePrint["To make active the program type in the following input data","Output",FontFamily->"Times-Plain",FontSize->10];
StylePrint["sys = list of the equations (f.e.{x' == y + x y, y' == - x - y \!\(x\^3\)};)","Output",FontFamily->"Times-Plain",FontSize->10];
StylePrint["unk = list of the unknowns (f.e.{x, y};)","Output",FontFamily->"Times-Plain",FontSize->10];
StylePrint["W = a homogeneous mth-order polynomial (f.e.\!\(x\^2\) + \!\(y\^2\);)","Output",FontFamily->"Times-Plain",FontSize->10];
StylePrint["m = order of W (f.e. 2;)","Output",FontFamily->"Times-Plain",FontSize->10];
StylePrint["viewx = interval around the origin on the first axis (f.e.{-1, 2;)","Output",FontFamily->"Times-Plain",FontSize->10];
StylePrint["viewy = interval around the origin on the second axis (f.e. {-1, 2;})","Output",FontFamily->"Times-Plain",FontSize->10];
StylePrint["valV = a prefixed value of the Liapunov function at the point {viewx[[1]], viewy[[1]]","Output",FontFamily->"Times-Plain",FontSize->10];
StylePrint["valV1 = a prefixed value of the derivative of Liapunov function at the point {viewx[[1]], viewy[[1]]","Output",FontFamily->"Times-Plain",FontSize->10];
StylePrint["options = number of plot points ","Output",FontFamily->"Times-Plain",FontSize->10];
StylePrint["Liapunov[sys, unk, W, m, viewx, viewy, valV, valV1, options]","Output",FontFamily->"Times-Plain",FontSize->10];
]


Liapunov[sys_, unk_, W_, m_, viewx_, viewy_, valV_, valV1_,points_]:=
Module[{},
Off[Solve::svars];
A = {{D[sys[[1,2]], unk[[1]]], D[sys[[1,2]], unk[[2]]]},
    {D[sys[[2,2]], unk[[1]]], D[sys[[2,2]], unk[[2]]]}} /.
    {unk[[1]] -> 0, unk[[2]] -> 0};
eig = Eigenvalues[A];
StylePrint["The eigenvalues of the Jacobian matrix at the origin are","Output",FontFamily->"Times-Plain",FontSize->12];

Print[Re[eig]+I Im[eig]];
se = Sign[Re[eig]];
tn = Select[se, #1 == -1 & ];
tp = Select[se, #1 == 1 & ];
t0 = Select[se, #1 == 0 & ];
If[Length[t0]+Length[tn]==2&&Length[t0]>0, StylePrint["This is a critical case. If the system is not linear, apply either CriticalEqS or CriticalEqN.","Output",FontFamily->"Times-Plain",FontSize->12];
Goto[2], Goto[1]];

Label[1];
V = Sum[a[j]*unk[[1]]^(m - j)*unk[[2]]^j, {j, 0, m}];
pareq = {D[V, unk[[1]]], D[V, unk[[2]]]}.A.{unk[[1]], unk[[2]]};

t1 = Table[Coefficient[pareq, unk[[1]]^(m - j)*unk[[2]]^j],{j, 0, m}];
t1coef=Table[Coefficient[t1[[i]], a[j]], {i, 1, m+1},{j, 0, m}];
d0=Det[t1coef];
eq = Which[Length[tn] == 2, Expand[pareq + W],Length[tp] >= 1&&d0!=0, Expand[pareq - W], Length[tp] >=
1&&d0==0,Expand[pareq-c V-W]];

If[d0==0,Goto[3],Goto[4]];
Label[3];

(*Analysis of the case d0=0*)
Label[3];
reeigp=Select[eig,Re[#]>0&,1];
tra=A[[1,1]]+A[[2,2]];
de=Det[A];
radD=Table[{(m/2)*tra-i*Sqrt[(tra^2)-4*de],(m/2)*tra+i*Sqrt[(tra^2)-4*de]},{i,0,m/2}]//Flatten;
reraD=DeleteCases[radD,_Complex];
craD=Select[reraD,#>0&;#<m*reeigp&];
maxc=Max[craD];
If[Length[craD]!=0,c11=(maxc+reeigp)/2,c11=(m/2)*reeigp];
Label[4];
coef = Table[Coefficient[eq, unk[[1]]^(m - j)*unk[[2]]^j] == 0,{j, 0, m}];
inc= Table[a[j], {j, 0, m}];
coefV =If[d0==0,(Flatten[Solve[coef,inc]]/.c->c11[[1]]),Flatten[Solve[coef, inc]]];
Vfi = V /. coefV;
V1 = Expand[{D[Vfi, unk[[1]]], D[Vfi, unk[[2]]]} . {sys[[1,2]], sys[[2,2]]}];

StylePrint["A possible Liapunov function is","Output",FontFamily->"Times-Plain",FontSize->12];
Print["V = ", Vfi];

StylePrint["and its derivative is","Output",FontFamily->"Times-Plain",FontSize->12];
Print["V' = ", V1];

StylePrint["On the dashed lines the Liapunov function is negative, on the bold lines it vanishes, and on the continuous lines it is positive.","Output",FontFamily->"Times-Plain",FontSize->12];
Print[""];

StylePrint["V - plot","Output",FontFamily->"Times-Plain",FontSize->12];
pl1 = ImplicitPlot[{Vfi == -valV, Vfi == -valV/2,Vfi == -valV/3, Vfi == 0, Vfi == valV/3,Vfi == valV/2, Vfi == valV}, {unk[[1]], viewx[[1]], viewx[[2]]},{unk[[2]], viewy[[1]], viewy[[2]]}, AxesOrigin -> {0, 0},AxesLabel ->{StyleForm[unk[[1]], FontSlant -> "Italic"], StyleForm[unk[[2]], FontSlant -> "Italic"]},PlotStyle -> {{GrayLevel[0], Dashing[{0.05, 0.03}]},{GrayLevel[0], Dashing[{0.05, 0.03}]}, {GrayLevel[0], Dashing[{0.05, 0.03}]},{GrayLevel[0], AbsoluteThickness[3]}, {GrayLevel[0]},{GrayLevel[0]}, {GrayLevel[0]}},PlotPoints->points];

StylePrint["On the dashed lines the Liapunov function derivative is negative, on the bold lines it vanishes, and on the continuous lines it is positive.","Output",FontFamily->"Times-Plain",FontSize->12];
Print[""];


StylePrint["V' - plot","Output",FontFamily->"Times-Plain",FontSize->12];


pl2 = ImplicitPlot[{V1 == -valV1, V1 == -valV1/2,V1 == -valV1/3, V1 == 0, V1 == valV1/3,V1 == valV1/2, V1 == valV1},{unk[[1]], viewx[[1]], viewx[[2]]},{unk[[2]], viewy[[1]], viewy[[2]]},AxesOrigin -> {0, 0}, AxesLabel ->{StyleForm[unk[[1]], FontSlant -> "Italic"], StyleForm[unk[[2]], FontSlant -> "Italic"]}, PlotStyle ->{{GrayLevel[0], Dashing[{0.05, 0.03}]}, {GrayLevel[0],Dashing[{0.05, 0.03}]},{GrayLevel[0], Dashing[{0.05, 0.03}]}, {GrayLevel[0], AbsoluteThickness[3]}, {GrayLevel[0]}, {GrayLevel[0]},{GrayLevel[0]}},PlotPoints->points];
Label[2];
]


(*
 --------------------------------------------------------------------------------------------
 -
 -   T H E   P R O G R A M  LStability:
 -
 -                 UsageLStability[]
 -                 HelpLStability[]
 -                 LStability[sys_, unk_, xe_]
 -
 --------------------------------------------------------------------------------------------
*)


UsageLStability[]:=
Module[{},
StylePrint["Aims of the program LStability","Output",FontFamily->"Times-Bold",FontSize->12];
StylePrint["Analysis of stability of already known equilibrium configurations by linear methods ","Output",FontFamily->"Times-Plain",FontSize->10];
StylePrint["The line command writes","Output",FontFamily->"Times-Plain",FontSize->10];
StylePrint["LStability[sys, unk, xe]","Output",FontFamily->"Times-Plain",FontSize->10];
]

HelpLStability[]:=
Module[{},
StylePrint["How to use the program LStability","Output",FontFamily->"Times-Bold",FontSize->12];
StylePrint["The program determines the stability of an equilibrium position by a linear analysis ","Output",FontFamily->"Times-Plain",FontSize->10];
StylePrint["To make active the program type in the following input data","Output",FontFamily->"Times-Plain",FontSize->10];
StylePrint["sys = list of the equations (f.e.{x' == y, y' == - x};)","Output",FontFamily->"Times-Plain",FontSize->10];
StylePrint["unk = list of the unknowns (f.e. {x, y};)","Output",FontFamily->"Times-Plain",FontSize->10];
StylePrint["xe = list of the equilibrium points (f.e. {{0, 0}};)","Output",FontFamily->"Times-Plain",FontSize->10];
StylePrint["LStability[sys, unk, xe]","Output",FontFamily->"Times-Plain",FontSize->10];
]


LStability[sys_, unk_, xe_] :=
 Module[{n,p,ml,contr,jacob,Jac,b,sl,spos,szero},

(*Control the equilibrium*)
 n = Length[unk];
 p = Length[xe];
 Do[ml[k] = Flatten[Table[{unk[[i]] -> xe[[k,i]]}, {i, 1, n}]];
 contr[k] = Table[sys[[i,2]] /. ml[k], {i, 1, n}] == Table[0, {n}];
 If[contr[k] == False, Print[xe[[k]], " is not an equilibrium position."];
        Goto[3]], {k, 1, p}];

(*Jacobian matrices at equilibrium and eigenvalues*)
 jacob = Table[D[sys[[i,2]], unk[[j]]], {i, 1, n}, {j, 1, n}];
 Do[Jac[k] = jacob /. ml[k];
 b[k] = Re[Eigenvalues[Jac[k]]] + I*Im[Eigenvalues[Jac[k]]];
 Print[StyleForm["The eigenvalues of the Jacobian matrix at ","Output",FontFamily->"Times-Plain",FontSize->12],xe[[k]],
 StyleForm[" are ","Output",FontFamily->"Times-Plain",FontSize->12]];
 Print[b[k]];
 sl[k] = Table[Sign[Re[b[k][[i]]]], {i, 1, n}];
 spos[k] = Cases[sl[k], 1];
 szero[k] = Cases[sl[k], 0];
 Which[sl[k] === -Table[1, {i, 1, n}],
    Print[StyleForm["The equilibrium point ","Output",
    FontFamily->"Times-Plain",FontSize->12],xe[[k]],
    StyleForm[" is asymptotically stable.","Output",
    FontFamily->"Times-Plain",FontSize->12]],
    Length[spos[k]] > 0, Print[StyleForm["The equilibrium point ","Output",
    FontFamily->"Times-Plain",FontSize->12],xe[[k]],
    StyleForm[" is unstable.","Output",FontFamily->"Times-Plain",FontSize->12]],
    Length[szero[k]] > 0&&Table[sys[[r,2]],{r,1,n}]===Jac[k].(unk-xe[[k]]),
    Print[StyleForm["The equilibrium point ","Output",
    FontFamily->"Times-Plain",FontSize->12],xe[[k]],
    StyleForm[" is neutrally stable.","Output",FontFamily->"Times-Plain",FontSize->12]],
    Length[szero[k]] > 0,
    Print[StyleForm["The equilibrium point ","Output",FontFamily->"Times-Plain",FontSize->12],xe[[k]],
    StyleForm[" is critical.","Output",FontFamily->"Times-Plain",FontSize->12]]],
  {k, 1, p}];
  Label[3];
]


(*
 --------------------------------------------------------------------------------------------
 -
 -   T H E   P R O G R A M  CriticalEqN:
 -
 -                 UsageCriticalEqN[]
 -                 HelpCriticalEqN[]
 -
 -                 CriticalEqN1[sys_, unk_]
 -                 CriticalEqN2[sys_,unk_]
 -                 CriticalEqN[sys_,unk_]
 -
 --------------------------------------------------------------------------------------------
*)


UsageCriticalEqN[]:=
Module[{},
StylePrint["Aims of the program CriticalEqN","Output",FontFamily->"Times-Bold",FontSize->12];
StylePrint["The origin equilibrium of the system is analyzed by Poincar's method","Output",FontFamily->"Times-Plain",FontSize->10];
Print[TableForm[{"x' = \!\(a\_11\) x + \!\(a\_12\) y + P(x, y)", "y' = \!\(a\_21\) x + \!\(a\_22\) y + Q(x, y)"}, TableDepth -> 2,TableSpacing->2]];
StylePrint["where the matrix with numerical coefficients", "Output", FontFamily -> "Times-Plain", FontSize -> 10];
Print[MatrixForm[{{\!\(Global`a\_11\), \!\(Global`a\_12\)}, {\!\(Global`a\_21\), \!\(Global`a\_22\)}}]];
StylePrint["has the eigenvalues  i \[Beta], \[Beta] > 0, the functions P and Q are at least of the order \!\(x\^2\) + \!\(y\^2\) and contain only numerical coefficients.", "Output", FontFamily -> "Times-Plain", FontSize -> 10];
StylePrint["The command is","Output",FontFamily->"Times-Plain",FontSize->10];
StylePrint["CriticalEqN[sys, unk]","Output",FontFamily->"Times-Plain",FontSize->10];
]

HelpCriticalEqN[]:=
Module[{},
StylePrint["How to use the program CriticalEqN","Output",FontFamily->"Times-Bold",FontSize->12];
StylePrint["The program CriticalEqN is applied to planar systems with numerical coefficients.","Output",FontFamily->"Times-Plain",FontSize->10];
StylePrint["To make active the program type in the following input data","Output",FontFamily->"Times-Plain",FontSize->10];
StylePrint["sys = list of the equations (f.e.{x' == y, y' == - x - \!\(x\^3\)};)","Output",FontFamily->"Times-Plain",FontSize->10];
StylePrint["unk = list of the unknowns (f.e.{x, y};)","Output",FontFamily->"Times-Plain",FontSize->10];
StylePrint["CriticalEqN[sys, unk]","Output",FontFamily->"Times-Plain",FontSize->10];
]


CriticalEqN1[sys_, unk_] :=
Module[{P,Q,A, dp, p, dq, q, f, gradf, S, W, g, c, cc, sost, eq, sol, matc,r1,ns,m,j,pr, seven, t1seven},
Off[Solve::svars];
ffin=sys[[1,2]];
gfin=sys[[2,2]];
jfin={{D[ffin,unk[[1]]],D[ffin,unk[[2]]]},{D[gfin,unk[[1]]],D[gfin,unk[[2]]]}}/.{unk[[1]]->0,unk[[2]]->0};
P=sys[[1,2]]-jfin[[1]].unk;
Q=sys[[2,2]]-jfin[[2]].unk;
A = {{0,jfin[[1,2]]},{jfin[[2,1]],0}};
If[P==0&&Q==0,StylePrint["The method is not applicable since the system is linear.","Output",FontFamily->"Times-Plain",FontSize->12];
Goto[a1]];

(*Terms of m-th order of Taylor's developments of P and Q at (0,0) *)
dp[m_,i_]:= D[P,{unk[[1]],m-i},{unk[[2]],i}]/.{unk[[1]]->0,unk[[2]]->0};
p[m_]:=Sum[(1/m!)*Binomial[m,i]*dp[m, i]*(unk[[1]]^(m - i))*unk[[2]]^i, {i,0,m}];
dq[m_,i_]:=D[Q,{unk[[1]],m-i},{unk[[2]],i}]/.{unk[[1]] -> 0, unk[[2]] -> 0};
q[m_]:= Sum[(1/m!)*Binomial[m, i]*dq[m, i]*(unk[[1]]^(m - i))*unk[[2]]^i, {i, 0, m}];

(*Auxiliary polinomials*)
f[m_]:=Sum[b[m-i,i]*(unk[[1]]^(m-i))*unk[[2]]^i,{i,0,m}];

(*Relations gradf[m].A.unk=-2{p[m-1],q[m-1]}.unk-Sum[gradf[i+1].{p[i],q[i]}, {i,2,m-2}]*)
gradf[m_]:={D[f[m],unk[[1]]],D[f[m],unk[[2]]]};
S[m_]:=ExpandAll[Sum[gradf[m+1-i].{p[i],q[i]},{i,2,m-2}]];
W[m_]:=ExpandAll[-2*{p[m-1],q[m-1]}.{unk[[1]],unk[[2]]}-S[m]];
g[m_]:=ExpandAll[G[m]*(unk[[1]]^2+unk[[2]]^2)^(m/2)];
c[m_]:=ExpandAll[gradf[m].A.{unk[[1]],unk[[2]]}];
b[m_]:=Table[Coefficient[f[m],(unk[[1]]^(m-i))*unk[[2]]^i],{i,0,m}];
cc[m_]:=Table[Coefficient[c[m],(unk[[1]]^(m-i))*unk[[2]]^i],{i,0,m}];
k=4;
Label[0];
Do[
sost[m]=Flatten[Table[sol[i],
{i,3,m-1}]];

eq[m]=If[OddQ[m]==True,c[m]-(W[m]/.sost[m])==0,c[m]==(W[m]/.sost[m])+g[m]];
coeffsys[m]=
If[OddQ[m]==True,sysodd[m]=Table[Coefficient[eq[m][[1]],(unk[[1]]^(m-i))*unk[[2]]^i]==0,{i,0,m}],
  syseven[m]=Table[Coefficient[eq[m][[1]],(unk[[1]]^(m-i))*unk[[2]]^i]==
  Coefficient[eq[m][[2]],(unk[[1]]^(m-i))*unk[[2]]^i],{i,0,m}]];

If[OddQ[m]==False,matc[m]=Table[Coefficient[coeffsys[m][[i+1,1]],b[m-j,j]],{i,0,m},{j,0,m}];
r1[m]=Table[coeffsys[m][[i,2]],{i,1,Length[matc[m]]}];
ns[m]=NullSpace[Transpose[matc[m]]];
valG[m]=Solve[ns[m].r1[m]==0,G[m]];

If[valG[m][[1,1,2]]===0,Goto[3],Goto[2]];

Label[3];
r2[m]=Flatten[coeffsys[m]/.valG[m]];
seven[m]=Solve[r2[m],b[m]]//Flatten;
t1seven[m]=Table[seven[m][[i,2]],{i,1,Length[seven[m]]}];
pr[m]=Cases[t1seven[m],xv_+yv_][[1,2]];
s[m]=Solve[r2[m]/.pr[m]->0,b[m]]//Flatten;
soleven[m]=Join[s[m],{pr[m]->0}]];
sol[m]=If[OddQ[m]==True,

If[m==3,Solve[coeffsys[m],b[m]]//Flatten,Solve[coeffsys[m],b[m]]//Flatten],soleven[m]],{m,3,k}];

k=k+2;
Goto[0];
Label[2];
G=valG[k][[1,1,2]];
Print["G[",k,"] = ",valG[k][[1,1,2]]];

If[valG[k][[1,1,2]]<0,StylePrint["The origin is asymptotically stable.","Output",FontFamily->"Times-Plain",FontSize->12],
   StylePrint["The origin is unstable.","Output",FontFamily->"Times-Plain",FontSize->12]];

Label[a1];
]


CriticalEqN2[sys_,unk_]:=
Module[{},
sys1=sys/.unk[[1]]->ninc;
unk1=unk/.unk[[1]]->ninc;
fu=sys1[[1,2]];
gu=sys1[[2,2]];
If[(fu/.{unk1[[1]]->0,unk1[[2]]->0})==0&&(gu/.{unk1[[1]]->0,unk1[[2]]->0})==0,Goto[u1],
   StylePrint["The origin is not an equilibrium point.","Output",FontFamily->"Times-Plain",FontSize->12];
   Goto[u2]];

Label[u1];
ju={{D[fu,ninc],D[fu,unk[[2]]]},{D[gu,ninc],D[gu,unk[[2]]]}}/.{ninc->0,unk[[2]]->0};

Which[ju[[1,1]]==0&&ju[[2,2]]==0&&Re[Eigenvalues[ju][[1]]]==0&&Re[Eigenvalues[ju][[2]]]==0,
      CriticalEqN1[sys1,unk1],ju[[1,1]]!=0&&Re[Eigenvalues[ju][[1]]]==0
      &&Re[Eigenvalues[ju][[2]]]==0||ju[[2,2]]!=0&&Re[Eigenvalues[ju][[1]]]==0
      &&Re[Eigenvalues[ju][[2]]]==0,NormalForm[sys1,unk1];CriticalEqN1[sysin,{X,Y}],
      Re[Eigenvalues[ju][[1]]]!=0||Re[Eigenvalues[ju][[2]]]!=0,StylePrint["The origin is not critical.","Output",FontFamily->"Times-Plain",FontSize->12]];
Label[u2];
]


CriticalEqN[sys_,unk_]:=
Module[{},
If[unk[[1]]===Global`u,CriticalEqN2[sys,unk];Goto[b2],Label[b1]];
Label[b1];
fin=sys[[1,2]];
gin=sys[[2,2]];

If[(fin/.{unk[[1]]->0,unk[[2]]->0})!=0||(gin/.{unk[[1]]->0,unk[[2]]->0})!=0,

StylePrint["The origin is not an equilibrium point.","Output",FontFamily->"Times-Plain",FontSize->12];
Goto[b2]];
jin={{D[fin,unk[[1]]],D[fin,unk[[2]]]},{D[gin,unk[[1]]],D[gin,unk[[2]]]}}/.{unk[[1]]->0,unk[[2]]->0};

Which[jin[[1,1]]==0&&jin[[2,2]]==0&&Re[Eigenvalues[jin][[1]]]==0&&Re[Eigenvalues[jin][[2]]]==0,
      CriticalEqN1[sys,unk],jin[[1,1]]!=0&&Re[Eigenvalues[jin][[1]]]==0&&
      Re[Eigenvalues[jin][[2]]]==0||jin[[2,2]]!=0&&Re[Eigenvalues[jin][[1]]]==0&&
      Re[Eigenvalues[jin][[2]]]==0,NormalForm[sys,unk];
      CriticalEqN1[sysin,{X,Y}],Re[Eigenvalues[jin][[1]]]!=0||Re[Eigenvalues[jin][[2]]]!=0,
      StylePrint["The origin is not critical.","Output",FontFamily->"Times-Plain",FontSize->12]];
Label[b2];
]


(*
 --------------------------------------------------------------------------------------------
 -
 -   T H E   P R O G R A M  CriticalEqS:
 -
 -                 UsageCriticalEqS[]
 -                 HelpCriticalEqS[]
 -
 -                 CriticalEqS1[sys_, unk_,order_]
 -                 CriticalEqS2[sys_,unk_,order_]
 -                 CriticalEqS[sys_,unk_,order_]
 -
 --------------------------------------------------------------------------------------------
*)


UsageCriticalEqS[]:=
Module[{},
StylePrint["Aims of the program CriticalEqS","Output",FontFamily->"Times-Bold",FontSize->12];
StylePrint["Poincar's method is applied to analyze the origin equilibrium of the system in the canonical form", "Output",FontFamily->"Times-Plain",FontSize->10];
Print[TableForm[{"x' = - \[Beta] y + P(x, y)", "y' = + \[Beta] x + Q(x, y)"}, TableDepth -> 2,TableSpacing->2]];
StylePrint["where \[Beta] is a symbol, P and Q are symbolic functions at least of the order of \!\(x\^2\) + \!\(y\^2\).", "Output", FontFamily -> "Times-Plain", FontSize -> 10];
StylePrint["The command line writes","Output",FontFamily->"Times-Plain",FontSize->10];
StylePrint["CriticalEqS[sys, unk, order]","Output",FontFamily->"Times-Plain",FontSize->10];
]

HelpCriticalEqS[]:=
Module[{},
StylePrint["How to use the program CriticalEqS","Output",FontFamily->"Times-Bold",FontSize->12];
StylePrint["The program CriticalEqS applies to systems of two differential equations with symbolic coefficients in the canonical form","Output",FontFamily->"Times-Plain",FontSize->10];
Print[TableForm[{"x' = - \[Beta] y + P(x, y)","y' = + \[Beta] x + Q(x, y)"},TableDepth->2]];
StylePrint["where P and Q are functions at least of order two with respect their variables.","Output",FontFamily->"Times-Plain",FontSize->10];
StylePrint["To make active the program type in the following input data","Output",FontFamily->"Times-Plain",FontSize->10];
StylePrint["sys = list of the equations (f.e.{x' == - a y, y' == a x - \!\(x\^3\)};)","Output",FontFamily->"Times-Plain",FontSize->10];
StylePrint["unk = list of the unknowns (f.e.{x, y};)","Output",FontFamily->"Times-Plain",FontSize->10];
StylePrint["order = the order of the constant G[n] evaluated by the program (f.e. 10;)","Output",FontFamily->"Times-Plain",FontSize->10];
StylePrint["CriticalEqS[sys, unk, order]","Output",FontFamily->"Times-Plain",FontSize->10];
]


CriticalEqS1[sys_, unk_,order_] :=
Module[{P,Q,A, dp, p, dq, q, f, gradf, S, W, g,b, c, cc, sost, eq,sol, matc,r1, ns,m,j,pr, seven, t1seven},
Off[Solve::svars];
ffin=sys[[1,2]];
gfin=sys[[2,2]];
jfin={{D[ffin,unk[[1]]],D[ffin,unk[[2]]]},{D[gfin,unk[[1]]],D[gfin,unk[[2]]]}}/.{unk[[1]]->0,unk[[2]]->0};
P=sys[[1,2]]-jfin[[1]].unk;
Q=sys[[2,2]]-jfin[[2]].unk;
A = {{0,jfin[[1,2]]},{jfin[[2,1]],0}};

If[P===0&&Q===0,StylePrint["The method is not applicable since the system is linear.",
                      "Output",FontFamily->"Times-Plain",FontSize->12];Goto[a1]];

(*Terms of mth order of Taylor's expansion of P and Q at (0,0)*)
dp[m_,i_]:= D[P,{unk[[1]],m-i},{unk[[2]],i}]/.{unk[[1]]->0,unk[[2]]->0};
p[m_]:=Sum[(1/m!)*Binomial[m,i]*dp[m, i]*(unk[[1]]^(m - i))*unk[[2]]^i, {i,0,m}];
dq[m_,i_]:=D[Q,{unk[[1]],m-i},{unk[[2]],i}]/.{unk[[1]] -> 0, unk[[2]] -> 0};
q[m_]:= Sum[(1/m!)*Binomial[m, i]*dq[m, i]*(unk[[1]]^(m - i))*unk[[2]]^i,{i,0, m}];

(*Auxiliary polinomials*)
f[m_]:=Sum[b[m-i,i]*(unk[[1]]^(m-i))*unk[[2]]^i,{i,0,m}];

(*Relations gradf[m].A.unk=-2{p[m-1],q[m-1]}.unk-Sum[gradf[i+1].{p[i],q[i]}, {i,2,m-2}]*)

gradf[m_]:={D[f[m],unk[[1]]],D[f[m],unk[[2]]]};
S[m_]:=ExpandAll[Sum[gradf[m+1-i].{p[i],q[i]},{i,2,m-2}]];
W[m_]:=ExpandAll[-2*{p[m-1],q[m-1]}.{unk[[1]],unk[[2]]}-S[m]];
g[m_]:=ExpandAll[G[m]*(unk[[1]]^2+unk[[2]]^2)^(m/2)];
c[m_]:=ExpandAll[gradf[m].A.{unk[[1]],unk[[2]]}];
b[m_]:=Table[Coefficient[f[m],(unk[[1]]^(m-i))*unk[[2]]^i],{i,0,m}];
cc[m_]:=Table[Coefficient[c[m],(unk[[1]]^(m-i))*unk[[2]]^i],{i,0,m}];
k=4;
Label[0];

If[k===order,Print[StyleForm["Up to the order n = ","Output",FontFamily->"Times-Plain",FontSize->12],order];Print["G[",order,"] = 0 "];Goto[5]];
Do[
sost[m]=Flatten[Table[sol[i],
{i,3,m-1}]];

eq[m]=If[OddQ[m]==True,c[m]-(W[m]/.sost[m])==0,c[m]==(W[m]/.sost[m])+g[m]];
coeffsys[m]=If[OddQ[m]==True,sysodd[m]=Table[Coefficient[eq[m][[1]],
            (unk[[1]]^(m-i))*unk[[2]]^i]==0,{i,0,m}],
            syseven[m]=Table[Coefficient[eq[m][[1]],
            (unk[[1]]^(m-i))*unk[[2]]^i]==Coefficient[eq[m][[2]],
            (unk[[1]]^(m-i))*unk[[2]]^i],{i,0,m}]];

If[OddQ[m]==False,matc[m]=Table[Coefficient[coeffsys[m][[i+1,1]],b[m-j,j]],{i,0,m},{j,0,m}];
r1[m]=Table[coeffsys[m][[i,2]],{i,1,Length[matc[m]]}];
ns[m]=NullSpace[Transpose[matc[m]]];
valG[m]=Solve[ns[m].r1[m]==0,G[m]];

If[valG[m][[1,1,2]]===0,Goto[3],Goto[2]];
Label[3];
r2[m]=Flatten[coeffsys[m]/.valG[m]];
seven[m]=Solve[r2[m],b[m]]//Flatten;
t1seven[m]=Table[seven[m][[i,2]],{i,1,Length[seven[m]]}];
pr[m]=Cases[t1seven[m],xv_+yv_][[1,2]];
s[m]=Solve[r2[m]/.pr[m]->0,b[m]]//Flatten;
soleven[m]=Join[s[m],{pr[m]->0}]];

sol[m]=If[OddQ[m]==True,If[m==3,Solve[coeffsys[m],b[m]]//Flatten,
          Solve[coeffsys[m],b[m]]//Flatten],soleven[m]],{m,3,k}];
k=k+2;
Goto[0];
Label[2];
Print["G[",k,"] = ",valG[k][[1,1,2]]];
If[valG[k][[1,1,2]]<0,StylePrint["The origin is asymptotically stable.","Output",FontFamily->"Times-Plain",FontSize->12],
StylePrint["The origin is unstable.","Output",FontFamily->"Times-Plain",FontSize->12]];
Label[a1];
Label[5];
]


CriticalEqS2[sys_,unk_,order_]:=
Module[{},
sys1=sys/.unk[[1]]->ninc;
unk1=unk/.unk[[1]]->ninc;
fu=sys1[[1,2]];
gu=sys1[[2,2]];
If[(fu/.{unk1[[1]]->0,unk1[[2]]->0})==0&&(gu/.{unk1[[1]]->0,unk1[[2]]->0})==0,
    Goto[u1],StylePrint["The origin is not an equilibrium point.","Output",FontFamily->
                        "Times-Plain",FontSize->12];Goto[u2]];

Label[u1];
ju={{D[fu,ninc],D[fu,unk[[2]]]},{D[gu,ninc],D[gu,unk[[2]]]}}/.{ninc->0,unk[[2]]->0};

If[ju[[1,1]]===0&&ju[[2,2]]===0&&ju[[1,2]]===-ju[[2,1]],
      CriticalEqS1[sys1,unk1,order],StylePrint["The system is not given in the canonical form.","Output",FontFamily->"Times-Plain",FontSize->12]];

Label[u2];
]


CriticalEqS[sys_,unk_,order_]:=
Module[{},
If[unk[[1]]===Global`u,CriticalEqS2[sys,unk,order];Goto[b2],Label[b1]];

Label[b1];
fin=sys[[1,2]];
gin=sys[[2,2]];
If[(fin/.{unk[[1]]->0,unk[[2]]->0})!=0||(gin/.{unk[[1]]->0,unk[[2]]->0})!=0,
   StylePrint["The origin is not an equilibrium point.","Output",FontFamily->"Times-Plain",
              FontSize->12];Goto[b2]];

jin={{D[fin,unk[[1]]],D[fin,unk[[2]]]},{D[gin,unk[[1]]],D[gin,unk[[2]]]}}/.{unk[[1]]->0,unk[[2]]->0};

If[jin[[1,1]]===0&&jin[[2,2]]===0&&jin[[1,2]]===-jin[[2,1]],
   CriticalEqS1[sys,unk,order],StylePrint["The system is not given in the canonical form.","Output",FontFamily->"Times-Plain",FontSize->12]];
Label[b2];
]


(*
 --------------------------------------------------------------------------------------------
 -
 -   T H E   P R O G R A M  CManifold:
 -
 -                 UsageCManifold[]
 -                 HelpCManifold[]
 -                 CManifold[sys_,unk_,r_]
 -
 --------------------------------------------------------------------------------------------
*)


UsageCManifold[]:=
Module[{},
StylePrint["Aims of the program CManifold","Output",FontFamily->"Times-Bold",FontSize->12];
StylePrint["The method of center manifold is applied to study the equilibrium stability in the particular critical case described by the following system","Output", FontFamily -> "Times-Plain",FontSize -> 10];
Print[DisplayForm[\(\(x\^,\)\_i\)], " = ",DisplayForm[UnderoverscriptBox["\[Sum]","j=1","m"]],DisplayForm[\(a\_ij\)],DisplayForm[\(x\_j\)], " + ",DisplayForm[\(f\_i\)], "(", "x, y), ","  i = 1,..., n"];
Print[DisplayForm[\(\(y\^,\)\_\[Alpha]\)]," = ", DisplayForm[UnderoverscriptBox["\[Sum]","\[Alpha]=1","n"]],DisplayForm[\(b\_\[Alpha]\[Beta]\)],DisplayForm[\(y\_\[Beta]\)], " + ",DisplayForm[\(g\_\[Alpha]\)], "(","x, y), ", "  \[Beta] = 1,..., m"];
StylePrint["where all the eigenvalues of A = ( \!\(a\_ij\) ) have zero real parts, all the eigenvalues of B = ( \!\(b\_\[Alpha]\[Beta]\) ) have negative real parts, f =  ( \!\(f\_i\) ) and   g =  ( \!\(g\_\[Alpha]\) ) are functions at least of the second order of their variables in a neighborhood of the origin.", "Output", FontFamily -> "Times-Plain", FontSize -> 10];
StylePrint["The command line writes","Output",FontFamily->"Times-Plain",FontSize->10];
StylePrint["CManifold[sys, unk, r]","Output",FontFamily->"Times-Plain",FontSize->10];
]

HelpCManifold[]:=
Module[{},
StylePrint["How to use the program CManifold","Output",FontFamily->"Times-Bold",FontSize->12];
StylePrint["The program determines the center manifold in the critical case corresponding to eigenvalues of the Jacobian matrix having vanishing or negative real parts.","Output",FontFamily->"Times-Plain",FontSize->10];
StylePrint["To make active the program type","Output",FontFamily->"Times-Plain",FontSize->10];
StylePrint["sys = the system to study (f.e. {{x' == - x + x y}, {y' == z - y z, z' == - y + \!\(y\^2\)}}) ","Output",FontFamily->"Times-Plain",FontSize->10];
StylePrint["unk = the unknowns (f.e. {{x}, {y, z}}) ","Output",FontFamily->"Times-Plain",FontSize->10];
StylePrint["r = the approximation order ","Output",FontFamily->"Times-Plain",FontSize->10];
StylePrint["CManifold[sys, unk, r]","Output",FontFamily->"Times-Plain",FontSize->10];
]


CManifold[sys_,unk_,r_]:=
Module[{a,n,m,sost,J1,J2,P,Q,eig1,zeroeig1,negeig1,ind,v21,v2,c,g,h,l,co,prod},

nc=Length[sys[[1]]];
mc=Length[sys[[2]]];
sostc1=Table[unk[[1,i]]->0,{i,1,nc}];
sostc2=Table[unk[[2,i]]->0,{i,1,mc}];
sostc0=Join[sostc1,sostc2]//Flatten;
J1=Table[D[sys[[1,i,2]],unk[[1,j]]],{i,1,nc},{j,1,nc}]/.sostc0;
J2=Table[D[sys[[2,i,2]],unk[[2,j]]],{i,1,mc},{j,1,mc}]/.sostc0;
P[i_]:=sys[[1,i,2]]-J1[[i]].unk[[1]];
Q[i_]:=sys[[2,i,2]]-J2[[i]].unk[[2]];
eig1=Eigenvalues[J1];
zeroeig1=Select[eig1,Re[#1]==0&];
negeig1=Select[eig1,Re[#1]<0&];
eig2=Eigenvalues[J2];

StylePrint["Eigenvalues of the linear part are", "Output", FontFamily -> "Times-Plain", FontSize -> 12];
Print[eig1,", ",eig2];

zeroeig2=Select[eig2,Re[#1]==0&];
negeig2=Select[eig2,Re[#1]<0&];

If[(Length[zeroeig1]==nc&&Length[negeig2]==mc)||(Length[negeig1]==nc&&Length[zeroeig2]==mc),
     Goto[1],
     StylePrint["The eigenvalues do not verify the hypotheses of the theorem ", "Output", FontFamily -> "Times-Plain", FontSize -> 12];
     Goto[3]];

Label[1];

If[Length[zeroeig1]==nc&&Length[negeig2]==mc,Goto[2a],Goto[2b]];

Label[2a];
ind=Join[Table[in[i],{i,1,nc}],{0}];
v21=Table[{ind[[i]]-ind[[i+1]]},{i,1,nc}]//Flatten;
v2=Table[{ind[[i]],0,c[i]},{i,1,nc}];
c[1]=r;
c[i_]:=ind[[i-1]]/;i>1;
g[w1_,{z1__}]:=Flatten[Table[w1,z1],nc-1];
h=g[v21,v2];
co[va1_,{q__}]:=au[va1,q];
prod[i_,k_]:=Product[unk[[1,j]]^h[[k,j]],{j,1,nc}];
prodfi[i_,k_]:=co[i,h[[k]]]*prod[i,k];
yc[alp1_]:=Sum[prodfi[alp1,k],{k,nc+2,Length[h]}];
jacy[alha1_,j_]:=D[yc[alha1],unk[[1,j]]];
sostc=Table[unk[[2,ala]]->yc[ala],{ala,1,mc}];
eqc1[ala_]:=(sys[[2,ala]][[2]]-Sum[jacy[ala,i]*sys[[1,i]][[2]],{i,1,nc}])/.sostc;

Do[
TaylorCM[eqc1[vap],unk[[1]],Table[0,{nc}],r];
eqcfi[vap]=tay,
{vap,1,mc}];

coff[i_,j_]:=yc[i][[j,Length[yc[i][[j]]]]];
coff1[i_,j_]:=eqcfi[i][[j,Length[eqcfi[i][[j]]]]];
nunk=Table[coff[i,j],{i,1,mc},{j,1,Length[yc[i]]}]//Flatten;
sy=Table[coff1[i,j]==0,{i,1,mc},{j,1,Length[eqcfi[i]]}]//Flatten;
solc=Solve[sy,nunk]//Flatten;

StylePrint["The parametric equations of the center manifold are", "Output", FontFamily -> "Times-Plain", FontSize -> 12];

Do[
yfi[i]=yc[i]/.solc;
Print[unk[[2,i]]," = ",yfi[i]],
{i,1,mc}];

(*The reduced system on the center manifold*)
sostred=Table[unk[[2]][[i]]->yfi[i],{i,1,mc}]//Flatten;
sysred[i_]:=sys[[1]][[i,2]]/.sostred;

StylePrint["The reduced system is", "Output", FontFamily -> "Times-Plain", FontSize -> 12];

Do[
Print[unk[[1]][[i]],"' = ",sysred[i]],
{i,1,nc}];

Label[2b];
Label[3];
Clear[tay];
]


(*
 --------------------------------------------------------------------------------------------
 -
 -   T H E   P R O G R A M  Bif1:
 -
 -                 UsageBif1[]
 -                 HelpBif1[]
 -                 Bif1[eq_, unk_, par_]
 -
 --------------------------------------------------------------------------------------------
*)


UsageBif1[]:=
Module[{},
StylePrint["Aims of the program Bif1","Output",FontFamily->"Times-Bold",FontSize->12];
StylePrint["The program determines and studies the stability of the equilibrium positions of the equation ","Output",FontFamily->"Times-Plain",FontSize->10];
Print[DisplayForm[SuperscriptBox["x", ","]]," = ", DisplayForm["F"],"(", DisplayForm["x"], ", \[Lambda]), "];
StylePrint["where", "Output",FontFamily->"Times-Plain",FontSize->10];
Print[DisplayForm["F"],"(",DisplayForm["0, 0) = 0,  "],DisplayForm[SubscriptBox["F", "\[Lambda]"]],"(",DisplayForm["0, 0) \[NotEqual] 0,"]];
StylePrint["on varying the parameter \[Lambda], in the following cases","Output",FontFamily->"Times-Plain",FontSize->10];
Print["1) ",DisplayForm[SubscriptBox["F", "x"]],"(",DisplayForm["0, 0) \[NotEqual] 0;"]];
Print["2) ",DisplayForm[SubscriptBox["F", "x"]],"(",DisplayForm["0, 0) = 0,  "],DisplayForm[SubscriptBox["F", "xx"]],"(",DisplayForm["0, 0) \[NotEqual] 0;"]];
Print["3) ",DisplayForm[SubscriptBox["F", "x"]],"(",DisplayForm["0, 0) = "]DisplayForm[SubscriptBox["F", "xx"]],"(",DisplayForm["0, 0) = 0,  "],DisplayForm[SubscriptBox["F", "xxx"]],"(",DisplayForm["0, 0) \[NotEqual] 0."]];
StylePrint["The command line writes ","Output",FontFamily->"Times-Plain",FontSize->10];
StylePrint["Bif1[eq, unk, par]","Output",FontFamily->"Times-Plain",FontSize->10];
StylePrint["where eq is a first order differential equation in the unknown unk and par is a parameter. ","Output",FontFamily->"Times-Plain",FontSize->10];
]

HelpBif1[]:=
Module[{},
StylePrint["How to use the program Bif1","Output",FontFamily->"Times-Bold",FontSize->12];
StylePrint["The program studies the bifurcation of the origin of an equation containing one parameter.","Output",FontFamily->"Times-Plain",FontSize->10];
StylePrint["To make active the program type in the following input data","Output",FontFamily->"Times-Plain",FontSize->10];
StylePrint["eq = the equation (f.e. x' == x + \[Lambda] \!\(x\^2\); )","Output",FontFamily->"Times-Plain",FontSize->10];
StylePrint["unk = the unknown (f.e. x;)","Output",FontFamily->"Times-Plain",FontSize->10];
StylePrint["par = the parameter (f.e. \[Lambda];)","Output",FontFamily->"Times-Plain",FontSize->10];
StylePrint["Bif1[eq, unk, par]","Output",FontFamily->"Times-Plain",FontSize->10];
]


Bif1[eq_, unk_, par_] :=
  Module[{F, de, x1, s1, x2, lambda, x3},
   F = eq[[2]];
    de1unk = D[F, unk] /. {unk -> 0, par -> 0};
    de2unk = D[F, {unk, 2}] /. {unk -> 0, par -> 0};
    de3unk = D[F, {unk, 3}] /. {unk -> 0, par -> 0};
    de1par = D[F, par] /. {unk -> 0, par -> 0};
    de2par = D[F, {par, 2}] /. {unk -> 0, par -> 0};
    de3par = D[F, {par, 3}] /. {unk -> 0, par -> 0};
    deunkpar = D[F, unk, par] /. {unk -> 0, par -> 0};

(*Test on the value of F(unk, par) at the origin*)

If[(F /. {unk -> 0, par -> 0}) != 0,
     Print[StyleForm["The analysis can not be carried out because the rigth-hand side of equation does not vanish at the origin.",
             "Output", FontFamily -> "Times-Plain", FontSize -> 12]]; Goto[20], Goto[0]];

Label[0];      (*F(0,0)=0*)

(*Test on the derivative value of F(unk,par) with respect unk and par at the origin*)
If[de1unk == 0 && de1par == 0, Goto[12], Goto[1]];
Label[1];

If[de1unk != 0, Goto[2], Goto[5]];
Label[2];

x1 = -((de1par*par)/de1unk);

If[de1unk < 0, Goto[3], Goto[4]];
Label[3];

Print[StyleForm["For any value of  ", "Output", FontFamily -> "Times-Plain",FontSize -> 12],
      par, StyleForm[" near 0 there is only one stable equilibrium configuration:",
      "Output", FontFamily -> "Times-Plain", FontSize -> 12]];
Print[unk[par], " = ", x1];
Goto[20];

Label[4];

Print[StyleForm["For any value of ", "Output", FontFamily -> "Times-Plain", FontSize -> 12],
      par, StyleForm[" near 0 there is only one unstable equilibrium configuration:",
      "Output", FontFamily -> "Times-Plain", FontSize -> 12]];
Print[unk[par], "=", x1];
Goto[20];
Label[5];

If[de2unk != 0, Goto[6], Goto[9]];

Label[6];
s1 = Sqrt[-((2*de1par*par)/de2unk)];
x1 = -s1;
x2 = s1;

If[-(de1par/de2unk) > 0, Goto[7], Goto[8]];
Label[7];

Print[StyleForm["For any value of ", "Output", FontFamily -> "Times-Plain", FontSize -> 12],
      par, StyleForm["\[GreaterEqual] 0 near 0 there are two equilibrium configurations  that coincide for ",
      "Output", FontFamily -> "Times-Plain", FontSize -> 12], par, "= 0:"];
Print[unk, "e1 = ", x1];
Print[unk, "e2 = ", x2];

If[
de2unk < 0, Print[StyleForm["where ", "Output", FontFamily -> "Times-Plain", FontSize -> 12],
                  unk, StyleForm["e1 is unstable and ", "Output", FontFamily -> "Times-Plain",
                  FontSize -> 12], unk, StyleForm["e2 is stable.", "Output",
                  FontFamily -> "Times-Plain", FontSize -> 12]],

            Print[StyleForm["where ", "Output", FontFamily -> "Times-Plain", FontSize -> 12],
                  unk, StyleForm["e1 is stable and ", "Output", FontFamily -> "Times-Plain",
                  FontSize -> 12], unk, StyleForm["e2 is unstable.", "Output",
                  FontFamily -> "Times-Plain", FontSize -> 12]]];
            Print[StyleForm["If ", "Output", FontFamily -> "Times-Plain", FontSize -> 12],
            par, StyleForm["<0 near 0, there is no equilibrium position.", "Output",
            FontFamily -> "Times-Plain", FontSize -> 12]];

Goto[20];

Label[8];

Print[StyleForm["For any value of ", "Output", FontFamily -> "Times-Plain", FontSize -> 12],
      par, StyleForm["\[LessEqual] 0 near 0 there are two equilibrium configurations which are equal for ",
      "Output", FontFamily -> "Times-Plain", FontSize -> 12], par, StyleForm["= 0:", "Output",
      FontFamily -> "Times-Plain", FontSize -> 12]];

Print[unk, "e1 = ", x1];
Print[unk, "e2 = ", x2];

If[
de2unk < 0, Print[StyleForm["where ", "Output", FontFamily -> "Times-Plain", FontSize -> 12],
                  unk, StyleForm["e1 is unstable and ", "Output",
                  FontFamily -> "Times-Plain", FontSize -> 12], unk,
                  StyleForm["e2 is stable.", "Output", FontFamily -> "Times-Plain",
                  FontSize -> 12]],

            Print[StyleForm["where ", "Output", FontFamily -> "Times-Plain", FontSize -> 12],
                  unk, StyleForm["e1 is stable and ", "Output", FontFamily -> "Times-Plain",
                  FontSize -> 12], unk, StyleForm["e2 is unstable.", "Output",
                  FontFamily -> "Times-Plain", FontSize -> 12]]];
            Print[StyleForm["If ", "Output", FontFamily -> "Times-Plain", FontSize -> 12],
                  par, StyleForm[" > 0 near 0, there is no equilibrium position.", "Output",
                  FontFamily -> "Times-Plain", FontSize -> 12]];

Goto[20];
Label[9];

If[de3unk != 0, Goto[10], Goto[11]];

Label[10];
x1 = (-((6*de1par*par)/de3unk))^(1/3);
Print[StyleForm["For any value of ", "Output", FontFamily -> "Times-Plain", FontSize -> 12],
      par, StyleForm[" near 0 there is one equilibrium configuration:", "Output",
      FontFamily -> "Times-Plain", FontSize -> 12]];
Print[unk, "e = ", x1];

If[
de3unk < 0, Print[StyleForm[" which is stable.", "Output", FontFamily -> "Times-Plain",
                  FontSize -> 12]]; Goto[20],

            Print[StyleForm[" which is unstable.", "Output", FontFamily -> "Times-Plain",
            FontSize -> 12]]; Goto[20]];

Label[11];
Print[StyleForm["The program is not applicable.", "Output", FontFamily -> "Times-Plain",
      FontSize -> 12]]; Goto[20];

Label[12];
Print[StyleForm["Critical case.", "Output", FontFamily -> "Times-Plain", FontSize -> 12]];

If[de2par != 0, Goto[14], Goto[19]];
Label[14];

If[de2unk != 0, Goto[15], Goto[16]];

Label[15];
lambda1 = (-deunkpar - Sqrt[deunkpar^2 - de2unk*de2par])/ de2par;
lambda2 = (-deunkpar + Sqrt[deunkpar^2 - de2unk*de2par])/de2par;

x1 = par/lambda2;
x2 = par/lambda1;

Print[StyleForm["For any value of ", "Output", FontFamily -> "Times-Plain", FontSize -> 12],
      par, StyleForm[" near 0, there are two equilibrium positions:", "Output",
      FontFamily -> "Times-Plain", FontSize -> 12]];

Print[unk, "e1 = ", x1];
Print[unk, "e2 = ", x2];

If[
de2unk > 0, Print[StyleForm["where ", "Output", FontFamily -> "Times-Plain", FontSize -> 12],
                  unk, StyleForm["e1 is stable and ", "Output", FontFamily -> "Times-Plain",
                  FontSize -> 12], unk, StyleForm["e2 is unstable.", "Output",
                  FontFamily -> "Times-Plain", FontSize -> 12]]; Goto[20],

            Print[StyleForm["where ", "Output", FontFamily -> "Times-Plain", FontSize -> 12],
                  unk, StyleForm["e1 is unstable and ", "Output", FontFamily -> "Times-Plain",
                  FontSize -> 12], unk, StyleForm["e2 is stable.", "Output",
                  FontFamily -> "Times-Plain", FontSize -> 12]]; Goto[20]];

Label[16];

If[de3unk != 0 && deunkpar != 0, Goto[17], Goto[19]];

Label[17];
 x1 = -2*Sqrt[-((3*deunkpar*par)/de3unk)];
 x2 = -((de2par*par)/(2*deunkpar));
 x3 = 2*Sqrt[-((3*deunkpar*par)/de3unk)];

If[de3unk < 0, Goto[18], Goto[18*b]];

Label[18];

If[
-((3*deunkpar)/de3unk) > 0, Print[StyleForm["There are, for ", "Output",
                                  FontFamily -> "Times-Plain", FontSize -> 12], par,
                                  StyleForm[">0 near 0, the following equilibrium positions:",
                                  "Output", FontFamily -> "Times-Plain", FontSize -> 12]];
                            Print[unk, "e1 = ", x1];
                            Print[unk, "e2 = ", x2];
                            Print[unk, "e3 = ", x3];
                            Print[StyleForm["where ", "Output", FontFamily -> "Times-Plain",
                                  FontSize -> 12], unk, StyleForm["e1 is stable, ", "Output",
                                  FontFamily -> "Times-Plain", FontSize -> 12], unk,
                                  StyleForm["e2 is unstable and ", "Output",
                                  FontFamily -> "Times-Plain", FontSize -> 12], unk,
                                  StyleForm["e3 is stable.", "Output", FontFamily ->
                                  "Times-Plain", FontSize -> 12]];
                            Print[StyleForm["For ", "Output", FontFamily -> "Times-Plain",
                                  FontSize -> 12], par, StyleForm["<0 near 0, there is only one stable equilibrium position ",
                                  "Output", FontFamily -> "Times-Plain", FontSize -> 12], unk,
                                  "e2."],
                            Print[StyleForm["There are, for ", "Output",
                            FontFamily -> "Times-Plain", FontSize -> 12], par,
                            StyleForm["<0 near 0, the following equilibrium positions:", "Output",
                            FontFamily -> "Times-Plain", FontSize -> 12]];
                            Print[unk, "e1 = ", x1];
                            Print[unk, "e2 = ", x2];
                            Print[unk, "e3 = ", x3];
                            Print[StyleForm["where ", "Output", FontFamily -> "Times-Plain",
                                  FontSize -> 12], unk, StyleForm["e1 is stable, ", "Output",
                                  FontFamily -> "Times-Plain", FontSize -> 12], unk,
                                  StyleForm["e2 is unstable and ", "Output", FontFamily ->
                                  "Times-Plain", FontSize -> 12], unk, StyleForm["e3 is stable.",
                                  "Output", FontFamily -> "Times-Plain", FontSize -> 12]];
                            Print[StyleForm["For ", "Output", FontFamily -> "Times-Plain",
                                  FontSize -> 12], par,
                                  StyleForm[">0 near 0, there is only one stable equilibrium position ",
                                  "Output", FontFamily -> "Times-Plain", FontSize -> 12],
                                  unk, StyleForm["e2.", "Output", FontFamily -> "Times-Plain",
                                  FontSize -> 12]]];

Goto[20];

Label[18*b];

If[
-((3*deunkpar)/de3unk) > 0, Print[StyleForm["There are, for ", "Output",
                                  FontFamily -> "Times-Plain", FontSize -> 12], par,
                                  StyleForm[">0 near 0, the following equilibrium positions:",
                                  "Output", FontFamily -> "Times-Plain", FontSize -> 12]];
                            Print[unk, "e1 = ", x1];
                            Print[unk, "e2 = ", x2];
                            Print[unk, "e3 = ", x3];
                            Print[StyleForm["where ", "Output", FontFamily -> "Times-Plain",
                                  FontSize -> 12], unk, StyleForm["e1 is unstable, ",
                                  "Output", FontFamily -> "Times-Plain", FontSize -> 12], unk,
                                  StyleForm["e2 is stable and ", "Output", FontFamily ->
                                  "Times-Plain", FontSize -> 12], unk, StyleForm["e3 is unstable.",
                                  "Output", FontFamily -> "Times-Plain", FontSize -> 12]];
                            Print[StyleForm["For ", "Output", FontFamily -> "Times-Plain",
                                  FontSize -> 12], par, StyleForm["<0 near 0, there is only one unstable equilibrium position ",
                                  "Output", FontFamily -> "Times-Plain", FontSize -> 12],
                                  unk, StyleForm["e2.", "Output", FontFamily -> "Times-Plain",
                                  FontSize -> 12]],

                            Print[StyleForm["There are, for ", "Output", FontFamily ->
                                  "Times-Plain", FontSize -> 12], par,
                                  StyleForm["<0 near 0, the following equilibrium positions:",
                                  "Output", FontFamily -> "Times-Plain", FontSize -> 12]];
                            Print[unk, "e1 = ", x1];
                            Print[unk, "e2 = ", x2];
                            Print[unk, "e3 = ", x3];
                            Print[StyleForm["where ", "Output", FontFamily -> "Times-Plain",
                                  FontSize -> 12], unk, StyleForm["e1 is stable, ", "Output",
                                  FontFamily -> "Times-Plain", FontSize -> 12], unk,
                                  StyleForm["e2 is unstable and ", "Output", FontFamily ->
                                  "Times-Plain", FontSize -> 12], unk, StyleForm["e3 is stable.",
                                  "Output", FontFamily -> "Times-Plain", FontSize -> 12]];
                            Print[StyleForm["For ", "Output", FontFamily -> "Times-Plain",
                                  FontSize -> 12], par, StyleForm[">0 near 0, there is only one unstable equilibrium position ",
                                  "Output", FontFamily -> "Times-Plain", FontSize -> 12],unk,
                                  StyleForm["e2.", "Output", FontFamily -> "Times-Plain",
                                  FontSize -> 12]]];

Goto[20];
Label[19];

Print[StyleForm["The program is not applicable.", "Output", FontFamily -> "Times-Plain",
      FontSize -> 12]];
Label[20];

]


(*
 --------------------------------------------------------------------------------------------
 -
 -   T H E   P R O G R A M  Bif1G:
 -
 -                 UsageBif1G[]
 -                 HelpBif1G[]
 -                 Bif1G[eq_, unk_, par_, a_, b_, points_]
 -
 --------------------------------------------------------------------------------------------
*)


UsageBif1G[]:=
Module[{},
StylePrint["Aims of the program Bif1G","Output",FontFamily->"Times-Bold",FontSize->12];
StylePrint["The program supplies a graphical analysis of the existence and stability properties of the equilibrium positions of the equation ","Output",FontFamily->"Times-Plain",FontSize->10];
Print[DisplayForm[SuperscriptBox["x", ","]]," = ", DisplayForm["F"],"(",DisplayForm["x"],", \[Lambda]), "];
StylePrint["on varying the parameter \[Lambda].","Output",FontFamily->"Times-Plain",FontSize->10];
StylePrint["The command line is ","Output",FontFamily->"Times-Plain",FontSize->10];
StylePrint["Bif1G[eq, unk, par, a, b, points]","Output",FontFamily->"Times-Plain",FontSize->10];
StylePrint["where eq is a first order differential equation in the unknown unk containing the parameter \[Lambda], 2a and 2b are the dimensions of the rectangle in which the curve of the equilibrium positions is represented, and finally points is a graphical option fixing the number of the points used to draw the curve F(x, \[Lambda]) = 0. ","Output",FontFamily->"Times-Plain",FontSize->10];
]

HelpBif1G[]:=
Module[{},
StylePrint["How to use the program Bif1G","Output",FontFamily->"Times-Bold",FontSize->12];
StylePrint["The program analyzes graphically the bifurcation of the origin.","Output",FontFamily->"Times-Plain",FontSize->10];
StylePrint["To make active the program type in the following input data","Output",FontFamily->"Times-Plain",FontSize->10];
StylePrint["eq = the equation (f.e. x' == x + \[Mu]\!\(x\^3\); )","Output",FontFamily->"Times-Plain",FontSize->10];
StylePrint["unk = the unknown (f.e. x;)","Output",FontFamily->"Times-Plain",FontSize->10];
StylePrint["par = the parameter (f.e. \[Mu];)","Output",FontFamily->"Times-Plain",FontSize->10];
StylePrint["a = the semi-length of the parameter interval (f.e. 1;)","Output",FontFamily->"Times-Plain",FontSize->10];
StylePrint["b = the semi-length of the unknown interval (f.e. 1;)","Output",FontFamily->"Times-Plain",FontSize->10];
StylePrint["points = a graphical option fixing the number of the points used to draw the curve F(x, \[Lambda]) = 0 (f.e. 20;)","Output",FontFamily->"Times-Plain",FontSize->10];
StylePrint["Bif1G[eq, unk, par, a, b, points]","Output",FontFamily->"Times-Plain",FontSize->10];
]


Bif1G[eq_, unk_, par_, a_, b_, points_] :=
Module[{F,g,h,m,pl,q,plo},

F= eq[[2]];
g[i_, j_] := F /. {par -> i, unk -> j};
h[i_, j_] :=If[g[i, j] < 0, {i, j}, {0, 0}];
m = Table[h[i, j], {i, -a, a,(2*a)/points}, {j, -b, b, (2*b)/points}];
m1 = Flatten[m, 1];
neg =DeleteCases[m1, {0, 0}]; pl[i_] := Graphics[Circle[neg[[i]],0.005],
     AspectRatio -> Automatic,Axes -> Automatic,
     AxesLabel ->{StyleForm[par,FontSlant->"Italic"],
     StyleForm[unk,FontSlant->"Italic"]}];
q = Table[pl[i], {i, 1, Length[neg]}];
plo = ImplicitPlot[F == 0,{par, -a, a}, {unk, -b, b},
      PlotPoints -> 200, DisplayFunction -> Identity];
StylePrint["In the dashed region the right-hand side of the equation is negative.","Output",FontFamily->"Times-Plain",FontSize->12];
Show[q, plo, DisplayFunction -> $DisplayFunction];
]


(*
 --------------------------------------------------------------------------------------------
 -
 -   T H E   P R O G R A M  Bif2:
 -
 -                 UsageBif2[]
 -                 HelpBif2[]
 -                 Bif2[eq_, unk_, par_]
 -
 --------------------------------------------------------------------------------------------
*)


UsageBif2[]:=
Module[{},
StylePrint["Aims of the program Bif2","Output",FontFamily->"Times-Bold",FontSize->12];
StylePrint["The program determines the stability properties of the equilibrium positions of the equation ","Output",FontFamily->"Times-Plain",FontSize->10];
Print[DisplayForm[SuperscriptBox["x", ","]]," = ", DisplayForm["F"],"(", DisplayForm["x"], ", \[Mu], \[Lambda]), "];
StylePrint["containing the parameters \[Mu] and \[Lambda], where", "Output",FontFamily->"Times-Plain",FontSize->10];
Print[DisplayForm["F"],"(",DisplayForm["0, 0, 0) = 0,  "], DisplayForm[SubscriptBox["F", "\[Lambda]"]],"(", DisplayForm["0, 0, 0) \[NotEqual] 0,"]];
StylePrint["in the following cases","Output",FontFamily->"Times-Plain",FontSize->10];
Print["1) ",DisplayForm[SubscriptBox["F", "x"]],"(", DisplayForm["0, 0, 0) \[NotEqual] 0;"]];
Print["2) ",DisplayForm[SubscriptBox["F", "x"]],"(", DisplayForm["0, 0, 0) = 0,  "],DisplayForm[SubscriptBox["F", "xx"]],"(", DisplayForm["0, 0, 0) \[NotEqual] 0;"]];
Print["3) ",DisplayForm[SubscriptBox["F", "x"]],"(", DisplayForm["0, 0, 0) = "]DisplayForm[SubscriptBox["F", "xx"]],"(", DisplayForm["0, 0, 0) = 0,  "],DisplayForm[SubscriptBox["F", "xxx"]],"(", DisplayForm["0, 0, 0) \[NotEqual] 0."]];
StylePrint["The command line is ","Output",FontFamily->"Times-Plain",FontSize->10];
StylePrint["Bif2[eq, unk, par]","Output",FontFamily->"Times-Plain",FontSize->10];
StylePrint["where eq is a first order differential equation in the unknown unk and the parameters par. ","Output",FontFamily->"Times-Plain",FontSize->10];
]

HelpBif2[]:=
Module[{},
StylePrint["How to use the program Bif2","Output",FontFamily->"Times-Bold",FontSize->12];
StylePrint["The program studies the origin bifurcation of an equation containing two parameters","Output",FontFamily->"Times-Plain",FontSize->10];
StylePrint["To make active the program type in the following input data","Output",FontFamily->"Times-Plain",FontSize->10];
StylePrint["eq = the equation (f.e. x' == x + \[Mu] \!\(x\^2\) + \[Rho] \!\(x\^3\);)","Output",FontFamily->"Times-Plain",FontSize->10];
StylePrint["unk = the unknown (f.e. x;)","Output",FontFamily->"Times-Plain",FontSize->10];
StylePrint["par = the parameters (f.e. {\[Mu], \[Rho]};)","Output",FontFamily->"Times-Plain",FontSize->10];
StylePrint["Bif2[eq, unk, par]","Output",FontFamily->"Times-Plain",FontSize->10];
]


Bif2[eq_, unk_, par_] :=
Module[{},
  FA = eq[[2]];
  de1par1 = D[FA, par[[1]]] /. {unk -> 0, par[[1]] -> 0, par[[2]] -> 0};
  Which[(FA /. {unk -> 0, par[[1]] -> 0, par[[2]] -> 0}) != 0,
    Print[StyleForm["The origin is not an equilibrium point.", "Output",
    FontFamily -> "Times-Plain", FontSize -> 12]];
    Goto[20], (FA /. {unk -> 0, par[[1]] -> 0, par[[2]] -> 0}) == 0 && de1par1 == 0,
    Print[StyleForm["The program is not applicable; try again exchanging the parameter order ",
    "Output", FontFamily -> "Times-Plain", FontSize -> 12], par,
    StyleForm[" into ", "Output", FontFamily -> "Times-Plain", FontSize -> 12],
    "{", par[[2]], ",", par[[1]], "}."]; Goto[20],
    (F /. {unk -> 0, par[[1]] -> 0, par[[2]] -> 0}) == 0 && de1par1 != 0, Goto[0]];
    Label[0];
    de1unk = D[FA, unk] /. {unk -> 0, par[[1]] -> 0, par[[2]] -> 0};
    de2unk = D[FA, {unk, 2}] /. {unk -> 0, par[[1]] -> 0, par[[2]] -> 0};
    de3unk = D[FA, {unk, 3}] /. {unk -> 0, par[[1]] -> 0, par[[2]] -> 0};
    de2par1 = D[FA, {par[[1]], 2}] /. {unk -> 0, par[[1]] -> 0, par[[2]] -> 0};
    de1par2 = D[FA, par[[2]]] /. {unk -> 0, par[[1]] -> 0, par[[2]] -> 0};
    de2par2 = D[FA, {par[[2]], 2}] /. {unk -> 0, par[[1]] -> 0, par[[2]] -> 0};
    deunkpar1 = D[FA, par[[1]], unk] /. {unk -> 0, par[[1]] -> 0, par[[2]] -> 0};
    deunkpar2 = D[FA, par[[2]], unk] /. {unk -> 0, par[[1]] -> 0, par[[2]] -> 0};
    depar1par2 = D[FA, par[[1]], par[[2]]] /. {unk -> 0, par[[1]] -> 0, par[[2]] -> 0};

    If[de1unk != 0, Goto[1], Goto[2]];
    Label[1];
    xe = -(de1par1*par[[1]] + de1par2*par[[2]])/de1unk;

    If[de1unk < 0, Print[StyleForm["For any value of ", "Output", FontFamily ->
    "Times-Plain", FontSize -> 12], par[[1]], ", ", par[[2]],
       StyleForm[" in a neighbourhood of the origin there is one stable equilibrium configuration:",
       "Output", FontFamily -> "Times-Plain", FontSize -> 12]];
      Print[unk, StyleForm["e(", "Output", FontFamily -> "Times-Plain",
        FontSize -> 12], par[[1]], ",", par[[2]], StyleForm[") = ", "Output",
        FontFamily -> "Times-Plain", FontSize -> 12], xe],
     Print[StyleForm["For any value of ", "Output", FontFamily -> "Times-Plain",
        FontSize -> 12], par[[1]], ", ", par[[2]], StyleForm[" in a neighbourhood of the origin there is one unstable equilibrium configuration:", "Output",
        FontFamily -> "Times-Plain", FontSize -> 12]];
      Print[unk, StyleForm["e(", "Output", FontFamily -> "Times-Plain",
        FontSize -> 12], par[[1]], ",", par[[2]], StyleForm[") = ", "Output",
        FontFamily -> "Times-Plain", FontSize -> 12], xe]]; Goto[20];
    Label[2];

    If[de2unk != 0, Goto[3], Goto[4]];
    Label[3];
    dlpar2 = -de1par2/de1par1;
    dlunkunk = -de2unk/de1par1;
    dlunkpar2 = -(deunkpar2 + deunkpar1*dlpar2)/de1par1;
    dlpar2par2 = -(de2par2 + 2*depar1par2*dlpar2 + de2par1*dlpar2^2)/de1par1;
    lim = -((dlunkpar2^2 - dlunkunk*dlpar2par2)*par[[2]]^2 - 2*dlpar2*dlunkunk*par[[2]])/(2*dlunkunk);
    eq2 = dlunkunk*unk^2 + 2*dlunkpar2*par[[2]]*unk + dlpar2par2*par[[2]]^2 + 2*dlpar2*par[[2]] - 2*par[[1]] == 0;
    xe = Flatten[Solve[eq2, unk]];

    If[de2unk > 0, Goto[3*a], Goto[3*b]];
    Label[3*a];

    If[dlunkunk > 0, Print[StyleForm["For any value of ", "Output", FontFamily -> "Times-Plain", FontSize -> 12], par[[1]], StyleForm[" > ", "Output", FontFamily -> "Times-Plain",
        FontSize -> 12], lim, StyleForm[" and ", "Output", FontFamily -> "Times-Plain",
        FontSize -> 12], par[[2]], StyleForm[" in a neighbourhood of the origin there are two equilibrium positions:",
        "Output", FontFamily -> "Times-Plain", FontSize -> 12]];
      Print[unk, StyleForm["e1 = ", "Output", FontFamily -> "Times-Plain",
        FontSize -> 12], xe[[1,2]], StyleForm[", which is stable;", "Output",
        FontFamily -> "Times-Plain",FontSize -> 12]];
      Print[unk, StyleForm["e2 = ", "Output", FontFamily -> "Times-Plain",
        FontSize -> 12], xe[[2,2]], StyleForm[", which is unstable.",
        "Output", FontFamily -> "Times-Plain", FontSize -> 12]]; Print[""];
      Print[StyleForm["If ", "Output", FontFamily -> "Times-Plain",
        FontSize -> 12], par[[1]], StyleForm[" = ", "Output", FontFamily -> "Times-Plain",
        FontSize -> 12], lim, StyleForm[" for any ", "Output", FontFamily -> "Times-Plain",
        FontSize -> 12], par[[2]], StyleForm[" in a neighbourhood of the origin there is one equilibrium position:",
        "Output", FontFamily -> "Times-Plain", FontSize -> 12]];
      Print[unk, StyleForm["e = ", "Output", FontFamily -> "Times-Plain", FontSize -> 12],
       xe[[1,2]] /. par[[1]] -> lim, StyleForm[", which is unstable.", "Output", FontFamily -> "Times-Plain",
        FontSize -> 12]];
        Print[""];
        Print[StyleForm["If ", "Output", FontFamily -> "Times-Plain", FontSize -> 12], par[[1]],
       StyleForm[" < ", "Output", FontFamily -> "Times-Plain", FontSize -> 12], lim,
       StyleForm[" there are no equilibrium positions.", "Output", FontFamily -> "Times-Plain",
        FontSize -> 12]], Print[StyleForm["For any value of ", "Output", FontFamily -> "Times-Plain",
        FontSize -> 12], par[[1]], StyleForm[" < ", "Output", FontFamily -> "Times-Plain",
        FontSize -> 12], lim, StyleForm[" and ", "Output", FontFamily -> "Times-Plain",
        FontSize -> 12], par[[2]], StyleForm[" in a neighbourhood of the origin there are two equilibrium positions:",
        "Output", FontFamily -> "Times-Plain", FontSize -> 12]];
      Print[unk, StyleForm["e1 = ", "Output", FontFamily -> "Times-Plain",
        FontSize -> 12], xe[[1,2]], StyleForm[", which is stable;", "Output",
        FontFamily -> "Times-Plain",FontSize -> 12]];
      Print[unk, StyleForm["e2 = ", "Output", FontFamily -> "Times-Plain",
        FontSize -> 12], xe[[2,2]], StyleForm[", which is unstable.",
        "Output", FontFamily -> "Times-Plain", FontSize -> 12]];
      Print[""];
      Print[StyleForm["If ", "Output", FontFamily -> "Times-Plain",
        FontSize -> 12], par[[1]], StyleForm[" = ", "Output", FontFamily -> "Times-Plain",
        FontSize -> 12], lim, StyleForm[" for any ", "Output",
        FontFamily -> "Times-Plain", FontSize -> 12], par[[2]],
        StyleForm[" in a neighbourhood of the origin there is one equilibrium position:",
        "Output", FontFamily -> "Times-Plain", FontSize -> 12]];
      Print[unk, StyleForm["e = ", "Output", FontFamily -> "Times-Plain", FontSize -> 12],
       xe[[1,2]] /. par[[1]] -> lim, StyleForm[", which is unstable.",
        "Output", FontFamily -> "Times-Plain", FontSize -> 12]]; Print[""];
      Print[StyleForm["If ", "Output", FontFamily -> "Times-Plain",
        FontSize -> 12], par[[1]], StyleForm[" > ", "Output", FontFamily -> "Times-Plain",
        FontSize -> 12], lim, StyleForm[" there are no equilibrium positions.", "Output",
        FontFamily -> "Times-Plain", FontSize -> 12]]];
      Goto[20];
      Label[3*b];

    If[dlunkunk > 0, Print[StyleForm["For any value of ", "Output",
      FontFamily -> "Times-Plain", FontSize -> 12], par[[1]],
      StyleForm[" > ", "Output", FontFamily -> "Times-Plain", FontSize -> 12], lim,
       StyleForm[" and ", "Output", FontFamily -> "Times-Plain", FontSize -> 12], par[[2]],
       StyleForm[" in a neighbourhood of the origin there are two equilibrium positions:",
        "Output", FontFamily -> "Times-Plain", FontSize -> 12]];
      Print[unk, StyleForm["e1 = ", "Output", FontFamily -> "Times-Plain",
        FontSize -> 12], xe[[1,2]], StyleForm[", which is unstable;",
        "Output", FontFamily -> "Times-Plain", FontSize -> 12]];
      Print[unk, StyleForm["e2 = ", "Output", FontFamily -> "Times-Plain",
        FontSize -> 12], xe[[2,2]], StyleForm[", which is stable.", "Output",
        FontFamily -> "Times-Plain", FontSize -> 12]]; Print[""];
      Print[StyleForm["If ", "Output", FontFamily -> "Times-Plain",
        FontSize -> 12], par[[1]], StyleForm[" = ", "Output", FontFamily -> "Times-Plain",
        FontSize -> 12], lim, StyleForm[" for any ", "Output", FontFamily -> "Times-Plain",
        FontSize -> 12], par[[2]], StyleForm[" in a neighbourhood of the origin there is one equilibrium position:",
        "Output", FontFamily -> "Times-Plain", FontSize -> 12]];
      Print[unk, StyleForm["e = ", "Output", FontFamily -> "Times-Plain", FontSize -> 12],
       xe[[1,2]] /. par[[1]] -> lim, StyleForm[", which is stable.", "Output",
        FontFamily -> "Times-Plain", FontSize -> 12]];
      Print[""];
      Print[StyleForm["If ", "Output", FontFamily -> "Times-Plain", FontSize -> 12], par[[1]],
       StyleForm[" < ", "Output", FontFamily -> "Times-Plain", FontSize -> 12], lim,
       StyleForm[" there are no equilibrium positions.", "Output", FontFamily -> "Times-Plain",
        FontSize -> 12]],
     Print[StyleForm["For any value of ", "Output", FontFamily -> "Times-Plain",
        FontSize -> 12], par[[1]], StyleForm[" < ", "Output",
        FontFamily -> "Times-Plain", FontSize -> 12], lim,
       StyleForm[" and ", "Output", FontFamily -> "Times-Plain", FontSize -> 12], par[[2]],
       StyleForm[" in a neighbourhood of the origin there are two equilibrium positions:",
        "Output", FontFamily -> "Times-Plain", FontSize -> 12]];
      Print[unk, StyleForm["e1 = ", "Output", FontFamily -> "Times-Plain",
        FontSize -> 12], xe[[1,2]], StyleForm[", which is unstable;", "Output", FontFamily -> "Times-Plain",
        FontSize -> 12]];
      Print[unk, StyleForm["e2 = ", "Output", FontFamily -> "Times-Plain", FontSize -> 12], xe[[2,2]],
       StyleForm[", which is stable.", "Output", FontFamily -> "Times-Plain",
        FontSize -> 12]];
      Print[""];
      Print[StyleForm["If ", "Output", FontFamily -> "Times-Plain", FontSize -> 12], par[[1]],
       StyleForm[" = ", "Output", FontFamily -> "Times-Plain", FontSize -> 12], lim,
       StyleForm[" for any ", "Output", FontFamily -> "Times-Plain", FontSize -> 12], par[[2]],
       StyleForm[" in a neighbourhood of the origin there is one equilibrium position:",
        "Output", FontFamily -> "Times-Plain", FontSize -> 12]];
      Print[unk, StyleForm["e = ", "Output", FontFamily -> "Times-Plain",
        FontSize -> 12], xe[[1,2]] /. par[[1]] -> lim, StyleForm[", which is stable.", "Output",
        FontFamily -> "Times-Plain", FontSize -> 12]];
      Print[""];
      Print[StyleForm["If ", "Output", FontFamily -> "Times-Plain", FontSize -> 12], par[[1]],
       StyleForm[" > ", "Output", FontFamily -> "Times-Plain", FontSize -> 12], lim,
       StyleForm[" there are no equilibrium positions.", "Output", FontFamily -> "Times-Plain",
        FontSize -> 12]]];
      Goto[20];
      Label[4];
      dlunkpar2 = -(deunkpar2 + deunkpar1*dlpar2)/de1par1;

    If[de3unk != 0, Goto[5], Print[StyleForm["The program is not applicable.", "Output",
        FontFamily -> "Times-Plain", FontSize -> 12]]; Goto[20]]; Label[5];
    Print[StyleForm["In this case we have that the first and second derivatives with  respect to the unknown of the right-hand side of the equation vanish at the origin, whereas the third derivative is different from zero.", "Output", FontFamily -> "Times-Plain",
      FontSize -> 12]];
Print[""];
    Print[StyleForm["Qualitative analysis of bifurcation.", "Output",
      FontFamily -> "Times-Plain", FontSize -> 12]];
Print[""];
    de3par1 = D[FA, {par[[1]], 3}] /. {unk -> 0, par[[1]] -> 0, par[[2]] -> 0};
    rad = (dlunkpar2*de1par1)^3/de3unk;
    l1 = (-de1par2*par[[2]] - Sqrt[(8*rad*par[[2]]^3)/9])/de1par1;
    l2 = (-de1par2*par[[2]] + Sqrt[(8*rad*par[[2]]^3)/9])/de1par1;
    l11 = If[de1par1 > 0, l1, l2];
    l22 = If[de1par1 > 0, l2, l1];
    Off[Plot::plnr];
    plotl1 = l1 /. par[[2]] -> mu;
    plotl2 = l2 /. par[[2]] -> mu;
    Plot[{plotl1, plotl2}, {mu, -1, 1}, AxesLabel -> {par[[2]], par[[1]]}, AxesOrigin -> {0, 0}];
    Which[rad > 0 && de3unk > 0, Goto[6], rad > 0 && de3unk < 0, Goto[7],
     rad < 0 && de3unk > 0, Goto[8], rad < 0 && de3unk < 0, Goto[9],
     rad == 0 && de3unk > 0, Goto[12], rad == 0 && de3unk < 0, Goto[11]];
    Label[6];
    Print[StyleForm["For any value of ", "Output", FontFamily -> "Times-Plain", FontSize -> 12], l11,
     StyleForm[" < ", "Output", FontFamily -> "Times-Plain", FontSize -> 12], par[[1]],
     StyleForm[" < ", "Output", FontFamily -> "Times-Plain", FontSize -> 12], l22,
     StyleForm[" and ", "Output", FontFamily -> "Times-Plain", FontSize -> 12], par[[2]],
     StyleForm[" > 0,", "Output", FontFamily -> "Times-Plain", FontSize -> 12],
     StyleForm["three real equilibrium positions exist ", "Output", FontFamily -> "Times-Plain",
      FontSize -> 12], unk, StyleForm["e1 < ", "Output", FontFamily -> "Times-Plain",
      FontSize -> 12], unk, StyleForm["e2 < ", "Output", FontFamily -> "Times-Plain",
      FontSize -> 12], unk, StyleForm["e3 where:", "Output", FontFamily -> "Times-Plain",
      FontSize -> 12]];
    Print[unk, StyleForm["e1  is unstable;", "Output", FontFamily -> "Times-Plain", FontSize -> 12]];
    Print[unk, StyleForm["e2  is stable;", "Output", FontFamily -> "Times-Plain", FontSize -> 12]];
    Print[unk, StyleForm["e3  is unstable.", "Output", FontFamily -> "Times-Plain", FontSize -> 12]]; Print[""];
    Print[StyleForm["In particular, if ", "Output", FontFamily -> "Times-Plain",
      FontSize -> 12], par[[1]], StyleForm[" = ", "Output", FontFamily -> "Times-Plain",
      FontSize -> 12], l1, StyleForm[" or ", "Output", FontFamily -> "Times-Plain", FontSize -> 12], par[[1]],
     StyleForm[" = ", "Output", FontFamily -> "Times-Plain", FontSize -> 12], l2,
     StyleForm[", two of them coincide.", "Output", FontFamily -> "Times-Plain", FontSize -> 12]]; Print[""];
    Print[StyleForm["If ", "Output", FontFamily -> "Times-Plain", FontSize -> 12], par[[1]],
     StyleForm[" < ", "Output", FontFamily -> "Times-Plain", FontSize -> 12], l11,
     StyleForm[" or ", "Output", FontFamily -> "Times-Plain", FontSize -> 12], par[[1]],
     StyleForm[" > ", "Output", FontFamily -> "Times-Plain", FontSize -> 12], l22,
     StyleForm[", only one equilibrium position exists.", "Output", FontFamily -> "Times-Plain", FontSize -> 12]]; Goto[20]; Label[7];
    Print[StyleForm["For any value of ", "Output", FontFamily -> "Times-Plain", FontSize -> 12], l11,
     StyleForm[" < ", "Output", FontFamily -> "Times-Plain", FontSize -> 12], par[[1]],
     StyleForm[" < ", "Output", FontFamily -> "Times-Plain", FontSize -> 12], l22,
     StyleForm[" and ", "Output", FontFamily -> "Times-Plain", FontSize -> 12], par[[2]],
     StyleForm[" > 0,", "Output", FontFamily -> "Times-Plain", FontSize -> 12],
     StyleForm[" three real equilibrium positions exist ", "Output", FontFamily -> "Times-Plain",
      FontSize -> 12], unk, StyleForm["e1 < ", "Output", FontFamily -> "Times-Plain", FontSize -> 12], unk,
     StyleForm["e2 < ", "Output", FontFamily -> "Times-Plain", FontSize -> 12], unk,
     StyleForm["e3 where:", "Output", FontFamily -> "Times-Plain", FontSize -> 12]];
    Print[unk, StyleForm["e1  is stable;", "Output", FontFamily -> "Times-Plain", FontSize -> 12]];
    Print[unk, StyleForm["e2  is unstable;", "Output", FontFamily -> "Times-Plain", FontSize -> 12]];
    Print[unk, StyleForm["e3  is stable.", "Output", FontFamily -> "Times-Plain", FontSize -> 12]]; Print[""];
    Print[StyleForm["In particular, if ", "Output", FontFamily -> "Times-Plain", FontSize -> 12], par[[1]],
     StyleForm[" = ", "Output", FontFamily -> "Times-Plain", FontSize -> 12], l1,
     StyleForm[" or ", "Output", FontFamily -> "Times-Plain", FontSize -> 12], par[[1]],
     StyleForm[" = ", "Output", FontFamily -> "Times-Plain", FontSize -> 12], l2,
     StyleForm[", two of them coincide.", "Output", FontFamily -> "Times-Plain", FontSize -> 12]]; Print[""];
    Print[StyleForm["If ", "Output", FontFamily -> "Times-Plain", FontSize -> 12], par[[1]],
     StyleForm[" < ", "Output", FontFamily -> "Times-Plain", FontSize -> 12], l11,
     StyleForm[" or ", "Output", FontFamily -> "Times-Plain", FontSize -> 12], par[[1]],
     StyleForm[" > ", "Output", FontFamily -> "Times-Plain", FontSize -> 12], l22,
     StyleForm[", only one equilibrium position exists.", "Output", FontFamily -> "Times-Plain",
      FontSize -> 12]];
    Goto[20];
    Label[8];
    Print[StyleForm["For any value of ", "Output", FontFamily -> "Times-Plain", FontSize -> 12], l11,
     StyleForm[" < ", "Output", FontFamily -> "Times-Plain", FontSize -> 12], par[[1]],
     StyleForm[" < ", "Output", FontFamily -> "Times-Plain", FontSize -> 12], l22,
     StyleForm[" and ", "Output", FontFamily -> "Times-Plain", FontSize -> 12], par[[2]],
     StyleForm[" < 0,", "Output", FontFamily -> "Times-Plain", FontSize -> 12],
     StyleForm[" three real equilibrium positions exist ", "Output",
      FontFamily -> "Times-Plain", FontSize -> 12], unk, StyleForm["e1 < ", "Output",
      FontFamily -> "Times-Plain", FontSize -> 12], unk, StyleForm["e2 < ", "Output",
      FontFamily -> "Times-Plain", FontSize -> 12], unk, StyleForm["e3 where:", "Output",
      FontFamily -> "Times-Plain", FontSize -> 12]];
     Print[unk, StyleForm["e1  is unstable;", "Output", FontFamily -> "Times-Plain",
      FontSize -> 12]];
     Print[unk, StyleForm["e2  is stable;", "Output", FontFamily -> "Times-Plain", FontSize -> 12]];
    Print[unk, StyleForm["e3  is unstable.", "Output", FontFamily -> "Times-Plain", FontSize -> 12]];
    Print[""];
    Print[StyleForm["In particular, if ", "Output", FontFamily -> "Times-Plain", FontSize -> 12], par[[1]],
     StyleForm[" = ", "Output", FontFamily -> "Times-Plain", FontSize -> 12], l1,
     StyleForm[" or ", "Output", FontFamily -> "Times-Plain", FontSize -> 12], par[[1]],
     StyleForm[" = ", "Output", FontFamily -> "Times-Plain", FontSize -> 12], l2,
     StyleForm[", two of them coincide.", "Output", FontFamily -> "Times-Plain", FontSize -> 12]];
    Print[""];
    Print[StyleForm["If ", "Output", FontFamily -> "Times-Plain", FontSize -> 12], par[[1]],
     StyleForm[" < ", "Output", FontFamily -> "Times-Plain", FontSize -> 12], l11,
     StyleForm[" or ", "Output", FontFamily -> "Times-Plain", FontSize -> 12], par[[1]],
     StyleForm[" > ", "Output", FontFamily -> "Times-Plain", FontSize -> 12], l22,
     StyleForm[", only one equilibrium position exists.", "Output", FontFamily -> "Times-Plain", FontSize -> 12]];
    Goto[20];
    Label[9];
    Print[StyleForm["For any value of ", "Output", FontFamily -> "Times-Plain", FontSize -> 12], l11,
     StyleForm[" < ", "Output", FontFamily -> "Times-Plain", FontSize -> 12], par[[1]],
     StyleForm[" < ", "Output", FontFamily -> "Times-Plain", FontSize -> 12], l22,
     StyleForm[" and ", "Output", FontFamily -> "Times-Plain", FontSize -> 12], par[[2]],
     StyleForm[" < 0", "Output", FontFamily -> "Times-Plain", FontSize -> 12],
     StyleForm[", three real equilibrium positions exist ", "Output", FontFamily -> "Times-Plain",
      FontSize -> 12], unk, StyleForm["e1 < ", "Output", FontFamily -> "Times-Plain",
      FontSize -> 12], unk, StyleForm["e2 < ", "Output", FontFamily -> "Times-Plain",
      FontSize -> 12], unk, StyleForm["e3 where:", "Output", FontFamily -> "Times-Plain", FontSize -> 12]];
    Print[unk, StyleForm["e1  is stable;", "Output", FontFamily -> "Times-Plain", FontSize -> 12]];
    Print[unk, StyleForm["e2  is unstable;", "Output", FontFamily -> "Times-Plain", FontSize -> 12]];
    Print[unk, StyleForm["e3  is stable.", "Output", FontFamily -> "Times-Plain", FontSize -> 12]]; Print[""];
    Print[StyleForm["In particular, if ", "Output", FontFamily -> "Times-Plain", FontSize -> 12], par[[1]],
     StyleForm[" = ", "Output", FontFamily -> "Times-Plain", FontSize -> 12], l1,
     StyleForm[" or ", "Output", FontFamily -> "Times-Plain", FontSize -> 12], par[[1]],
     StyleForm[" = ", "Output", FontFamily -> "Times-Plain", FontSize -> 12], l2,
     StyleForm[", two of them coincide.", "Output", FontFamily -> "Times-Plain", FontSize -> 12]];
    Print[""];
    Print[StyleForm["If ", "Output", FontFamily -> "Times-Plain", FontSize -> 12], par[[1]],
     StyleForm[" < ", "Output", FontFamily -> "Times-Plain", FontSize -> 12], l11,
     StyleForm[" or ", "Output", FontFamily -> "Times-Plain", FontSize -> 12], par[[1]],
     StyleForm[" > ", "Output", FontFamily -> "Times-Plain", FontSize -> 12], l22,
     StyleForm[", only one equilibrium position exists.", "Output", FontFamily -> "Times-Plain", FontSize -> 12]];
    Goto[20];
    Label[12];
    Print[StyleForm["For  ", "Output", FontFamily -> "Times-Plain", FontSize -> 12], par[[1]],
     StyleForm[" = ", "Output", FontFamily -> "Times-Plain", FontSize -> 12], l22,
     StyleForm[" and for any ", "Output", FontFamily -> "Times-Plain", FontSize -> 12], par[[2]],
     StyleForm[" in a neighbourhood of the origin ", "Output", FontFamily -> "Times-Plain", FontSize -> 12], unk,
     StyleForm["e1 is the only unstable equilibrium position existing.", "Output", FontFamily -> "Times-Plain", FontSize -> 12]];
     Goto[20];
     Label[11];
    Print[StyleForm["For  ", "Output", FontFamily -> "Times-Plain", FontSize -> 12], par[[1]],
     StyleForm[" = ", "Output", FontFamily -> "Times-Plain", FontSize -> 12], l22,
     StyleForm[" and for any ", "Output", FontFamily -> "Times-Plain", FontSize -> 12], par[[2]],
     StyleForm[" in a neighbourhood of the origin ", "Output", FontFamily -> "Times-Plain", FontSize -> 12], unk,
     StyleForm["e1 is the only existing unstable equilibrium position.", "Output", FontFamily -> "Times-Plain",
      FontSize -> 12]];
     Goto[20];
     Label[20];
     ClearAll[];
]


(*
 --------------------------------------------------------------------------------------------
 -
 -   T H E   P R O G R A M  Bif2G:
 -
 -                 UsageBif2G[]
 -                 HelpBif2G[]
 -                 Bif2G[eq_, unk_, par_, a_, b_, p_, valpar_]
 -
 --------------------------------------------------------------------------------------------
*)


UsageBif2G[] :=
Module[{},
StylePrint["Aims of the program Bif2G", "Output", FontFamily -> "Times-Bold", FontSize -> 12];
StylePrint["The program supplies a graphical analysis of the existence and stability properties of the equilibrium positions of the equation ", "Output", FontFamily -> "Times-Plain", FontSize -> 10];
Print[DisplayForm[\(x\^,\)], " = ", DisplayForm["F"], "(", DisplayForm["x"], ", \[Mu], ", DisplayForm[\(\[Lambda]\^*\)], "),"];
StylePrint["when the parameter \[Mu] varies and \[Lambda] has a fixed value \!\(\(\[Lambda]\^*\)\).", "Output", FontFamily -> "Times-Plain", FontSize -> 10];
StylePrint["The command line is ", "Output", FontFamily -> "Times-Plain", FontSize -> 10];
StylePrint["Bif2G[eq, unk, par, a, b, points, valpar]", "Output", FontFamily -> "Times-Plain", FontSize -> 10];
StylePrint["where eq is a first-order differential equation in the unknown unk and the parameters par, 2a and 2b are the dimensions of the rectangle in which the curve F(x, \[Mu], \!\(\(\[Lambda]\^*\)\)) = 0 of the equilibrium positions is represented together with the region of the plane (x, \[Mu]) in which F(x, \[Mu], \!\(\(\[Lambda]\^*\)\)) < 0, points is a graphical option fixing the number of the points used to draw the curve F(x, \[Mu],  \!\(\(\[Lambda]\^*\)\)) = 0, and finally valpar is the fixed value \!\(\(\[Lambda]\^*\)\) of \[Lambda]. ", "Output", FontFamily -> "Times-Plain", FontSize -> 10];
]

HelpBif2G[]:=
Module[{},
StylePrint["How to use the program Bif2G","Output",FontFamily->"Times-Bold",FontSize->12];
StylePrint["The program graphically analyzes the bifurcation of the origin","Output",FontFamily->"Times-Plain",FontSize->10];
StylePrint["To make active the program type in the following input data","Output",FontFamily->"Times-Plain",FontSize->10];
StylePrint["eq = the equation (f.e. x' == x + \[Mu]\!\(x\^2\) + \[Lambda]\!\(x\^3\); )","Output",FontFamily->"Times-Plain",FontSize->10];
StylePrint["unk = the unknown (f.e. x;)","Output",FontFamily->"Times-Plain",FontSize->10];
StylePrint["par = the parameters (f.e. {\[Mu], \[Lambda]};)","Output",FontFamily->"Times-Plain",FontSize->10];
StylePrint["a = the semi-length of the first parameter interval, (f.e. 1;)","Output",FontFamily->"Times-Plain",FontSize->10];
StylePrint["b = the semi-length of the unknown interval, (f.e. 1;) ","Output",FontFamily->"Times-Plain",FontSize->10];
StylePrint["points = the plot point number, (f.e. 25;) ","Output",FontFamily->"Times-Plain",FontSize->10];
StylePrint["valpar = the fixed value of the second parameter \[Mu], (f.e. -0.1;)","Output",FontFamily->"Times-Plain",FontSize->10];
StylePrint["Bif2G[eq, unk, par, a, b, points, valpar]","Output",FontFamily->"Times-Plain",FontSize->10];
]


Bif2G[eq_, unk_, par_, a_, b_, p_, valpar_] :=
  Module[{gb,hb,mb,m1,pl,neg,qbif,plo},
  FG = eq[[2]] /. {par[[1]] -> valpar};
   gb[i_, j_] := FG /. {par[[2]] -> i, unk -> j};
   hb[i_, j_] := If[gb[i, j] < 0, {i, j}, {0, 0}];
   mb = Table[hb[i, j], {i, -a, a, (2*a)/p}, {j, -b, b, (2*b)/p}];
   m1 = Flatten[mb, 1];
   neg = DeleteCases[m1, {0, 0}];
   pl[i_] := Graphics[Circle[neg[[i]], 0.005], AspectRatio -> Automatic, Axes -> Automatic,
      AxesLabel -> {par[[2]], unk}];
   qbif = Table[pl[i], {i, 1, Length[neg]}];
   plo = ImplicitPlot[FG == 0, {par[[2]], -a, a}, {unk, -b, b}, PlotPoints -> 20, DisplayFunction -> Identity];
    Print[StyleForm["The curve in the plot represents the equilibrium points for ", "Output",
      FontFamily -> "Times-Plain", FontSize -> 12], par[[1]], StyleForm[" = ", "Output",
      FontFamily -> "Times-Plain", FontSize -> 12], valpar, StyleForm[" as a function of ",
      "Output", FontFamily -> "Times-Plain", FontSize -> 12], par[[2]],
     StyleForm[" and ", "Output", FontFamily -> "Times-Plain", FontSize -> 12],
     unk, StyleForm[".", "Output", FontFamily -> "Times-Plain", FontSize -> 12]];
    Print[StyleForm["In the dashed region ", "Output", FontFamily -> "Times-Plain",
      FontSize -> 12], unk, StyleForm["' is negative; in the other region it is positive.", "Output",
      FontFamily -> "Times-Plain", FontSize -> 12]];
    Show[qbif, plo, DisplayFunction -> $DisplayFunction];
]


(*
 ----------------------------------------------------------------------------------------------
 -
 -   T H E   P R O G R A M  HopfBif:
 -
 -                 UsageHopfBif[]
 -                 HelpHopfBif[]
 -                 HopfBif[sys_, unk_, par_, valpar_, {h_, n_}, {k_, m_}, {T1_, T2_}, steps_]
 -
 ----------------------------------------------------------------------------------------------
*)


UsageHopfBif[]:=
Module[{},
StylePrint["Aims of the program HopfBif","Output",FontFamily->"Times-Bold",FontSize->12];
StylePrint["The program, after trasforming the given system with numerical coefficients into the normal form","Output",FontFamily\[Rule]"Times-Plain",FontSize\[Rule]10];
Print[DisplayForm[SuperscriptBox["x",","]]," = ",DisplayForm["\[Alpha](\[Lambda])x - \[Beta](\[Lambda])y + P(x, y, \[Lambda]),"]];
Print[DisplayForm[SuperscriptBox["y",","]]," = ",DisplayForm["\[Beta](\[Lambda])x + \[Alpha](\[Lambda])y + Q(x, y, \[Lambda]),"]];
StylePrint["verifies all the hypoteses of Hopf's theorem and, when they are satisfied, shows the phase portraits of the system around the bifurcation value \[Lambda] = 0.","Output",FontFamily\[Rule]"Times-Plain",FontSize\[Rule]10];
StylePrint["The command is ","Output",FontFamily\[Rule]"Times-Plain",FontSize\[Rule]10];
StylePrint["HopfBif[eq, unk, par, valpar, {h, n}, {k, m}, {T1, T2}, steps]","Output",FontFamily\[Rule]"Times-Plain",FontSize\[Rule]10];
StylePrint["where sys is the system, unk is the list of unknowns, par is the parameter appearing into the system and valpar is a list of three values {\!\(\[Lambda]\_1\), 0, \!\(\[Lambda]\_2\)} of the parameter, where \!\(\[Lambda]\_1\)<0 and \!\(\[Lambda]\_2\)>0. The other input data have the same meaning of the program Phase2D.","Output",FontFamily\[Rule]"Times-Plain",FontSize\[Rule]10];
]

HelpHopfBif[]:=
Module[{},
StylePrint["How to use the program HopfBif","Output",FontFamily\[Rule]"Times-Bold",FontSize\[Rule]12];
StylePrint["The program HopfBif reduces the planar system with numeric coefficients to canonical form, verifies the hypotheses of Hopf's theorem, and shows the phase portrait for negative and positive parameter values. ","Output",FontFamily\[Rule]"Times-Plain",FontSize\[Rule]10];
StylePrint["To make active the program type in the following input data.","Output",FontFamily\[Rule]"Times-Plain",FontSize\[Rule]10];
StylePrint["sys = the equation system (f.e. {x' == \[Lambda]x - y - x(\!\(x\^2\) + \!\(y\^2\)), y' == x + \[Lambda]y - y(\!\(x\^2\) + \!\(y\^2\))};)","Output",FontFamily\[Rule]"Times-Plain",FontSize\[Rule]10];
StylePrint["unk = the unknown (f.e. {x, y};)","Output",FontFamily\[Rule]"Times-Plain",FontSize\[Rule]10];
StylePrint["par = the parameter (f.e. \[Lambda];)","Output",FontFamily\[Rule]"Times-Plain",FontSize\[Rule]10];
StylePrint["valpar = the variation interval of the parameter (f.e. {-0.5, 0.5};)","Output",FontFamily\[Rule]"Times-Plain",FontSize\[Rule]10];
StylePrint["{h, n} = the step and number of points on the first axis","Output",FontFamily\[Rule]"Times-Plain",FontSize\[Rule]10];
StylePrint["{k, m} = the step and number of points on the second axis","Output",FontFamily\[Rule]"Times-Plain",FontSize\[Rule]10];
StylePrint["{T1, T2} = the time interval of numerical integration (f.e. {0,10};)","Output",FontFamily\[Rule]"Times-Plain",FontSize\[Rule]10];
StylePrint["steps = the number of steps in the numerical integration (f.e. 2000;)","Output",FontFamily\[Rule]"Times-Plain",FontSize\[Rule]10];
StylePrint["HopfBif[sys, unk, par, valpar, {h, n}, {k, m}, {T1, T2}, steps]","Output",FontFamily\[Rule]"Times-Plain",FontSize\[Rule]10];
]


HopfBif[sys_, unk_, par_, valpar_, {h_, n_}, {k_, m_}, {T1_, T2_}, steps_] :=
 Module[{},
  f = sys[[1,2]];
  g = sys[[2,2]];
  sys0 = sys /. par -> 0;
  If[(f /. {unk[[1]] -> 0, unk[[2]] -> 0}) != 0 || (g /. {unk[[1]] -> 0, unk[[2]] -> 0}) != 0,
  Print[StyleForm["The origin is not an equilibrium point", "Output",
  FontFamily -> "Times-Plain", FontSize -> 12]]; Goto[3],
  Print[StyleForm["For ", "Output", FontFamily -> "Times-Plain", FontSize -> 12], par,
       StyleForm[" = 0,", "Output", FontFamily -> "Times-Plain", FontSize -> 12]];
   CriticalEqN[sys0, unk]];
    If[G < 0, Goto[0], Goto[3]];
    Label[0];
    F[i_] := sys[[i,2]];
    j = Table[D[F[i], unk[[j]]], {i, 1, 2}, {j, 1, 2}] /. {unk[[1]] -> 0, unk[[2]] -> 0};
    lambda = Apart[Eigenvalues[j]];
    Print[StyleForm["The eigenvalues of the jacobian matrix at the origin are", "Output",
      FontFamily -> "Times-Plain", FontSize -> 12]];
    Print[lambda[[1]], "      ", lambda[[2]]];
    lambda1 = If[Head[lambda[[1]][[1]]] + Head[lambda[[1]][[2]]] ===
       Complex + Symbol, lambda[[1]][[2]],lambda[[1]][[1]]];
    lambda2 = If[Head[lambda[[1]][[1]]] + Head[lambda[[1]][[2]]] ===
       Complex + Symbol, lambda[[1]][[1]],lambda[[1]][[2]]];
    im0 = lambda2 /. par -> 0;

    If[im0^2 < 0 && lambda1 /. par -> 0 == 0 && (D[lambda1, par] /. par -> 0) != 0 &&
        (lambda2 /. par -> 0) != 0, Goto[1], Goto[2]];
    Label[1];
    Print[StyleForm["The hypoteses of Hopf's theorem are satisfied.", "Output",
      FontFamily -> "Times-Plain", FontSize -> 12]];
    px = Table[(i - n - 1)*h, {i, 1, 2*n + 1}];
    py = Table[(j - m - 1)*k, {j, 1, 2*m + 1}];
    equaz1[l_] := Derivative[1][unk[[1]]][t] == (F[1] /. {par -> l, unk[[1]] -> unk[[1]][t],
         unk[[2]] -> unk[[2]][t]});
    equaz2[l_] := Derivative[1][unk[[2]]][t] == (F[2] /. {par -> l, unk[[1]] -> unk[[1]][t],
         unk[[2]] -> unk[[2]][t]});
    s[i_, j_, l_] := NDSolve[{equaz1[l], equaz2[l], unk[[1]][0] == px[[i]], unk[[2]][0] == py[[j]]},
      {unk[[1]], unk[[2]]}, {t, T1, T2}, MaxSteps -> steps];
    pl[i_, j_, l_] := ParametricPlot[Evaluate[{unk[[1]][t], unk[[2]][t]} /. s[i, j, l]],
      {t, T1, T2}, AspectRatio -> Automatic, DisplayFunction -> Identity];
    Print[" "];
    Print[StyleForm["Phase portrait for ", "Output", FontFamily -> "Times-Plain", FontSize -> 12], par,
     StyleForm[" = ", "Output", FontFamily -> "Times-Plain", FontSize -> 12], valpar[[1]]];
    gr1 = Show[Table[pl[i, j, valpar[[1]]], {i, 1, 2*n + 1}, {j, 1, 2*m + 1}],
      AxesLabel -> {StyleForm[unk[[1]], FontSlant -> "Italic"], StyleForm[unk[[2]],
         FontSlant -> "Italic"]}, DisplayFunction -> $DisplayFunction];
    Print[StyleForm["Phase portrait for ", "Output", FontFamily -> "Times-Plain",
      FontSize -> 12], par, StyleForm[" = 0", "Output", FontFamily -> "Times-Plain",
      FontSize -> 12]];
    gr0 = Show[Table[pl[i, j, 0], {i, 1, 2*n + 1}, {j, 1, 2*m + 1}], AxesLabel ->
       {StyleForm[unk[[1]], FontSlant -> "Italic"], StyleForm[unk[[2]], FontSlant -> "Italic"]},
      DisplayFunction -> $DisplayFunction];
    Print[StyleForm["Phase portrait for ", "Output", FontFamily -> "Times-Plain",
      FontSize -> 12], par, StyleForm[" = ", "Output", FontFamily -> "Times-Plain",
      FontSize -> 12], valpar[[2]]];
    gr2 = Show[Table[pl[i, j, valpar[[2]]], {i, 1, 2*n + 1}, {j, 1, 2*m + 1}],
      AxesLabel -> {StyleForm[unk[[1]], FontSlant -> "Italic"], StyleForm[unk[[2]],
         FontSlant -> "Italic"]}, DisplayFunction -> $DisplayFunction];
    Goto[3*b];
    Label[2];
    Label[3];
    Print[StyleForm["The hypotheses of Hopf's theorem are not verified.", "Output",
      FontFamily -> "Times-Plain", FontSize -> 12]];
    Label[3*b];
]


(*
 ----------------------------------------------------------------------------------------------
 -
 -   T H E   P R O G R A M  LindPoinc:
 -
 -                 UsageLindPoinc[]
 -                 HelpLindPoinc[]
 -                 LindPoinc[sys_, unk_, var_, par_, m_]
 -
 ----------------------------------------------------------------------------------------------
*)


UsageLindPoinc[]:=
Module[{},
StylePrint["Aims of the program LindPoinc","Output",FontFamily\[Rule]"Times-Bold",FontSize\[Rule]12];
StylePrint[" The program verifies the existence of periodic orbits and finds their approximate analytic expression up to the second order terms in the parameter \[CurlyEpsilon]. It is valid for a system of two differential equations having the following form","Output",FontFamily\[Rule]"Times-Plain", FontSize\[Rule]10];
Print[DisplayForm[SuperscriptBox["x",","]]," = ",DisplayForm["y + \[CurlyEpsilon]f(x, y),"]];
Print[DisplayForm[SuperscriptBox["y",","]]," = ",DisplayForm["-x + \[CurlyEpsilon]g(x, y),"]];
StylePrint["where f(x, y) and g(x, y) are regular linear or nonlinear functions.","Output",FontFamily\[Rule]"Times-Plain",FontSize\[Rule]10];
StylePrint["The command line is","Output",FontFamily\[Rule]"Times-Plain",FontSize\[Rule]10];
StylePrint["LindPoinc[sys, unk, var, par, m]","Output",FontFamily\[Rule]"Times-Plain",FontSize\[Rule]10];
StylePrint["where sys is the system, unk are the unknowns, var denotes the independent variable, par is the parameter, m is an integer which can assume the values 1 or 2 according with the approximation order.","Output",FontFamily\[Rule]"Times-Plain",FontSize\[Rule]10];
]

HelpLindPoinc[]:=
Module[{},
StylePrint["How to use the program LindPoinc","Output",FontFamily\[Rule]"Times-Bold",FontSize\[Rule]12];
StylePrint["The program determines the periodic orbit of a system containing one parameter ","Output",FontFamily\[Rule]"Times-Plain",FontSize\[Rule]10];
StylePrint["To make active the program type in the following input data","Output",FontFamily\[Rule]"Times-Plain",FontSize\[Rule]10];
StylePrint["sys = the list of the equations (f.e.{x' == y, y' == - x - \!\(\[CurlyEpsilon]x\^3\)}; )","Output",FontFamily\[Rule]"Times-Plain",FontSize\[Rule]10];
StylePrint["unk = the list of unknowns (f.e.{x, y};)","Output",FontFamily\[Rule]"Times-Plain",FontSize\[Rule]10];
StylePrint["var = the independent variable (f.e. t;)","Output",FontFamily\[Rule]"Times-Plain",FontSize\[Rule]10];
StylePrint["par = the parameter (f.e. \[Epsilon];)","Output",FontFamily\[Rule]"Times-Plain",FontSize\[Rule]10];
StylePrint["m = the approximation order (1 or 2;)","Output",FontFamily\[Rule]"Times-Plain",FontSize\[Rule]10];
StylePrint["LindPoinc[sys, unk, var, par, m]","Output",FontFamily\[Rule]"Times-Plain",FontSize\[Rule]10];
]


LindPoinc[sys_, unk_, var_, par_, m_] :=
Module[{n, data, xu, subst, sysfi, ap1, sol1, syst, z, yp, red, G,coefs,coefc},

If[m > 2,Print[StyleForm["The approximation order has to be 1 or 2.", "Output",
       FontFamily -> "Times-Plain", FontSize -> 12]]; Goto[2]];
n = Length[sys];
data = {X, 0};
xu[h_] := SeriesData[par, 0, Table[u1[h, k][var], {k, 0, m}]];
subst = Flatten[Table[{unk[[h]] -> xu[h],
Derivative[1][unk[[h]]] -> D[xu[h], var]}, {h, 1, n}]];
sysfi = sys /. subst;
apl[h_] := LogicalExpand[sysfi[[h]]];
sol[0] = Flatten[DSolve[Flatten[
            Table[{apl[h][[1]], u1[h, 0][0] == data[[h]]}, {h, 1, n}]],
            Table[u1[h, 0][var], {h, 1, n}], var]];
syst[k_] :=Flatten[{Table[apl[h][[k + 1]], {h, 1, n}] /.Flatten[Table[sol[j],
           {j, 0, k - 1}]],Table[u1[h, k][0] == 0, {h, 1, n}]}];
sol[k_] := Flatten[DSolve[syst[k], Table[u1[h, k][var], {h, 1, n}], var]];

Do[
sol[k],
{k, 1, m}];

Do[
vf[h][var] = ComplexExpand[sol[0][[h,2]] + Sum[par^k*sol[k][[h,2]], {k, 1, m}]],
{h, 1, n}];

tau[s_] := s/(1 + Sum[w[k]*par^k, {k, 1, m}]);

Do[
z[h][s] = vf[h][var] /. var -> tau[s];
yp[h][s] = Expand[Normal[Series[z[h][s], {par, 0, m}]]];
coefs1[h] = Coefficient[yp[h][s], par*s*Sin[s]];
coefc1[h] = Coefficient[yp[h][s], par*s*Cos[s]],
{h, 1, n}];

red1 = DeleteCases[Solve[{coefc1[1] == 0, coefs1[1] == 0},{X, w[1]}],{X -> 0}];
red2 = DeleteCases[Solve[{coefc1[2] == 0, coefs1[2] == 0},{X, w[1]}],{X -> 0}];
red11 = Flatten[red1];
red12 = Flatten[red2];
ch1 =Which[Length[red11] > 1, red11[[2,2]], Length[red11] == 1,
      red11[[1,2]], Length[red11] == 0, nper];

If[Head[ch1] === Symbol, Goto[3]; Break[]];
ch2 = Which[Length[red12] > 1, red12[[2,2]], Length[red12] == 1,
      red12[[1,2]], Length[red12] == 0, nper];
If[Head[ch2] === Symbol, Goto[3]; Break[]];
Do[   coefs21[h] = Coefficient[yp[h][s], par^2*s*Sin[s]] /. {red1,red2};
      coefc21[h] = Coefficient[yp[h][s], par^2*s*Cos[s]] /. {red1,red2};
      coefs22[h] = Coefficient[yp[h][s], par^2*s^2*Sin[s]] /. {red1,red2};
      coefc22[h] = Coefficient[yp[h][s], par^2*s^2*Cos[s]] /. {red1,red2},
{h, 1, n}];

red21 =DeleteCases[
        Solve[{coefc21[1][[1]] == 0, coefs21[1][[1]] == 0},{X, w[2]}],{X -> 0}];
red22 = DeleteCases[
        Solve[{coefc22[2][[1]] == 0,coefs22[2][[1]] == 0}, {X, w[2]}],{X -> 0}];
sub1 = Flatten[DeleteCases[Join[red1,red2, red21, red22], {X -> 0}]];
fnz1 = Expand[Flatten[yp[1][s] /. red1 /. red21 /. red22, 2]];
fnz2 = Expand[Flatten[yp[2][s] /. red2 /. red21 /. red22, 2]];
l1 = Length[fnz1];

fnzfi1 = Simplify[Coefficient[fnz1, par, 0]] + Simplify[Coefficient[fnz1, par, 1]]*par +
      Simplify[Coefficient[fnz1, par, 2]]*par^2;
fnzfi2 = Simplify[Coefficient[fnz2, par, 0]] + Simplify[Coefficient[fnz2, par, 1]]*par +
      Simplify[Coefficient[fnz2, par, 2]]*par^2;
Print[StyleForm["The expansion of periodic motions to within terms of order ", "Output",
       FontFamily -> "Times-Plain", FontSize -> 12], m+1,
      StyleForm[" are:", "Output",
       FontFamily -> "Times-Plain", FontSize -> 12]];

Do[
Print[unk[[1]], StyleForm[" = ", "Output", FontFamily -> "Times-Plain", FontSize -> 12],
       fnzfi1[[i]] /. {X ->Global`X, s ->Global`s}];
Print[unk[[2]], StyleForm[" = ", "Output", FontFamily -> "Times-Plain", FontSize -> 12],
       fnzfi2[[i]] /. {X ->Global`X, s ->Global`s}];
Print[" "],
{i, 1, l1}];
G = Flatten[1 + Sum[w[k]*par^k, {k, 1, m}] /. red1 /. red21 /.red22];

s1 = var*G[[1]];
Print[StyleForm["where", "Output", FontFamily -> "Times-Plain", FontSize -> 12]];
Print[StyleForm["s = ", "Output", FontFamily -> "Times-Plain", FontSize -> 12], s1 /. X -> Global`X];
Print[" "];
Print[StyleForm["Use GLindPoinc to compare graphically the numerical solution with the approximate one for a given value of the parameter in the time interval (T1,T2).",
      "Output", FontFamily -> "Times-Plain", FontSize -> 12]];
Print[StyleForm["We suggest choosing the initial value list of x0 with the same length of periodic orbits.", "Output",
       FontFamily -> "Times-Plain", FontSize -> 12]];
Print[StyleForm["When any orbit is periodic, the length of x0 is arbitrary.", "Output",
       FontFamily -> "Times-Plain", FontSize -> 12]];
Goto[2];
Label[3];
Print[StyleForm["There is no periodic solution.", "Output", FontFamily -> "Times-Plain", FontSize -> 12]];
Label[2];
Clear[X];
]


(*
 ----------------------------------------------------------------------------------------------
 -
 -   T H E   P R O G R A M  GLindPoinc:
 -
 -                 UsageGLindPoinc[]
 -                 HelpGLindPoinc[]
 -                 GLindPoinc[sys_, unk_, var_, par_, r_, x0_, {T1_, T2_}]
 -
 ----------------------------------------------------------------------------------------------
*)


UsageGLindPoinc[]:=
Module[{},
StylePrint["Aims of the program GLindPoinc","Output",FontFamily->"Times-Bold",FontSize->12];
StylePrint[" This program can be used after applying the LindPoinc program. It compares graphically the numeric periodic orbits with the approximate ones derived by the Lindsted-Poincar method.","Output",FontFamily->"Times-Plain",FontSize->10];
StylePrint["The command line is","Output",FontFamily->"Times-Plain",FontSize->10];
StylePrint["GLindPoinc[sys, unk, var, par, r, x0, {T1, T2}]","Output",FontFamily->"Times-Plain",FontSize->10];
StylePrint["where sys, unk, var, par have the same meaning of the program LindPoinc, r is the value of the parameter, x0 is the list of Cauchy data for the first unknown, because the data of the second unknown are always assumed equal to zero; finally {T1, T2} is the time interval in which the solutions are plotted.","Output",FontFamily->"Times-Plain",FontSize->10];
]

HelpGLindPoinc[]:=
Module[{},
StylePrint["How to use the program GLindPoinc","Output",FontFamily->"Times-Bold",FontSize->12];
StylePrint["The program compares graphically the periodic orbits determined by LindPoinc and the numerical ones","Output",FontFamily->"Times-Plain",FontSize->10];
StylePrint["To make active the program type in the following input data","Output",FontFamily->"Times-Plain",FontSize->10];
StylePrint["sys = the list of the equations (f.e.{x' == y, y' == - x - \!\(\[CurlyEpsilon]x\^3\)}; )","Output",FontFamily->"Times-Plain",FontSize->10];
StylePrint["unk = the list of unknowns (f.e.{x, y};)","Output",FontFamily->"Times-Plain",FontSize->10];
StylePrint["t = the independent variable (f.e. t;)","Output",FontFamily->"Times-Plain",FontSize->10];
StylePrint["par = the parameter appearing in the system (f.e. \[CurlyEpsilon];)","Output",FontFamily->"Times-Plain",FontSize->10];
StylePrint["r = the parameter value (f.e. 0.1;)","Output",FontFamily->"Times-Plain",FontSize->10];
StylePrint["x0 = the list of initial points of requested trajectories (f.e. {0.5, 1};)","Output",FontFamily->"Times-Plain",FontSize->10];
StylePrint["{T1, T2} = the time interval (f.e. {0, 10};)","Output",FontFamily->"Times-Plain",FontSize->10];
StylePrint["GLindPoinc[sys, unk, var, par, r, x0, {T1, T2}]","Output",FontFamily->"Times-Plain",FontSize->10];
]


GLindPoinc[sys_, unk_, var_, par_, r_, x0_, {T1_, T2_}] :=
Module[{gz1, gz2, appr, sost1, eq1, eq2, soluz, ma, pl},
gz1[i_, j_] :=Flatten[fnz1[[i]] //. {par -> r, X -> x0[[j]], s -> s1}, 2];
gz2[i_, j_] :=Flatten[fnz2[[i]] //. {par -> r, X -> x0[[j]], s -> s1}, 2];
appr[i_, j_] := {Evaluate[gz1[i, j]], Evaluate[gz2[i, j]]};
le = Length[fnz1];
lex0 = Length[x0];
ma = Table[ParametricPlot[Evaluate[appr[i, j]], {var, T1, T2},
     PlotRange -> All, AxesLabel ->{StyleForm[unk[[1]], FontSlant -> "Italic"],
     StyleForm[unk[[2]], FontSlant -> "Italic"]},DisplayFunction -> Identity],{i, 1, le}, {j, 1, lex0}];
sost1 = {unk[[1]] -> unk[[1]][var], Derivative[1][unk[[1]]] -> Derivative[1][unk[[1]]][var],
      unk[[2]] -> unk[[2]][var], Derivative[1][unk[[2]]] -> Derivative[1][unk[[2]]][var], par -> r};
eq1 = sys[[1]] /. sost1;
eq2 = sys[[2]] /. sost1;
soluz[i_] := NDSolve[{eq1, eq2, unk[[1]][0] == x0[[i]], unk[[2]][0] == 0},unk,
         {var, T1, T2}];
pl[i_] :=ParametricPlot[Evaluate[{unk[[1]][var], unk[[2]][var]} /.
        soluz[i]], {var, T1, T2}, DisplayFunction -> Identity,
        PlotRange -> All, PlotStyle -> {Dashing[{0.02, 0.02}]},
        AxesLabel ->{StyleForm[unk[[1]], FontSlant -> "Italic"],
        StyleForm[unk[[2]], FontSlant -> "Italic"]}];
plfi = Table[pl[i], {i, 1, lex0}];
Print[StyleForm["Graphical comparison between the approximate solutions (continuous lines) and the numerical ones (dashed lines).", "Output",
       FontFamily -> "Times-Plain", FontSize -> 12]];
Show[plfi, ma, DisplayFunction -> $DisplayFunction,AspectRatio->Automatic];
]


(*
 ----------------------------------------------------------------------------------------------
 -
 -   T H E   P R O G R A M  NBoundary:
 -
 -                 UsageNBoundary[]
 -                 HelpNBoundary[]
 -                 NBoundary[eq_,unk_,var_,{a_,b_},data_,{dymin_,dymax_},n_,steps_,ansol_]
 -
 ----------------------------------------------------------------------------------------------
*)


UsageNBoundary[]:=
Module[{},
StylePrint["Aims of the program NBoundary","Output",FontFamily->"Times-Bold",FontSize->12];
StylePrint["The program solves by the shooting method the following mixed boundary value problem for a second order differential equation ","Output",FontFamily\[Rule]"Times-Plain",FontSize\[Rule]10];
Print[DisplayForm[SuperscriptBox["y",",,"]]," = ",DisplayForm["F(x, y, "],DisplayForm[SuperscriptBox["y",","]],"),"];
Print[DisplayForm["\!\(l\_a\)y(a)+ \!\(m\_a\) "],DisplayForm[SuperscriptBox["y",","]],"(a) = \!\(v\_a\),"];
Print[DisplayForm["\!\(l\_b\)y(b)+ \!\(m\_b\) "],DisplayForm[SuperscriptBox["y",","]],"(b) = \!\(v\_b\),"];
StylePrint["where \!\(l\_a\), \!\(m\_a\), \!\(l\_b\) and \!\(m\_\(b \)\) are constants assigned together with \!\(v\_a\) and \!\(v\_b\).","Output",FontFamily\[Rule]"Times-Plain",FontSize\[Rule]10];
StylePrint["The command line is ","Output",FontFamily\[Rule]"Times-Plain",FontSize\[Rule]10];
StylePrint["NBoundary[eq, unk, var, {a, b}, data, {dymin, dymax}, n, steps, ansol] ","Output",FontFamily\[Rule]"Times-Plain",FontSize\[Rule]10];
StylePrint["where eq is the equation to solve, unk are the unknowns, var denotes the independent variable, {a, b} is the interval in which we search for the solution, data represent the boundary conditions, {dymin, dymax} is the arbitrary interval of derivative values, n is the number of divisions of the previous interval, steps is the number of steps of numerical integration, and finally ansol is equal to zero if the exact solution is unknown and it is equal to the solution when this is known.","Output",FontFamily\[Rule]"Times-Plain",FontSize\[Rule]10];
]

HelpNBoundary[]:=
Module[{},
StylePrint["How to use the program NBoundary","Output",FontFamily->"Times-Bold",FontSize->12];
StylePrint["The program solves by shooting method a mixed boundary problem for the differential equation y''= F(x, y, y'). ","Output",FontFamily->"Times-Plain",FontSize->10];
StylePrint["To make active the program type in the following input data","Output",FontFamily->"Times-Plain",FontSize->10];
StylePrint["eq = the equation to solve (f.e. y'' - xy' + \!\(y\^2\) == 0;) ","Output",FontFamily->"Times-Plain",FontSize->10];
StylePrint["unk = the unknown (f.e. y;)","Output",FontFamily->"Times-Plain",FontSize->10];
StylePrint["var = the independent variable (f.e. x;) ","Output",FontFamily->"Times-Plain",FontSize->10];
StylePrint["{a, b} = the interval in which we search for the solution (f.e. {0, 1};)","Output",FontFamily->"Times-Plain",FontSize->10];
StylePrint["data = the boundary conditions (f.e. {y[0] == 1, y[1] == 2};","Output",FontFamily->"Times-Plain",FontSize->10];
StylePrint["{dymin, dymax} = the arbitrary interval of derivatives at the point a (f.e. {-1, 2};)","Output",FontFamily->"Times-Plain",FontSize->10];
StylePrint["n = the number of divisions of {dymin,dymax} (f.e. 10; )","Output",FontFamily->"Times-Plain",FontSize->10];
StylePrint["steps = the number of steps of numerical integration (f.e. 1000;) ","Output",FontFamily->"Times-Plain",FontSize->10];
StylePrint["ansol = 0; if the exact solution is unknown, the solution if it is known (f.e. 0;)","Output",FontFamily->"Times-Plain",FontSize->10];
StylePrint["NBoundary[eq, unk, var, {a, b}, data, {dymin, dymax}, n, steps, ansol] ","Output",FontFamily->"Times-Plain",FontSize->10];
]


NBoundary[eq_,unk_,var_,{a_,b_},data_,{dymin_,dymax_},n_,steps_,ansol_] :=
Module[{de, s, u,c, f, g, pl, r, sol, z},

sost = {unk -> unk[var],unk'->unk'[var],unk''->unk''[var]};
eqfi = eq /. sost;
de[i_] := dymin+ i*(dymax - dymin)/n;
s[i_] := NDSolve[{eqfi, data[[1]],unk'[a]==de[i]}, unk, {var, a, b}, MaxSteps -> steps];
u[i_] :=Evaluate[unk[var] /. s[i]][[1]];
u1[i_] := D[u[i], var] /.var -> b;
c[i_] := (data[[2,1]] /. {unk[b] -> (u[i] /. var -> b), unk'[b] -> u1[i]}) - data[[2,2]];
f = Table[{de[i],Chop[c[i],10^(-5)]},{i, 0, n}];
g = Interpolation[f];

If[Sign[c[0]] == Sign[c[n]], Goto[1], Goto[2]];
Label[1];
Print[StyleForm["Wrong choice of the range of initial derivatives.", "Output", FontFamily -> "Times-Plain", FontSize -> 12]];
Print[StyleForm["Try again evaluating the region in which the curve in the following plot intersects the axis D(i)= 0.", "Output", FontFamily -> "Times-Plain", FontSize -> 12]];
t1 = Table[Chop[c[i], 10^(-4)], {i, 0, n}];
extr = If[Sign[c[0]] > 0, Max[t1], Min[t1]];
pl= Plot[Evaluate[g[var]], {var, dymin, dymax}, PlotRange ->
    {{dymin,dymax},{0,extr}},PlotStyle->{Thickness[0.015]},
    AxesLabel->{StyleForm[unk',FontSlant->"Italic"],
    StyleForm["D(i)",FontSlant->"Italic"]}];
Goto[3];
Label[2];
r =FindRoot[g[var] == 0, {var, (dymax+dymin)/2}];
Print[StyleForm["Value of the initial derivative whose corresponding solution verifies the condition at ", "Output", FontFamily -> "Times-Plain", FontSize -> 12],var," = ",b,
      StyleForm["  is ", "Output", FontFamily -> "Times-Plain", FontSize -> 12]];
Print[unk, "'[",a,"] = ",r[[1,2]]];
sol = NDSolve[{eqfi, data[[1]], unk'[a] == r[[1,2]]},
unk,{var, a, b}];

z = Evaluate[unk[var] /. sol];
Print[StyleForm["Numerical solution", "Output",
FontFamily -> "Times-Plain", FontSize -> 12]];
Plot[z, {var, a, b}, AxesLabel->{StyleForm[var,FontSlant->"Italic"],StyleForm[unk,FontSlant->"Italic"]},
      PlotRange -> All];
If[ansol!= 0, Goto[4], Goto[3]];
Label[4];
Print[StyleForm["Absolute error", "Output", FontFamily -> "Times-Plain", FontSize -> 12]];
Plot[Chop[z- ansol,10^(-4)], {var, a, b}, AxesLabel->{StyleForm[var,FontSlant->"Italic"],
     StyleForm["er",FontSlant->"Italic"]},PlotRange -> All,PlotStyle->{Thickness[0.015]}];
Label[3];
]


(*
 ----------------------------------------------------------------------------------------------
 -
 -   T H E   P R O G R A M  NBoundary1:
 -
 -                 UsageNBoundary1[]
 -                 HelpNBoundary1[]
 -                 NBoundary1[eq_, unk_, var_, {a_, b_}, data_, n_, ansol_]
 -
 ----------------------------------------------------------------------------------------------
*)


UsageNBoundary1[]:=
Module[{},
StylePrint["Aims of the program NBoundary1","Output",FontFamily->"Times-Bold",FontSize->12];
StylePrint["The program solves by the finite difference method the following mixed boundary value problem for a second order linear differential equation","Output", FontFamily -> "Times-Plain", FontSize -> 10];
Print[DisplayForm[SuperscriptBox["y", ",,"]], " = ", DisplayForm["p(x)"], DisplayForm[SuperscriptBox["y", ","]], " + q(x)y + r(x),"];
Print[DisplayForm["\!\(l\_a\) y(a)+ \!\(m\_a\)"],DisplayForm[SuperscriptBox["y", ","]],"(a) = \!\(v\_a\),"];
Print[DisplayForm["\!\(l\_b\) y(b)+ \!\(m\_b\)"], DisplayForm[SuperscriptBox["y", ","]],"(b) = \!\(v\_b\),"];
StylePrint["where \!\(l\_a\), \!\(m\_a\), \!\(l\_b\) and \!\(m\_\(b\\\ \)\) are given constants together with \!\(v\_a\) and \!\(v\_b\), p, q, r are functions of class \!\(C\^n\)[a,b].", "Output", FontFamily -> "Times-Plain", FontSize -> 10];
StylePrint["The command line is ", "Output", FontFamily -> "Times-Plain", FontSize -> 10];
StylePrint["NBoundary1[eq, unk, var, {a, b}, data, n, ansol] ", "Output", FontFamily -> "Times-Plain", FontSize -> 10];
StylePrint["where eq is the linear differential equation to solve, unk is the unknown, var denotes the independent variable, {a, b} is the interval in which we search for the solution, data represent the boundary conditions, n is the number of divisions of the previous interval, and finally ansol is equal to 0 if the exact solution is unknown and it is equal to the solution when it is known.", "Output", FontFamily -> "Times-Plain", FontSize -> 10];
]

HelpNBoundary1[]:=
Module[{},
StylePrint["How to use the program NBoundary1","Output",FontFamily->"Times-Bold",FontSize->12];
StylePrint["The program solves by finite difference method a mixed boundary problem for the linear differential equation y'' = p(x) y' + q(x) y + r(x) = 0. ","Output",FontFamily->"Times-Plain",FontSize->10];
StylePrint["To make active the program type in the following input data","Output",FontFamily->"Times-Plain",FontSize->10];
StylePrint["eq = the equation to solve (f.e. y'' - xy' + y == 0;) ","Output",FontFamily->"Times-Plain",FontSize->10];
StylePrint["unk = the unknown (f.e. y;)","Output",FontFamily->"Times-Plain",FontSize->10];
StylePrint["var = the independent variable (f.e. x;) ","Output",FontFamily->"Times-Plain",FontSize->10];
StylePrint["{a, b} = the interval in which we search for the solution (f.e. {0, 1};) ","Output",FontFamily->"Times-Plain",FontSize->10];
StylePrint["data = the boundary conditions (f.e. {y[0] == 1, y[1] == 2}; ","Output",FontFamily->"Times-Plain",FontSize->10];
StylePrint["n = the number of divisions of {a, b} (f.e. 10; )","Output",FontFamily->"Times-Plain",FontSize->10];
StylePrint["ansol = 0; if the exact solution is unknown, the solution if it is known (f.e. 0;)","Output",FontFamily->"Times-Plain",FontSize->10];
StylePrint["NBoundary1[eq, unk, var, {a, b}, data, n, ansol]","Output",FontFamily->"Times-Plain",FontSize->10];
]


NBoundary1[eq_, unk_, var_, {a_, b_}, data_, n_, ansol_] :=
Module[{h, sost, equaz, sys, inc, sol, val, g},
hN = (b - a)/n;
x[i_]:= a + i*hN;
sostN0 = {unk[a] -> y[0], unk'[a] -> (y[1] -y[0])/hN};
sostN[i_] :={var -> x[i], unk ->y[i], unk' ->(y[i + 1] - y[i-1])/(2*hN), unk'' ->(y[i + 1] -2*y[i] + y[i - 1])/(hN^2)};
sostnN = {unk[b] -> y[n], unk'[b] -> (y[n] - y[n - 1])/hN};
equazN[i_] := Which[i == 0, data[[1]] /. sostN0,
              1 <= i <= n - 1, eq /. sostN[i], i == n, data[[2]] /. sostnN];
sysN = Table[equazN[i], {i, 0, n}];
incN = Table[y[i], {i, 0, n}];
solN = Flatten[NSolve[sysN, incN]];

pos[i_] := Flatten[Position[solN, y[i] -> _Real]];
valN = Table[{x[i], solN[[pos[i],2]]}, {i, 0, n}];
gN = Interpolation[valN];
StylePrint["Numerical solution","Output",FontFamily->"Times-Plain",FontSize->12];
Plot[gN[var], {var, a, b}, PlotRange -> All, AxesLabel->{StyleForm[var,FontSlant->"Italic"], StyleForm[unk,FontSlant->"Italic"]}];

If[ansol != 0, Goto[1], Goto[2]];
Label[1];
StylePrint["Absolute error","Output",FontFamily->"Times-Plain",FontSize->12];
Plot[gN[var] - ansol, {var, a, b}, AxesOrigin -> {0, 0},AxesLabel -> {StyleForm[var,FontSlant->"Italic"],
StyleForm["er",FontSlant->"Italic"]},PlotStyle->{Thickness[0.015]}];
Label[2];
]


(*
 ----------------------------------------------------------------------------------------------
 -
 -   T H E   P R O G R A M  NBoundary2:
 -
 -                 UsageNBoundary2[]
 -                 HelpNBoundary2[]
 -                 NBoundary2[eq_, unk_, var_, {a_, b_}, data_, n_, s_, steps_, ansol_]
 -
 ----------------------------------------------------------------------------------------------
*)


UsageNBoundary2[]:=
Module[{},
StylePrint["Aims of the program NBoundary2","Output",FontFamily->"Times-Bold",FontSize->12];
StylePrint["The program solves by the finite difference method a mixed boundary value problem for a second order nonlinear differential equation ", "Output", FontFamily -> "Times-Plain", FontSize -> 10];
Print[DisplayForm[SuperscriptBox["y", ",,"]], " = ",DisplayForm["F(x, y, "],DisplayForm[SuperscriptBox["y", ","]], "),"];
Print[DisplayForm["\!\(l\_a\)y(a)+ \!\(m\_a\)"],DisplayForm[SuperscriptBox["y", ","]],"(a) = \!\(v\_a\),"];
Print[DisplayForm["\!\(l\_b\)y(b)+ \!\(m\_b\) "],DisplayForm[SuperscriptBox["y", ","]],"(b) = \!\(v\_b\),"];
StylePrint["where \!\(l\_a\), \!\(m\_a\), \!\(l\_b\) and \!\(m\_b\) are given constants together with \!\(v\_a\) and \!\(v\_b\), F is a nonlinear function.", "Output", FontFamily -> "Times-Plain",FontSize -> 10];
StylePrint["The command line is ", "Output",FontFamily -> "Times-Plain", FontSize -> 10];
StylePrint["NBoundary2[eq, unk, var, {a, b}, data, n, s, steps, ansol] ", "Output", FontFamily -> "Times-Plain", FontSize -> 10];
StylePrint["where eq is the nonlinear equation to solve, unk is the unknown, var denotes the independent variable, {a, b} is the interval in which we search for the solution, data represent the boundary conditions, n is the number of divisions of the previous interval, s is the number of the approximates boundary value problems of the iterative method used, steps is the maximum number of iterations used by FindRoot, and finally ansol is equal to zero if the exact solution is unknown and it is equal to the solution when it is known.", "Output", FontFamily -> "Times-Plain", FontSize -> 10];
]

HelpNBoundary2[]:=
Module[{},
StylePrint["How to use the program NBoundary2","Output",FontFamily->"Times-Bold",FontSize->12];
StylePrint["The program solves by finite difference method a mixed boundary problem for the nonlinear differential equation y'' - F(x,y,y') = 0. ","Output",FontFamily->"Times-Plain",FontSize->10];
StylePrint["To make active the program type in the following input data","Output",FontFamily->"Times-Plain",FontSize->10];
StylePrint["eq = the equation to solve (f.e. y'' + \!\(y\^2\) == 0;) ","Output",FontFamily->"Times-Plain",FontSize->10];
StylePrint["unk = the unknown (f.e. y;)","Output",FontFamily->"Times-Plain",FontSize->10];
StylePrint["var = the independent variable (f.e. x;) ","Output",FontFamily->"Times-Plain",FontSize->10];
StylePrint["{a, b} = the interval in which we search for the solution (f.e. {0, 1};) ","Output",FontFamily->"Times-Plain",FontSize->10];
StylePrint["data = the boundary conditions (f.e. {y[0] == 1, y[1] == 2}; ","Output",FontFamily->"Times-Plain",FontSize->10];
StylePrint["n = the number of divisions of {a, b} (f.e. 10; )","Output",FontFamily->"Times-Plain",FontSize->10];
StylePrint["s = the number of approximate boundary value problems (f.e. 1;)","Output",FontFamily->"Times-Plain",FontSize->10];
StylePrint["steps = value of MaxIterations of the built-in function FindRoot (f.e. 30;);","Output",FontFamily->"Times-Plain",FontSize->10];
StylePrint["ansol = 0; if the exact solution is unknown, the solution if it is known (f.e. 0;)","Output",FontFamily->"Times-Plain",FontSize->10];
StylePrint["NBoundary2[eq, unk, var, {a, b}, data, n, s, steps, ansol]","Output",FontFamily->"Times-Plain",FontSize->10];
]


NBoundary2[eq_, unk_, var_, {a_, b_}, data_, n_, s_, steps_, ansol_] :=
Module[{h2, m, sost2, equaz2, sys2, ite2r, sol2, val2, g2},

eqkt = eq[[1]] /.{unk -> 0, Derivative[1][unk] -> 0,
       Derivative[1][Derivative[1][unk]] -> 0};
eqhom = eq[[1]] - eqkt;
h2 = (b - a)/n;
x2[i_] := a + i*h2;
sost2[i_, j_] := Which[i == 0, {unk[a] -> y[0, j],
       Derivative[1][unk][a] -> (y[1, j] - y[0, j])/h}, 1 <= i <= n - 1,
      {var -> x2[i], unk -> y[i, j],
       Derivative[1][unk] -> (y[i + 1, j] - y[i - 1, j])/(2*h2),
       Derivative[1][Derivative[1][unk]] ->
        (y[i + 1, j] - 2*y[i, j] + y[i - 1, j])/(h2)^2}, i == n,
      {unk[b] -> y[n, j], Derivative[1][unk][b] -> (y[n, j] - y[n - 1,j])/h2}];
equaz2[i_, j_] :=
     Which[i == 0, (data[[1,1]] /. sost2[0, j]) == j/s*data[[1,2]],
      1 <= i <= n - 1, (eqhom + j/s*eqkt /. sost2[i, j]) == 0, i == n,
      (data[[2,1]] /. sost2[n, j]) == j/s*data[[2,2]]];
kT[taT_, {wT__}] := FindRoot[taT, wT, MaxIterations -> steps];

Do[
sol2[0] = Table[y[i, 0] -> 0,
{i, 0, n}];

sys2[j] = Table[equaz2[i, j], {i, 0, n}];
unk12[j] = Table[y[i, j], {i, 0, n}];
iter2[j] = Table[{unk12[j][[i]], sol2[j - 1][[i,2]]}, {i, 1, n + 1}];
sol2[j] = Flatten[kT[sys2[j], iter2[j]]], {j, 1, s}];
val2 = Table[{x2[i], sol2[s][[i + 1,2]]}, {i, 0, n}];
g2 = Interpolation[val2];

StylePrint["Numerical solution","Output",FontFamily->"Times-Plain",FontSize->12];
Plot[g2[var], {var, a, b}, PlotRange -> All,
     AxesLabel ->{StyleForm[var, FontSlant -> "Italic"],
       StyleForm[unk, FontSlant -> "Italic"]}];
If[ansol != 0, Goto[1], Goto[2]];
Label[1];
StylePrint["Absolute error","Output",FontFamily->"Times-Plain",FontSize->12];
Plot[g2[var] - ansol, {var, a, b}, AxesOrigin -> {0, 0},
     AxesLabel ->{StyleForm[var, FontSlant -> "Italic"],
     StyleForm["er", FontSlant -> "Italic"]},PlotStyle->{Thickness[0.015]}];
Label[2];
]


(*
 ----------------------------------------------------------------------------------------------
 -
 -   T H E   P R O G R A M  NormalForm:
 -
 -                 UsageNormalForm[]
 -                 HelpNormalForm[]
 -                 NormalForm[sys_, unk_]
 -
 ----------------------------------------------------------------------------------------------
*)


UsageNormalForm[]:=
Module[{},
StylePrint["Aims of the program NormalForm","Output",FontFamily->"Times-Bold",FontSize->12];
StylePrint["The program reduces a planar differential system to the normal form when the eigenvalues of the linear part are immaginary.","Output",FontFamily->"Times-Plain",FontSize->10];
StylePrint["The command line is","Output",FontFamily->"Times-Plain",FontSize->10];
StylePrint["NormalForm[sys, unk]","Output",FontFamily->"Times-Plain",FontSize->10];
StylePrint["where sys is the planar differential system to reduce, and unk are the unknowns.","Output",FontFamily->"Times-Plain",FontSize->10];
]

HelpNormalForm[]:=
Module[{},
StylePrint["How to use the program NormalForm","Output",FontFamily->"Times-Bold",FontSize->12];
StylePrint[" This program reduces to the normal form a nonlinear planar system","Output",FontFamily->"Times-Plain",FontSize->10];
StylePrint["To make active the program type","Output",FontFamily->"Times-Plain",FontSize->10];
StylePrint["sys = the system to transform (f.e.\!\({x' == x + y + x\^2 + 2\ x\ y + \(x\^2\) y,
    y' == \(-2\)\ x - y - x\^2 - 2\ x\ y - x\^2\ y}\);)","Output",FontFamily->"Times-Plain",FontSize->10];
StylePrint["unk = the unknowns (f.e. {x, y};)","Output",FontFamily->"Times-Plain",FontSize->10];
StylePrint["NormalForm[sys, unk]","Output",FontFamily->"Times-Plain",FontSize->10];
]


NormalForm[sys_, unk_] :=
Module[{ff, gg, j, reeigve, imeigve, eq, sost, sol, sost2, sys1,solfi},
ff = sys[[1,2]];
gg = sys[[2,2]];
If[(ff /. {unk[[1]] -> 0, unk[[2]] -> 0}) == 0 &&
   (gg /. {unk[[1]] -> 0, unk[[2]] -> 0}) == 0, Goto[0],
   Print[StyleForm["The origin is not an equilibrium point.","Output",
          FontFamily -> "Times-Plain", FontSize -> 12]]; Goto[3]];
Label[0];
j ={{D[ff, unk[[1]]], D[ff, unk[[2]]]},{D[gg, unk[[1]]],
   D[gg, unk[[2]]]}} /.{unk[[1]] -> 0, unk[[2]] -> 0};
If[Re[Eigenvalues[j][[1]]] == 0 && Re[Eigenvalues[j][[2]]] == 0,Goto[1],
      Goto[2]];
Label[1];
reeigve = Re[Eigenvectors[j][[2,1]]*{1, 0}+ Eigenvectors[j][[2,2]]*{0,1}];
imeigve = Im[Eigenvectors[j][[2,1]]*{1, 0}+ Eigenvectors[j][[2,2]]*{0,1}];
eq = X[t]*reeigve + Y[t]*imeigve;
eq1 = X*reeigve + Y*imeigve;
sost = {unk[[1]] -> eq[[1]], unk[[2]] -> eq[[2]],unk[[1]]' -> D[eq[[1]], t],
       unk[[2]]' -> D[eq[[2]], t]};
sys1 = sys /. sost;
sol = Flatten[Solve[sys1, {D[X[t], t], D[Y[t], t]}]];
sost2 = {X[t] -> X, Y[t] -> Y, D[X[t], t] -> X',D[Y[t], t] ->Y'};
solfi = sol /. sost2;
sysin = {X' == Expand[solfi[[1,2]]],Y' == Expand[solfi[[2,2]]]};
Print[StyleForm[
        "The coordinate transformation that reduces the system to the normal form is:","Output",FontFamily -> "Times-Plain", FontSize -> 12]];
Print[unk[[1]],
      StyleForm[" = ","Output",FontFamily -> "Times-Plain", FontSize -> 12],
      eq1[[1]]/.{X->Global`X,Y->Global`Y}];
Print[unk[[2]],
      StyleForm[ " = ","Output",FontFamily -> "Times-Plain", FontSize -> 12],
      eq1[[2]]/.{X->Global`X,Y->Global`Y}];
Print[StyleForm[
        "The normal form of the system in the new variables (X,Y) is:",
        "Output",FontFamily -> "Times-Plain", FontSize -> 12]];
Print[StyleForm["X'= ","Output",FontFamily -> "Times-Plain", FontSize -> 12],
      Expand[solfi[[1,2]]/.{X->Global`X,Y->Global`Y}]];
Print[StyleForm["Y'= ","Output",FontFamily -> "Times-Plain", FontSize -> 12],
      Expand[solfi[[2,2]]/.{X->Global`X,Y->Global`Y}]];
Goto[3];
Label[2];
Print[StyleForm["The origin is not a critical equilibrium point.","Output",
        FontFamily -> "Times-Plain", FontSize -> 12]];
Print[StyleForm["The eigenvalues of the jacobian matrix at the origin are",
        "Output",FontFamily -> "Times-Plain", FontSize -> 12]];
Print[Eigenvalues[j]];
Label[3];
]


(*
 ----------------------------------------------------------------------------------------------
 -
 -   T H E   P R O G R A M  Zeros:
 -
 -                 UsageZeros[]
 -                 HelpZeros[]
 -                 Zeros[sys_, unk_, int_, n_, tol_]
 -
 ----------------------------------------------------------------------------------------------
*)


UsageZeros[]:=
Module[{},
StylePrint["Aims of the program Zeros","Output",FontFamily->"Times-Bold",FontSize->12];
StylePrint["The program searches for the zeros of a system of two or three finite equations, by numeric procedures based on the built-in function FindRoot.","Output",FontFamily->"Times-Plain",FontSize->10];
StylePrint["The command line is","Output",FontFamily->"Times-Plain",FontSize->10];
StylePrint["Zeros[sys, unk, int, n, tol] ","Output",FontFamily->"Times-Plain",FontSize->10];
StylePrint["where sys is the system, unk is the list of the unknowns, int is the region of the plane or space in which we search for the roots, n is the number of divisions of the int edges, and finally tol is the accuracy of the root values.","Output",FontFamily->"Times-Plain",FontSize->10];
]

HelpZeros[]:=
Module[{},
StylePrint["How to use the program Zeros","Output",FontFamily->"Times-Bold",FontSize->12];
StylePrint["The program is applicable to a system of two or three equations.","Output",FontFamily->"Times-Plain",FontSize->10];
StylePrint["sys = the system to solve (f.e.{x + xy == 0, y - x\!\(y\^2\) == 0};) ","Output",FontFamily->"Times-Plain",FontSize->10];
StylePrint["unk = the unknowns (f.e.{x, y};)","Output",FontFamily->"Times-Plain",FontSize->10];
StylePrint["int = the region of the plane or space in which we search for the roots (f.e.{{-1, 1}, {-2, 2}};)","Output",FontFamily->"Times-Plain",FontSize->10];
StylePrint["n = the number of divisions of the int edges (f.e. n=10;)","Output",FontFamily->"Times-Plain",FontSize->10];
StylePrint["tol = the accuracy of the root values (f.e. 0.1;)","Output",FontFamily->"Times-Plain",FontSize->10];
StylePrint["Zeros[sys, unk, int, n, tol] ","Output",FontFamily->"Times-Plain",FontSize->10];
]


Zeros[sys_, unk_, int_, n_, tol_] :=
Module[{l, x, ind, s, sol, w, solpar, f, g, z, z1, ra},
l =Length[unk];
pa= (int[[h,2]] - int[[h,1]])/n;
x[h_, j_] :=int[[h,1]] + (j - 1)*(int[[h,2]] - int[[h,1]])/n;
ind = Table[{it[h], 1, n + 1}, {h, 1, l}];

s[{uu__},{rr__}] := Table[Table[FindRoot[sys,uu][[h,2]],{h,1,l}],rr];
sol =s[Table[{unk[[h]], x[h, it[h]]},{h, 1, l}], ind];
sol1=Flatten[sol,l-1];

(*Roots in the domain*)
cont=Table[Table[If[int[[h,1]]<=sol1[[j]][[h]]<=int[[h,2]],sol1[[j]][[h]],q],
     {h,1,l}], {j,1,Length[sol1]}];
w=Join[Table[i,{i,1,Length[unk]}],Table[Length[unk],{3-Length[unk]}]];
test[u1_,u2_]:=Abs[u1[[w[[1]]]]-u2[[w[[1]]]]]<tol&&Abs[u1[[w[[2]]]]-
              u2[[w[[2]]]]]<tol&&Abs[u1[[w[[3]]]]-u2[[w[[3]]]]]<tol;
tfin=Cases[cont,Table[_Real,{Length[unk]}]];
ra=Union[tfin,SameTest->test];
rad=Chop[N[ra,3],10^-4];
Print[rad];
]


(*
 ----------------------------------------------------------------------------------------------
 -
 -   T H E   P R O G R A M  Taylor:
 -
 -                 UsageTaylor[]
 -                 HelpTaylor[]
 -                 Taylor[f_,var_,var0_,r_]
 -
 ----------------------------------------------------------------------------------------------
*)


UsageTaylor[]:=
Module[{},
StylePrint["Aims of the program Taylor","Output",FontFamily\[Rule]"Times-Bold",FontSize\[Rule]12];
StylePrint["The program gives the Taylor expansion","Output",FontFamily\[Rule]"Times-Plain",FontSize\[Rule]10];
StylePrint["\!\(\[Sum]\+\(i\_1 = 0\)\%r\)   \!\(\[Sum]\+\(i\_2 = 0\)\%i\_1\) \[CenterEllipsis]\!\(\[Sum]\+\(i\_n = 0\)\%\(i\_n - 1\)\)\!\(1\/\(\(\((i\_1 - i\_2)\)!\) \[CenterEllipsis] \(\((i\_\(n - 1\) - i\_n)\)!\) \(i\_n!\)\)\)\!\(\((\[PartialD]\^i\_1 f\/\(\[PartialD]\^\((i\_1 - i\_2)\)x\_1  \[CenterEllipsis] \[PartialD]\^\(i\_n \)x\_n\))\)\_0\)\!\(\((x\_1 - x\_1\^0)\)\^\((i\_1 - i\_2)\)\)\[CenterEllipsis](\!\(x\_n\)-\!\(x\_n\^0\)\!\(\()\^i\_n\)\)","Output",FontFamily\[Rule]"Times-Plain",FontSize\[Rule]10];
StylePrint["of a function f depending on the variables \!\(x\_1\), ..., \!\(x\_n\).","Output",FontFamily\[Rule]"Times-Plain",FontSize\[Rule]10];
StylePrint["The command line is","Output",FontFamily->"Times-Plain",FontSize->10];
StylePrint["Taylor[f, var, var0, r]","Output",FontFamily->"Times-Plain",FontSize->10];
StylePrint["where f is the function of the variable var, var0 is the initial point of Taylor expansion, and r is the approximation order.","Output",FontFamily->"Times-Plain",FontSize->10];
]

HelpTaylor[]:=
Module[{},
StylePrint["How to use the program Taylor","Output",FontFamily->"Times-Bold",FontSize->12];
StylePrint["The program gives the Taylor expansion of the function f","Output",FontFamily->"Times-Plain",FontSize->10];
StylePrint["f = the function to expand (f.e. Sin[x*y];)","Output",FontFamily->"Times-Plain",FontSize->10];
StylePrint["var = the list of independent variables (f.e.{x, y};)","Output",FontFamily->"Times-Plain",FontSize->10];
StylePrint["var0 = the initial point (f.e. {2, 1};)","Output",FontFamily->"Times-Plain",FontSize->10];
StylePrint["r = the approximation order (f.e. 3;)","Output",FontFamily->"Times-Plain",FontSize->10];
StylePrint["Taylor[f, var, var0, r]","Output",FontFamily->"Times-Plain",FontSize->10];
]


Taylor[f_,var_,var0_,r_]:=
Module[{n,m,v,c,g,h,l,fac,pr,dein,ap,de,sost},
n=Length[var];
m=Join[Table[in[i],{i,1,n}],{0}];
v1=Table[{m[[i]]-m[[i+1]]},{i,1,n}]//Flatten;
v=Table[{m[[i]],0,c[i]},{i,1,n}];
c[1]=r;
c[i_]:=m[[i-1]]/;
i>1;
g[w1_,{z1__}]:=Flatten[Table[w1,z1],n-1];
h=g[v1,v];
l=Length[h];
fac[i_]:=Product[Factorial[h[[i,j]]],{j,1,Length[h[[i]]]}];
pr[i_]:=Product[(var[[j]]-var0[[j]])^h[[i,j]],{j,1,n}];
dein[i_]:=Table[{var[[j]],h[[i,j]]},{j,1,n}];
ap[{q__}]:=q;
de[i_]:=D[f,ap[dein[i]]];
sost=Table[var[[i]]->var0[[i]],{i,1,n}];
tay=Sum[(1/fac[i])*(de[i]/.sost)*pr[i],{i,1,l}];
Print[f," = ",tay,"+o[",r+1,"]"]
]


(*
 ----------------------------------------------------------------------------------------------
 -
 -       THE  SUBROUTINE  TaylorCM:
 -
 -                 TaylorCM[f_,x_,x0_,r_]
 -
 ----------------------------------------------------------------------------------------------
*)


TaylorCM[f_,x_,x0_,r_]:=
Module[{n,m,v,c,g,h,l,fac,pr,dein,ap,de,sost},
n=Length[x];
m=Join[Table[in[i],{i,1,n}],{0}];
v1=Table[{m[[i]]-m[[i+1]]},{i,1,n}]//Flatten;
v=Table[{m[[i]],0,c[i]},{i,1,n}];
c[1]=r;
c[i_]:=m[[i-1]]/;
i>1;
g[w1_,{z1__}]:=Flatten[Table[w1,z1],n-1];
h=g[v1,v];
l=Length[h];
fac[i_]:=Product[Factorial[h[[i,j]]],{j,1,Length[h[[i]]]}];
pr[i_]:=Product[(x[[j]]-x0[[j]])^h[[i,j]],{j,1,n}];
dein[i_]:=Table[{x[[j]],h[[i,j]]},{j,1,n}];
ap[{q__}]:=q;
de[i_]:=D[f,ap[dein[i]]];
sost=Table[x[[i]]->x0[[i]],{i,1,n}];
tay=Sum[(1/fac[i])*(de[i]/.sost)*pr[i],{i,1,l}];
]


(*
 ----------------------------------------------------------------------------------------------
 -
 -   T H E   P R O G R A M  Poinsot:
 -
 -                 UsagePoinsot[]
 -                 HelpPoinsot[]
 -                 Poinsot[A_, B_, C0_, r0_, theta0_, b_, alpha_,steps_]
 -
 ----------------------------------------------------------------------------------------------
*)


UsagePoinsot[]:=
Module[{},
StylePrint["Aims of the program Poinsot","Output",FontFamily->"Times-Bold",FontSize->12];
StylePrint["The program gives the free motions of a solid S with a fixed point O.","Output",FontFamily->"Times-Plain",FontSize->10];
StylePrint["The command line writes","Output",FontFamily->"Times-Plain",FontSize->10];
StylePrint["Poinsot[A, B, C0, \!\(r\_0\), \!\(\[Theta]\_0\), b, \[Alpha], steps]","Output",FontFamily->"Times-Plain",FontSize->10];
StylePrint["where A, B, C0 are the eigenvalues of the tensor of inertia relative to O, \!\(r\_0\) is the component of the angular velocity along the Oz'-axis of the body frame, \!\(\[Theta]\_0\) is the initial nutation angle, b is a real number fixing the time interval, \[Alpha] is the angle determining the part of ellipsoid of inertia plotted with the polhode, and steps is the number of steps in the numeric integration.","Output",FontFamily->"Times-Plain",FontSize->10];
]

HelpPoinsot[]:=
Module[{},
StylePrint["How to use the program Poinsot","Output",FontFamily->"Times-Bold",FontSize->12];
StylePrint["The program gives the free motions of a solid with a fixed point O, in particular the plots of the herpolhode and polhode. ", "Output", FontFamily -> "Times-Plain", FontSize -> 10];
StylePrint["To make active the program type in the following input data", "Output", FontFamily -> "Times-Plain", FontSize -> 10];
StylePrint["A, B, C0 = the eigenvalues of the tensor of inertia relative to O (f.e. A = 0.5; B = 1; C0 = 1.5;)", "Output", FontFamily -> "Times-Plain", FontSize -> 10];
StylePrint["\!\(r\_0\) = the component of the angular velocity along the Oz'-axis of the body frame (f.e. \!\(r\_0\) = 3;)", "Output", FontFamily -> "Times-Plain", FontSize -> 10];
StylePrint["\!\(\[Theta]\_0\) = the initial nutation angle (f.e. \!\(\[Theta]\_0\) = \[Pi]/4;)", "Output", FontFamily -> "Times-Plain", FontSize -> 10];
StylePrint["b = the real number fixing the time interval (f.e. b = 3;)", "Output", FontFamily -> "Times-Plain", FontSize -> 10];
StylePrint["\[Alpha] = the angle determining the part of ellipsoid of inertia plotted with the polhode (f.e. \[Alpha] = \[Pi]/2;)", "Output", FontFamily -> "Times-Plain", FontSize -> 10];
StylePrint["steps = the number ot steps in the numeric integration (f.e. steps = 1000;)", "Output", FontFamily -> "Times-Plain", FontSize -> 10];
StylePrint["The command line is", "Output", FontFamily -> "Times-Plain", FontSize -> 10];
StylePrint["Poinsot[A, B, C0, \!\(r\_0\), \!\(\[Theta]\_0\), b, \[Alpha], steps]", "Output", FontFamily -> "Times-Plain", FontSize -> 10];
]


Poinsot[A_, B_, C0_, r0_, theta0_, b_, alpha_,steps_] :=
 Module[{q0, trif, K2, TP, hP, eq,solP,c,z,X,Y,Z,e},
     q0 = (C0*r0*Tan[theta0])/B;

(*trif = (2*Pi*B)/(r0*(A*Cos[theta0] + C0*Sin[theta0]*Tan[theta0]));*)
     K2 = (B*q0)^2 + (C0*r0)^2;
     TP = 1/2*(B*q0^2 + C0*r0^2);
     hP = Sqrt[(2*TP)/K2];
     eq1P = Derivative[1][p][t] == ((B - C0)*q[t]*r[t])/A;
     eq2P = Derivative[1][q][t] == ((C0 - A)*p[t]*r[t])/B;
     eq3P = Derivative[1][r][t] == ((A - B)*p[t]*q[t])/C0;
     eq4P = Derivative[1][psi][t] ==
       (Cos[phi[t]]*q[t] + p[t]*Sin[phi[t]])/Sin[theta[t]];
     eq5P = Derivative[1][phi][t] ==
       r[t] - Cot[theta[t]]*(Cos[phi[t]]*q[t] + p[t]*Sin[phi[t]]);
     eq6P = Derivative[1][theta][t] == p[t]*Cos[phi[t]] - q[t]*Sin[phi[t]];
     w1P = (Cos[phi[t]]*Cos[psi[t]] - Sin[phi[t]]*Sin[psi[t]]*Cos[theta[t]])*
        p[t] - (Cos[psi[t]]*Sin[phi[t]] +
          Cos[phi[t]]*Cos[theta[t]]*Sin[psi[t]])*q[t] +
       r[t]*Sin[psi[t]]*Sin[theta[t]];
     w2P = (Cos[psi[t]]*Cos[theta[t]]*Sin[phi[t]] + Cos[phi[t]]*Sin[psi[t]])*
        p[t] + (Cos[phi[t]]*Cos[psi[t]]*Cos[theta[t]] -
          Sin[phi[t]]*Sin[psi[t]])*q[t] - Cos[psi[t]]*r[t]*Sin[theta[t]];
     w3P = Cos[theta[t]]*r[t] + Cos[phi[t]]*q[t]*Sin[theta[t]] +
       p[t]*Sin[phi[t]]*Sin[theta[t]];
     solP = NDSolve[{eq1P, eq2P, eq3P, eq4P, eq5P, eq6P, p[0] == 0, q[0] == q0,
        r[0] == r0, psi[0] == 0, phi[0] == 0, theta[0] == theta0},
       {p, q, r, psi, phi, theta}, {t, 0, b},MaxSteps->steps];
     {x, y} = Flatten[{-((w1P*hP)/w3P), -((w2P*hP)/w3P)} /. solP];
     z = x^2 + y^2;
     m = FindMinimum[z, {t, 0.1}];
     M = FindMinimum[-z, {t, 0.1}];
     ra1 = Sqrt[m[[1]]];
     ra2 = Sqrt[-M[[1]]];
     StylePrint["The herpolhode is contained in an annulus having internal radius ra1 and external radius ra2, where","Output",FontFamily->"Times-Plain",
      FontSize->12];
     Print["ra1 = ", ra1",      ra2 = ",ra2];
     c1 = ParametricPlot[{ra1*Sin[u], ra1*Cos[u]}, {u, 0, 2*Pi},
       AspectRatio -> 1, DisplayFunction -> Identity,
       PlotStyle -> RGBColor[0.8669, 0.258, 0.227]];
     c2 = ParametricPlot[{ra2*Sin[u], ra2*Cos[u]}, {u, 0, 2*Pi},
       AspectRatio -> 1, DisplayFunction -> Identity,
       PlotStyle -> RGBColor[0.925, 0.14, 0.129]];
       StylePrint["The distance ra of the herpolhoid versus time","Output",FontFamily->"Times-Plain",FontSize->12];
       If[A==B||A==C0||B==C0,Goto[1],Goto[2]];
       Label[1];
       Plot[r0,{t, 0, b}, AxesLabel -> {"t","ra"},AxesOrigin->{0,0}];
       Goto[3];
       Label[2];
     Plot[Sqrt[z], {t, 0, b}, AxesLabel -> {"t", "ra"}];
     Label[3];
     erp = ParametricPlot[{x, y}, {t, 0, b}, AspectRatio -> 1,
       PlotRange -> All, DisplayFunction -> Identity];
       StylePrint["Herpolhode","Output",FontFamily->"Times-Plain",FontSize->12];
     Show[erp, c1, c2, DisplayFunction -> $DisplayFunction];
     Label[3]; xp = p[t]/Sqrt[2*TP] /. solP;
     yp = q[t]/Sqrt[2*TP] /. solP;
     zp = r[t]/Sqrt[2*TP] /. solP;
     X = (Cos[u]*Sin[v])/Sqrt[A];
     Y = (Sin[u]*Sin[v])/Sqrt[B];
     Z = Cos[v]/Sqrt[C0];
     el =
      ParametricPlot3D[{X, Y, Z}, {u, 0, 2*Pi}, {v, 0, alpha},
       LightSources -> {{{-1, -1, 3}, GrayLevel[0.999]}}, Boxed -> False,
       DisplayFunction -> Identity];
     pol = ParametricPlot3D[Evaluate[Flatten[{xp, yp, zp}] /. solP],
       {t, 0, b}, PlotPoints -> 200, DisplayFunction -> Identity];
       StylePrint["The polhode on the ellipsoid of inertia","Output",
      FontFamily->"Times-Plain",FontSize->12];
     Show[el, pol, DisplayFunction -> $DisplayFunction];
     ClearAll[X];
]


(*
 ----------------------------------------------------------------------------------------------
 -
 -   T H E   P R O G R A M  Solid:
 -
 -                 UsageSolid[]
 -                 HelpSolid[]
 -                 Solid[A_, B_, C0_, {psi_,phi_,theta_,t_},z_, F_, p0_, q0_, r0_,
 -                       psi0_, phi0_, theta0_, T_, steps_]
 -
 ----------------------------------------------------------------------------------------------
*)


UsageSolid[]:=
Module[{},
StylePrint["The aims of the program Solid","Output",FontFamily->"Times-Bold",FontSize->12];
StylePrint["The program gives the motions of a solid with a fixed point O. In particular it supplies the plots of Euler's angles versus time and the curve \[Gamma] that the gyroscopic axis draws on the unit sphere with the center at O.", "Output", FontFamily -> "Times-Plain", FontSize -> 10];
StylePrint["The command line is", "Output", FontFamily -> "Times-Plain", FontSize -> 10];
StylePrint["Solid[A, B, C0, var, z, F, \!\(p\_0\), \!\(q\_0\), \!\(r\_0\), \!\(\[Psi]\_0\), \!\(\[CurlyPhi]\_0\), \!\(\[Theta]\_0\), T, steps]", "Output", FontFamily -> "Times-Plain", FontSize -> 10];
StylePrint["where A, B, and C0 are the moments of inertia of the solid with respect to the body axes Ox', Oy', Oz' respectively, z is the coordinate list in the body frame of the point at which the acting force is applied, var is the list of Euler's angles and time, F is the component list of the force in the lab frame, \!\(p\_0\), \!\(q\_0\), \!\(r\_0\) are the initial components of angular velocity in the body frame, \!\(\[Psi]\_0\), \!\(\[CurlyPhi]\_0\), \!\(\[Theta]\_0\) are the initial Euler's angles, T is the upper bound of the time interval (0, T), steps is the number of steps in the numeric integration.", "Output", FontFamily -> "Times-Plain", FontSize -> 10];
]

HelpSolid[]:=
Module[{},
StylePrint["How to use the program Solid","Output",FontFamily->"Times-Bold",FontSize->12];
StylePrint["The program gives the motions of a solid with a fixed point O.", "Output", FontFamily -> "Times-Plain", FontSize -> 10];
StylePrint["To make active the program type in the following input data", "Output", FontFamily -> "Times-Plain", FontSize -> 10];
StylePrint["A, B, C0 = eigenvalues of the tensor of inertia relative to O (f.e. A = 1; B = 0.8; C0 = 1.2;)", "Output", FontFamily -> "Times-Plain", FontSize -> 10];
StylePrint["z = the coordinate list in the body frame of the point at which the acting force is applied (f.e. z = {0, 0, 1};)", "Output", FontFamily -> "Times-Plain", FontSize -> 10];
StylePrint["var = the list of Euler's angles and time (f.e. va = {\[Psi], \[CurlyPhi], \[Theta], t};)", "Output", FontFamily -> "Times-Plain", FontSize -> 10];
StylePrint["F = the component list of the force in the lab frame (f.e. F = {0, 0, -1- Cos[ \[Theta] [t] ]};)", "Output", FontFamily -> "Times-Plain", FontSize -> 10];
StylePrint["\!\(p\_0\) = the component of angular velocity along the Ox'-axis of the body frame (f.e. \!\(p\_0\) = 0;)", "Output", FontFamily -> "Times-Plain", FontSize -> 10];
StylePrint["\!\(q\_0\) = the component of angular velocity along the Oy'-axis of the body frame (f.e. \!\(q\_0\) = 0;)", "Output", FontFamily -> "Times-Plain", FontSize -> 10];
StylePrint["\!\(r\_0\) = the component of angular velocity along the Oz'-axis of the body frame (f.e. \!\(r\_0\) = 6;)", "Output", FontFamily -> "Times-Plain", FontSize -> 10];
StylePrint["\!\(\[Psi]\_0\) = the initial precession angle (f.e. \!\(\[Psi]\_0\) = -\[Pi]/2;)", "Output", FontFamily -> "Times-Plain", FontSize -> 10];
StylePrint["\!\(\[CurlyPhi]\_0\) = the initial proper rotation angle (f.e. \!\(\[CurlyPhi]\_0\) = 0;)", "Output", FontFamily -> "Times-Plain", FontSize -> 10];
StylePrint["\!\(\[Theta]\_0\) = the initial nutation angle (f.e. \!\(\[Theta]\_0\) = \[Pi]/4;)", "Output", FontFamily -> "Times-Plain", FontSize -> 10];
StylePrint["T = the real number fixing the time interval (0, T) in which the motion is analyzed (f.e. T = 10;)", "Output", FontFamily -> "Times-Plain", FontSize -> 10];
StylePrint["steps = the number of steps in the numeric integration (f.e. steps = 3000;)", "Output", FontFamily -> "Times-Plain", FontSize -> 10];
StylePrint["Solid[A, B, C0, var, z, F, \!\(p\_0\), \!\(q\_0\), \!\(r\_0\), \!\(\[Psi]\_0\), \!\(\[CurlyPhi]\_0\), \!\(\[Theta]\_0\), T, steps]", "Output", FontFamily -> "Times-Plain", FontSize -> 10];
]


Solid[A_, B_, C0_, {psi_,phi_,theta_,t_},z_, F_, p0_, q0_, r0_, psi0_, phi0_,
   theta0_, T_, steps_] :=
 Module[{Q, F1, M, eq, sol, curva, gr},
  Q = {{Cos[phi[t]]*Cos[psi[t]] - Sin[phi[t]]*Sin[psi[t]]*Cos[theta[t]],
       Cos[phi[t]]*Sin[psi[t]] + Sin[phi[t]]*Cos[psi[t]]*Cos[theta[t]],
       Sin[phi[t]]*Sin[theta[t]]}, {-(Sin[phi[t]]*Cos[psi[t]] +
       Cos[phi[t]]*Sin[psi[t]]*Cos[theta[t]]), -Sin[phi[t]]*Sin[psi[t]] +
        Cos[phi[t]]*Cos[psi[t]]*Cos[theta[t]], Cos[phi[t]]*Sin[theta[t]]},
      {Sin[psi[t]]*Sin[theta[t]], -Cos[psi[t]]*Sin[theta[t]], Cos[theta[t]]}};
  F1 = Q . F;
  M = Cross[z, F1];
 eq1 = Derivative[1][p][t] == 1/A*(B - C0)*q[t]*r[t] + M[[1]]/A;
 eq2 = Derivative[1][q][t] == 1/B*(C0 - A)*p[t]*r[t] + M[[2]]/B;
 eq3 = Derivative[1][r][t] == 1/C0*(A - B)*p[t]*q[t] + M[[3]]/C0;
 eq4 = Derivative[1][psi][t] == (p[t]*Sin[phi[t]] + q[t]*Cos[phi[t]])/Sin[theta[t]];
 eq5 = Derivative[1][phi][t] ==  -((p[t]*Sin[phi[t]] + q[t]*Cos[phi[t]])*Cot[theta[t]]) + r[t];
 eq6 = Derivative[1][theta][t] == p[t]*Cos[phi[t]] - q[t]*Sin[phi[t]];

 sol = NDSolve[{eq1, eq2, eq3, eq4, eq5, eq6, p[0] == p0, q[0] == q0,
       r[0] == r0, psi[0] == psi0, phi[0] == phi0,theta[0] == theta0}, {p, q, r, psi, phi, theta},
      {t, 0, T}, MaxSteps -> steps];
 Plot[Evaluate[psi[t] /. sol]*180/Pi, {t, 0, T}, AxesLabel -> {"t", "\[Psi]"}];
 Plot[Evaluate[phi[t] /. sol]*180/Pi, {t, 0, T}, AxesLabel -> {"t", "\[CurlyPhi]"}];
 Plot[Evaluate[theta[t] /. sol]*180/Pi, {t, 0, T}, AxesLabel -> {"t", "\[CurlyTheta]"}];
 curva = ParametricPlot3D[Evaluate[{Cos[psi[t] - Pi/2]*Sin[theta[t]], Sin[psi[t] - Pi/2]*Sin[theta[t]],
         Cos[theta[t]]} /. sol], {t, 0, T}, PlotPoints -> 500,DisplayFunction -> Identity];
 gr = ParametricPlot3D[{Sin[u]*Sin[v], Cos[u]*Sin[v], Cos[v]}, {u, 0, 2*Pi}, {v, 0, Pi},
      LightSources -> {{{-1, -1, 3}, GrayLevel[0.999]}}, PlotPoints -> {20, 20}, DisplayFunction -> Identity];

Show[gr, curva, DisplayFunction -> $DisplayFunction];
]


(* = = = = = = = = = = = = = = =  E N D  P R O G R A M S  = = = = = = = = = = = = = = = = *)


End[]


(* = = = = = = = = = = = = = = =  E N D  P A C K A G E  = = = = = = = = = = = = = = = = = *)


EndPackage[]
