#include "ap.h"


// The bigint functions
// Bigints are simply arrays of unsigned integers, they can be seen as
// base-2^52 numbers. Arithmetic on them is quite fast.
// The least significant word is stored first.

const double eps = 1.0 / 4503599627370496.0;    // 2^-52

// Adds n words from s to d, returns overflow carry bit
rawtype bigadd (rawtype *d, rawtype *s, size_t n, size_t c)
{
    size_t t;
    rawtype r, b = 0.0;

    for (t = 0; t < n; t++)
    {
        r = *d + *s + b;

        if (r >= base)
        {
            r -= base;
            b = 1.0;
        }
        else b = 0.0;

        *d = r;

        d++;
        s++;
    }


    for (t = 0; t < c && b != 0.0; t++)
    {
        if ((*d += b) == base)
            *d -= base;
        else
            b = 0.0;

        d++;
    }

    return b;
}

// Subtracts n words s from d, returns subtract carry bit (1)
rawtype bigsub (rawtype *d, rawtype *s, size_t n, size_t c)
{
    size_t t;
    rawtype r, b = 0.0;

    for (t = 0; t < n; t++)
    {
        r = *d - *s - b;

        if (r < 0.0)
        {
            r += base;
            b = 1.0;
        }
        else b = 0.0;

        *d = r;

        d++;
        s++;
    }

    for (t = 0; t < c && b != 0.0; t++)
    {
        if ((*d -= b) == -1.0)
            *d += base;
        else
            b = 0.0;

        d++;
    }

    return b;
}

// Multiplicates n words in s by f, stores result in d, returns overflow word
rawtype bigmul (rawtype *d, rawtype *s, rawtype f, size_t n)
{
    size_t t;
    rawtype fl, fh, sl, sh, rl, rh, c;

    fh = quick_floor (invsqrtbase * f);
    fl = f - sqrtbase * fh;

    rl = 0.0;

    for (t = 0; t < n; t++)
    {
        sh = quick_floor (invsqrtbase * *s);
        sl = *s - sqrtbase * sh;

        c = fl * sh + fh * sl;
        rh = quick_floor (invsqrtbase * c);
        rl += sqrtbase * c - base * rh;
        rh += fh * sh;

        if (rl >= base)
        {
            rl -= base;
            rh += 1.0;
        }

        rl += fl * sl;

        if (rl >= base)
        {
            rl -= base;
            rh += 1.0;
        }

        *d = rl;
        rl = rh;

        d++;
        s++;
    }

    return rh;
}

// Divides n words in s by f, stores result in d, returns remainder
rawtype bigdiv (rawtype *d, rawtype *s, rawtype f, size_t n)
{
    size_t t;
    rawtype b, c, r, fl, fh, rl, rh, pl, ph;
    double invf = (1.0 - 2.0 * eps) / f;                // Round down

    if (!n) return 0.0;

    fh = quick_floor (invsqrtbase * f);
    fl = f - sqrtbase * fh;

    d += n;
    s += n;

    if ((b = *(s - 1)) < f)
    {
        d--;
        s--;
        n--;
        *d = 0;
    }
    else b = 0.0;

    for (t = 0; t < n; t++)
    {
        d--;
        s--;

        r = quick_floor (invf * (*s + base * b));       // Approximate Quotient

        ph = quick_floor (invsqrtbase * r);
        pl = r - sqrtbase * ph;

        c = fl * ph + fh * pl;
        rl = fl * pl;
        rh = quick_floor (invsqrtbase * c);
        rl += sqrtbase * c - base * rh;
        rh += fh * ph;

        if (rl >= base)
        {
            rl -= base;
            rh += 1.0;
        }

        rl = *s - rl;
        rh = b - rh;

        if (rl < 0.0)
        {
            rl += base;
            rh -= 1.0;
        }

        while (rh > 0.0 || rl >= f)                     // Quotient was too small
        {
            rl -= f;

            if (rl < 0.0)
            {
                rl += base;
                rh -= 1.0;
            }

            r += 1.0;
        }

        b = rl;                                         // Remainder
        *d = r;                                         // Quotient
    }

    return b;

}

// Shifts s right one bit to d, returns carry bit
int bigshr (rawtype *d, rawtype *s, size_t n)
{
    size_t t;
    rawtype tmp1, tmp2;
    int b = 0;

    if (!n) return 0;

    d += n;
    s += n;

    for (t = 0; t < n; t++)
    {
        d--;
        s--;

        tmp1 = *s * 0.5;
        tmp2 = quick_floor (tmp1);
        *d = tmp2 + (b ? halfbase : 0);
        b = (tmp1 == tmp2 ? 0 : 1);
    }

    return b;
}

// Compares n words, returns -1 if d < s, 1 if d > s, 0 if d == s
int bigcmp (rawtype *d, rawtype *s, size_t n)
{
    size_t t;

    d += n - 1;
    s += n - 1;

    for (t = 0; t < n; t++)
    {
        if (*d < *s) return -1;
        if (*d > *s) return 1;

        d--;
        s--;
    }

    return 0;
}
