#if !defined(__RAW_H)
#define __RAW_H


// The modint raw data type, unsigned if possible
typedef double rawtype;


const rawtype MAXMODULUS = 4503599627370495.0;
const rawtype MAXRAW = 4503599627370495.0;
const int RAWBITS = 52;

// Base (must be BASEDIGIT^BASEDIGITS = BASE < all MODULI)
const rawtype BASE = 1e15;
const rawtype BASEDIGIT = 10.0;
const int BASEDIGITS = 15;
const rawtype BASEFACTORS[] = {2.0, 5.0};       // All prime factors of BASE
const int NBASEFACTORS = 2;                     // Number of prime factors of BASE

const rawtype MODULI[3] = {4222124650659841.0, 3799912185593857.0, 1899956092796929.0};         // upto n = 3*2^46
const rawtype PRIMITIVEROOTS[3] = {19.0, 5.0, 7.0};
const rawtype WFTAMODULI[3] = {4194068336640001.0, 2742275450880001.0, 2661620290560001.0};     // upto n = 630*5040^3
const rawtype WFTAPRIMITIVEROOTS[3] = {17.0, 19.0, 43.0};

const size_t MAXTRANSFORMLENGTH = 105553116266496UL;    // Maximum transform length for these moduli

const rawtype base = 4503599627370496.0;                 // 2^52
const rawtype sqrtbase = 67108864.0;                     // 2^26
const rawtype invsqrtbase = 1.0 / 67108864.0;            // 2^-26
const rawtype halfbase = base * 0.5;                     // 2^51

extern rawtype imodulus;
extern rawtype lmodulus;
extern rawtype hmodulus;


// Low-level inline functions

#include <math.h>

// Implement your own quick floor function
#define quick_floor floor
// #define quick_floor (long long)


// Computes a * b % m
inline rawtype modmul (rawtype a, rawtype b, rawtype m)
{
    rawtype al, ah, bl, bh, rl, rh, pl, ph, c, cl, ch;

    ah = quick_floor (invsqrtbase * a);
    al = a - sqrtbase * ah;

    bh = quick_floor (invsqrtbase * b);
    bl = b - sqrtbase * bh;

    c = al * bh + ah * bl;
    rl = al * bl;
    rh = quick_floor (invsqrtbase * c);
    rl += sqrtbase * c - base * rh;
    rh += ah * bh;

    if (rl >= base)
    {
        rl -= base;
        rh += 1.0;
    }

    c = quick_floor (imodulus * a * b);

    ch = quick_floor (invsqrtbase * c);
    cl = c - sqrtbase * ch;

    pl = cl * lmodulus;
    c = cl * hmodulus + ch * lmodulus;
    ph = quick_floor (invsqrtbase * c);
    pl += sqrtbase * c - base * ph;
    ph += ch * hmodulus;

    if (pl >= base)
    {
        pl -= base;
        ph += 1.0;
    }

    rl -= pl;
    rh -= ph;

    if (rl < 0.0)
    {
        rl += base;
        rh -= 1.0;
    }

    rl += rh * (base - m);

    if (rl >= m)
        rl -= m;

    return rl;
}

inline rawtype modadd (rawtype a, rawtype b, rawtype m)
{
    rawtype r;

    r = a + b;
    return (r >= m ? r - m : r);
}

inline rawtype modsub (rawtype a, rawtype b, rawtype m)
{
    rawtype r;

    r = a - b;
    return (r < 0.0 ? r + m : r);
}


#endif  // __RAW_H
