/**********************************************************
*
* This ANSI C-Programm computes the p-th hexadecimal digit
* of pi (plus the following 7 digits).
*
* p comes from the command line.
* 1 <= p <= 2^29 (approx 536.8 million).
* p=1 means the first digit after the radix point.
*
*
* The program uses a "BBP"-formula of Gerhard Pschill.
*
**********************************************************/

#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <time.h>
#include <float.h>

/*  Limits:
   m_max = LONG_MAX = 2^31-1 from def'ion in series()
   m_max = 4*d_max+1 from 1st BBP series
   30*n_max = 4*d_max +1 // 30 = b[k] 
   n_max(1) = (2^31-1)/30
  
   a_max(1) = ULONG_MAX = 2^32-1 from def'ion in series()

   r_max^2 < 2^64 (length of unsigned long long = 64).
   therefore
   r_max   = 2^32-1
   a_max(2)   = r_max+1 = 2^32 

   a_max(3) = 60*n_max(1)+45 from 16th BBP series
            = 2^32 + 43
   therefore
   a_max = a_max(1) = 2^32-1
   n_max(2) = (a_max - 45)/ 60 from 6th series
            = (2^31-23)/30 = 2^26.093 < n_max(1)
   4*dmax+1 = 30*n_max(2) from term 2^(-30)
   d_max = 2^29-12
   p_max = d_max+1 = 2^29 -11 = 536 870 901
*/
#define p_max  ((1UL<<29)-11)

/* Global data */

/*
   BBPData:

   A BBP series has this form:
   
\pi = \sum_{k=1}^K { \sum_n=0^\infty {sign_k[n%2] * 2^{a_k + n*b_k} / (n*p_k + q_k)} }

*/

typedef struct{
   int    sign[2];
   int    a, b;
   int    p, q;
} _BBPData; 

_BBPData BBPData[] = 
   /* Pschill Formula (3) */
   {  { {+1, -1},  +1, -30, 20,  1 }
   ,  { {+1, -1},  -1, -30, 12,  1 }
   ,  { {+1, -1},  -1, -30, 10,  1 }
   ,  { {+1, -1},  -2, -30, 20,  3 }
   ,  { {+1, -1},  -4, -30,  6,  1 }
   ,  { {-1, +1},  -6, -30, 60, 15 }
   ,  { {-1, +1},  -7, -30, 10,  3 }
   ,  { {-1, +1},  -8, -30, 20,  7 }
   ,  { {-1, +1}, -11, -30, 12,  5 }
   ,  { {+1, -1}, -11, -30, 20,  9 }
   ,  { {+1, -1}, -14, -30, 30, 15 }
   ,  { {+1, -1}, -14, -30, 20, 11 }
   ,  { {-1, +1}, -16, -30, 12,  7 }
   ,  { {-1, +1}, -17, -30, 20, 13 }
   ,  { {-1, +1}, -19, -30, 10,  7 } 
   ,  { {-1, +1}, -21, -30, 60, 45 }
   ,  { {+1, -1}, -24, -30,  6,  5 }
   ,  { {+1, -1}, -23, -30, 20, 17 }
   ,  { {+1, -1}, -25, -30, 10,  9 }
   ,  { {+1, -1}, -26, -30, 12, 11 }
   ,  { {+1, -1}, -26, -30, 20, 19 }
   };
                
int    K   = sizeof(BBPData)/sizeof(_BBPData); 
long   d4;
double eps;

/**********************************************************
*
*       double expm(long m, long a)
*
*  Computes 2^m mod a using binary modulo exponentation.
*
**********************************************************/

double
expm(long m, unsigned long a)
{
   unsigned long long r = 2; // broadest possible data type
   long mt, m1;         // may be unsigned's

   if (a == 1)    return 0.0;
   if (m == 0)    return 1.0;
   if (m == 1)    return 2 % a;

   /*  make mt the largest power of 2 less than m/2. */
   mt = 1; m1 = m >> 1;
   while (m1 != 1) {
      mt += mt;
      m1 >>= 1;
   }
   /*  Binary modulo exponentiation modulo a */
   do
   {
      r = r*r % a;
      if ((m & mt) != 0)
         r = (r+r) % a;
      mt >>= 1;
   } while (mt != 0);
   return r;
}

/*****************************************************************
*
*  series(int k)
*
*  Computes the (k-th) sum
*  \sum_n=0^\infty {sign_k[n%2] * 2^{a_k + n*b_k} / (n*p_k + q_k)} 
*
******************************************************************/

double 
series(int k)
{
   _BBPData *pBBPData = BBPData+k;
   double s  = 0.0;
   unsigned long  a  = pBBPData->q;
   long   m  = d4 + pBBPData->a; // must be a signed 
   double x  = 0;
   int    l  = 0;

   for ( ; m > 0; m += pBBPData->b, l = !l) 
   {
      /* Modulo Exponentation */
      x   = expm(m, a) / (double)a;
      s  += pBBPData->sign[l] >= 0 ? +x : -x;
      s   = fmod(s, 1);
      a  += pBBPData->p;
   }

   /* Some additional terms for 8 hex digits accuracy */
   do
   {
      x = pow(2.0, m) / (double)a;
      s  += pBBPData->sign[l] >= 0 ? +x : -x;
      a  += pBBPData->p;
      m  += pBBPData->b;
      l = !l;
   } while (x >= eps);
   return s;
}

/**********************************************************
*
*              main()
*
**********************************************************/

#pragma argsused
int main(int argc, char *argv[])
{
   clock_t t_beg, t_end;
   long    p;
   double  S = 0.0;
   int     k;
   char    line[100];
   long    p_min = 1;

	printf("pschill: computes the n'th hexadecimal digit of pi.\n");
	printf("It uses a BBP series of Gerhard Pschill.\n");

   do
	{
      // try cmdline
	   if (argc > 1)
	      sscanf(argv[1], "%ld", &p);
	   else { // ask user
         printf("\nEnter n (or -1 to exit): ");
         gets(line);
         p = 0; 
         sscanf(line, "%ld", &p);
	   }
	   if (p < 0) break;
      if (p < p_min) p = p_min;   
   	if (p > p_max) p = p_max;
   	d4  = 4*(p-1);
      /* accuracy: 10 hex digits after point */
      eps = pow(16.0, -10.0);
      printf("Hexadecimal digits %lu to %lu of pi: ", p, p+7);
      t_beg=clock();
      for (k=0; k < K; ++k) {
         S += series(k);
      }
      S += (double)K;    /* ensure S >= 0 */
      S = fmod(S, 1.0);
   	t_end=clock();
      printf("%08lX\n", (unsigned long)(S*pow(2, 32)));
      printf("Time was: %.1f sec.\n\n",
                     1.0*(t_end-t_beg)/CLOCKS_PER_SEC);
   } while (argc <= 1);
   return 0;
}


