
#include <iostream.h>
#include <math.h>
#include <stdlib.h> // random()

#include "complextype.h"
#include "sincos.h"
#include "minmax.h"  // min_max()
#include "fxttypes.h"
#include "fxtdefs.h"  // SUMDIFF, CSQR, CMULT


double
rnd01()
// random number in [0,1[
{
    return  (double)random() * (1.0/(RAND_MAX-1));
}
// ----------------


double
white_noise()
// returns one sample of white noise (with mean=0, sigma=1)
{
    static double z2;
    static unsigned ct = 0;

    ++ct;
    if ( ct & 1 )  // compute two independent samples:
    {
        double r1 = sqrt( -2 * log(1.0-rnd01()) );  // arg of log != 0
        double r2 = 2.0 * M_PI * rnd01();
        double z1;
        sincos(r2, &z1, &z2);
        z1 *= r1;  // z1 = r1 * sin(r2)
        z2 *= r1;  // z2 = r1 * cos(r2)
        return  z1;
    }
    else  // ... next time there is one for free:
    {
        return  z2;
    }
}
// ----------------


double
pythag(double a, double b)
// return sqrt(a*a+b*b)
{
    a = fabs(a);
    b = fabs(b);
    if ( a>b )
    {
        double v = b/a;
        return  a * sqrt(1.0+v*v);
    }
    else
    {
        if ( b==0.0 )  return 0.0;
        double v = a/b;
        return  b * sqrt(1.0+v*v);
    }
}
// ----------------


double
extr3_estimate(const double *f, double *xvalp/*=0*/)
// return extremum (y-value) of parabola through 
// the points (-1,a), (0,b), (+1,c)
// optional also get the x-value
{
    double a = f[-1],  b = f[0],  c = f[1];
    double xval = 0.5*(a-c)/(a+c-b*2);   // position of maximum
    if ( 0!=xvalp )  *xvalp = xval;
    double yval = b-0.25*(a-c)*xval;  // value of maximum
    return  yval;
}
//-----------------------


double
sinc(double x)
// return sin(pi*x)/(pi*x)
{
    if ( x==0 )  return  1.0;
    else         return  sin(M_PI*x)*M_1_PI/x;  
}
// ----------------


void
sumdiff(double *x, double *y, ulong n)
// x[]=(x[]+y[]),  y[]=(x[]+y[])
{
    double *xp = x + n;
    double *yp = y + n;
    while ( n-- )
    {
        --xp;
        --yp;
        SUMDIFF2(*xp, *yp);
    }
}
//-----------------------


void
sumdiff_05(double *x, double *y, ulong n)
// x[]=(x[]+y[])*0.5,  y[]=(x[]+y[])*0.5
{
    double *xp = x + n;
    double *yp = y + n;
    while ( n-- )
    {
        --xp;
        --yp;
        SUMDIFF2_05(*xp, *yp);
    }
}
//-----------------------


double
norm(const double *f, ulong n)
// return sqrt(\sum_k{f[k]*f[k]})
{
    double v = 0.0;
    while ( n-- )  v += (f[n]*f[n]);
    return  sqrt(v);
}
//-----------------------


void
normalize(double *f, ulong n, double v/*=1.0*/)
// scale f[] to norm v
{
    double a = norm(f, n);
    v /= a;
    while ( n-- )  f[n] *= v;
}
//-----------------------


double
scalar_product(const double *f, const double *g, ulong n)
// return \sum_k{f[k]*g[k]}
{
    double p=0;
    for (ulong i=0; i<n; ++i)  p += f[i] * g[i];
    return  p;
}
//-----------------------


double
mean(const double *f, ulong n)
// return (\sum_k{f[k]})/n
{
    double m=0;
    for (ulong i=0; i<n; ++i)	m += f[i];
    m /= (double)n;
    return  m;
}
//-----------------------

double
subtract_mean(double *f, ulong n, double *dst=0)
// subtract from f[] its mean value
{
    if ( !dst )  dst = f;  // default dst = source
    double m = mean(f,n);
    for (ulong i=0; i<n; ++i)  dst[i] = f[i] - m;
    return  m;
}
//-----------------------


double
sigma(const double *f, ulong n, double *mp/*=0*/)
// return standard deviation
// = sqrt(\sum_k{sqr(f[k]-mean(f[]))}/n)
{
    double m;
    if ( 0!=mp )  m = *mp;
    else             m = mean(f,n);

    double s = 0;
    for (ulong i=0; i<n; ++i)
    {
	double t = f[i] - m;
	s += t*t;
    }
    return  sqrt(s/n);
}
//-----------------------

void
mean_sigma(const double *f, ulong n, double *mp, double *sp)
// set *mp to mean, sp *to standard deviation of f[]
{
    *mp = mean(f,n);
    *sp = sigma(f,n,mp);
}
//-----------------------


void
smooth(double *f, ulong n, ulong m/*=1*/)
//  apply (1/4,1/2,1/4) - filter
{
    if ( n<=2 )  return;

    for (ulong k=0; k<m; ++k)
    {
        double t1 = f[0], t2;
        for (ulong i=1; i<=n-2; ++i)
        {
            t2 = t1;
            t1 = 0.25*(f[i-1]+f[i+1]) + 0.5*f[i];
            f[i-1] = t2;
        }
        f[n-2] = t1;
    }
}
//-----------------------


double
rms_diff(const double *f, const double *g, ulong n)
// return sqrt(\sum_k{sqr(f[k]-g[k])})/n
{
    double err = 0;
    for (ulong i=0; i<n; ++i)
    {
        double d = f[i]-g[i];
        err += d*d;
    }

    return  sqrt(err)/n;
}
//-----------------------

double
rms_diff(const Complex *f, const Complex *g, ulong n)
// return sqrt(\sum_k{sqr(abs(f[k]-g[k]))})/n
{
    return  rms_diff((double *)f, (double *)g, 2*n);
}
//-----------------------


ulong *
histogram(const double *f, ulong n, ulong nbox, ulong *hh/*=0*/)
// put histogram of f[] (with nbox boxes) into hh[]
{
    ulong *h;
    if ( 0==hh )  h = new ulong[nbox];
    else             h = hh;
    for (ulong i=0; i<nbox; ++i)  h[i] = 0;

    double mi,ma;
    min_max(f, n, &mi, &ma);
    if ( ma==mi )
    {
        h[0] = n;
        return  h;
    }

    double v = 1.0*nbox/(ma-mi);

    for (ulong i=0; i<n; ++i)  ++h[(int)floor(v*(f[i]-mi))];

    return  h;
}
//-----------------------


void
welch_win(double *f, ulong n)
// multiply f[] with a welch window
{
    const double pin = M_PI/n;
    for (ulong i=0; i<n; i++)  f[i] *= (M_SQRT2*sin(i*pin));
}
//-----------------------


void
ri_multiply(const double *fr, const double *fi,
            double *gr, double *gi, ulong n)
// complex multiply array g[] by f[]:
//  Complex(gr[],gi[]) *= Complex(fr[],fi[])
{
    while ( n-- )  CMULT(fr[n], fi[n], gr[n], gi[n]);
}
//-----------------------

