#if !defined __AUXBIT_H
#define      __AUXBIT_H

#include "fxttypes.h"

#include <limits.h>


#if  ( 4294967295UL==ULONG_MAX )  // long is 32bit
#define  BITS_PER_LONG  32
#else
#define  BITS_PER_LONG  64
#endif

// cf.:
//  http://www.snippets.org/
//  http://www.mathematik.uni-bielefeld.de/~sillke/ALGORITHMS/bitmani/


inline ulong rotate_left(ulong x, ulong s)
{
    return  (x<<s) | (x>>(BITS_PER_LONG-s));
}
//------------------------


inline ulong rotate_right(ulong x, ulong s)
{
    return  (x>>s) | (x<<(BITS_PER_LONG-s));
}
//------------------------


inline ulong graycode(ulong x)
{
    return  x ^ (x>>1);
}
//------------------------


inline ulong inverse_graycode(ulong x)
// inverse if graycode()
{
// ----- version 1:
//    ulong h=1, r=0;
//    do
//    {
//        if ( x & 1 )  r^=h;
//        x >>= 1;
//        h = (h<<1)+1;
//    }
//    while ( x!=0 );
//    return r;

// ----- version 2:
    x ^= x>>1;
    x ^= x>>2;
    x ^= x>>4;
    x ^= x>>8;
    x ^= x>>16;
#if  BITS_PER_LONG==64
    x ^= x>>32;  // for 64bit words
#endif
    return x;
}
//------------------------


inline ulong parity(ulong x)
{
    x ^= x>>1;
    x ^= x>>2;
    x ^= x>>4;
    x ^= x>>8;
    x ^= x>>16;
#if  BITS_PER_LONG==64
    x ^= x>>32;  // for 64bit words
#endif
    return  x & 1;
}
//------------------------


inline ulong bit_count(ulong x)
// return number of bits set
{
#if  BITS_PER_LONG==32
//    x = (0x55555555 & x) + (0x55555555 & (x>> 1));  // 0-2 in 2 bits
//    x = (0x33333333 & x) + (0x33333333 & (x>> 2));  // 0-4 in 4 bits
//    x = (0x0f0f0f0f & x) + (0x0f0f0f0f & (x>> 4));  // 0-8 in 8 bits
//    x = (0x00ff00ff & x) + (0x00ff00ff & (x>> 8));  // 0-16 in 16 bits
//    x = (0x0000ffff & x) + (0x0000ffff & (x>>16));  // 0-31 in 32 bits
//    return x;

    x = ((x>> 1) & 0x55555555) + (x & 0x55555555);  // 0-2 in 2 bits
    x = ((x>> 2) & 0x33333333) + (x & 0x33333333);  // 0-4 in 4 bits
    x = ((x>> 4) + x) & 0x0f0f0f0f;                 // 0-8 in 8 bits
    x +=  x>> 8;                                    // 0-16 in 8 bits
    x +=  x>>16;                                    // 0-32 in 8 bits
    return  x & 0xff;
#else
//    x = (0x5555555555555555 & x) + (0x5555555555555555 & (x>> 1));
//    x = (0x3333333333333333 & x) + (0x3333333333333333 & (x>> 2));
//    x = (0x0f0f0f0f0f0f0f0f & x) + (0x0f0f0f0f0f0f0f0f & (x>> 4));
//    x = (0x00ff00ff00ff00ff & x) + (0x00ff00ff00ff00ff & (x>> 8));
//    x = (0x0000ffff0000ffff & x) + (0x0000ffff0000ffff & (x>>16));
//    x = (0x00000000ffffffff & x) + (0x00000000ffffffff & (x>>32));
//    return x;

    x = ((x>> 1) & 0x5555555555555555) + (x & 0x5555555555555555);  // 0-2 in 2 bits
    x = ((x>> 2) & 0x3333333333333333) + (x & 0x3333333333333333);  // 0-4 in 4 bits
    x = ((x>> 4) + x) & 0x0f0f0f0f0f0f0f0f;                         // 0-8 in 8 bits
    x +=  x>> 8;                                                    // 0-16 in 8 bits
    x +=  x>>16;                                                    // 0-32 in 8 bits
    x +=  x>>32;                                                    // 0-64 in 8 bits
    return  x & 0xff;
#endif
}
//------------------------


inline ulong bit_count_sparse(ulong x)
// return number of bits set
// the loop will execute once for each bit of x set
{
    if ( 0==x )  return 0;

    ulong n = 0;
    do  { ++n; }
    while ( (x = x & (x-1)) );

    return  n;
}
//------------------------


inline ulong highest_bit(ulong x)
// return index of highest bit set
// return ~0 if no bit is set
{
    if ( 0==x )  return  (ulong)~0;

    ulong r = 0;
#if  BITS_PER_LONG==64
    if ( x & 0xffffffff00000000 )  { x >>= 32;  r += 32; }  // for 64bit words
#endif
    if ( x & 0xffff0000 )  { x >>= 16;  r += 16; }
    if ( x & 0x0000ff00 )  { x >>=  8;  r +=  8; }
    if ( x & 0x000000f0 )  { x >>=  4;  r +=  4; }
    if ( x & 0x0000000c )  { x >>=  2;  r +=  2; }
    if ( x & 0x00000002 )  {            r +=  1; }
    return r;
}
//------------------------


inline ulong lowest_bit(ulong x)
// return index of lowest bit set
// return ~0 if no bit is set
{
    if ( 0==x )  return  (ulong)~0;

    ulong r = BITS_PER_LONG - 1;
#if  BITS_PER_LONG==64
    if ( x & 0x00000000ffffffff )  r  -= 32;
    else                           x >>= 32;
#endif
    if ( x & 0x0000ffff )  r  -= 16;
    else                   x >>= 16;

    if ( x & 0x000000ff )  r  -= 8;
    else                   x >>= 8;

    if ( x & 0x0000000f )  r  -= 4;
    else                   x >>= 4;

    if ( x & 0x00000003 )  r  -= 2;
    else                   x >>= 2;

    if ( x & 0x00000001 )  r -= 1;

    return r;
}
//------------------------


inline ulong ld(ulong x)
// returns k so that 2^k <= x < 2^(k+1)
{
//    ulong k = 0;
//    while ( x>>=1 )  ++k;
//    return k;

    if ( 0==x )  return  0;
    ulong r = 0;
#if  BITS_PER_LONG==64
    if ( x & 0xffffffff00000000 )  { x >>= 32;  r += 32; }  // for 64bit words
#endif
    if ( x & 0xffff0000 )  { x >>= 16;  r += 16; }
    if ( x & 0x0000ff00 )  { x >>=  8;  r +=  8; }
    if ( x & 0x000000f0 )  { x >>=  4;  r +=  4; }
    if ( x & 0x0000000c )  { x >>=  2;  r +=  2; }
    if ( x & 0x00000002 )  {            r +=  1; }
    return r;
}
//------------------------


inline ulong ldq(ulong x)
// return k if x=2**k
// else return ~0
{
    ulong k = ld(x);
    if ( x==(1UL<<k) )  return  k;
    else                return  (ulong)~0;
}
//------------------------

inline unsigned long long ld(unsigned long long x)
// returns k so that 2^k <= x < 2^(k+1)
{
    if ( 0==x )  return  0;
    unsigned long long r = 0;
    if ( x & 0xffffffff00000000ULL )  { x >>= 32;  r += 32; }
    if ( x & 0xffff0000 )  { x >>= 16;  r += 16; }
    if ( x & 0x0000ff00 )  { x >>=  8;  r +=  8; }
    if ( x & 0x000000f0 )  { x >>=  4;  r +=  4; }
    if ( x & 0x0000000c )  { x >>=  2;  r +=  2; }
    if ( x & 0x00000002 )  {            r +=  1; }
    return r;
}
//------------------------


inline int is_pow_of_2(ulong x)
// return 1 if x == 0(!) or x == 2**k
{
    return  (x & -x) == x;
}
//------------------------

inline int one_bit_q(ulong x)
// return 1 iff x \in {1,2,4,8,16,...}
{
    ulong m = x-1;
    return  (x^m)>>1 == m;
}
//------------------------



inline ulong next_pow_of_2(ulong x)
// return x if x=2**k
// else return 2**ceil(log_2(x))
{
    ulong n = (ulong)1<<ld(x);  // n<=x
    if ( n==x )  return x;
    else         return n<<1;
}
//------------------------

inline ulong next_exp_of_2(ulong x)
// return k if x=2**k
// else return k+1
{
    ulong ldx = ld(x);
    ulong n = (ulong)1<<ldx;  // n<=x
    if ( n==x )  return ldx;
    else         return ldx+1;
}
//------------------------


inline ulong next_comb(ulong w)
// return smallest integer greater than u with the same number of bits set.
//
//  examples (binary notation):
//    1 -> 10 -> 100 -> 1000 -> 10000 -> 100000 -> 1000000 -> ...
//    11 -> 101 -> 110 -> 1001 -> 1010 -> 1100 -> 10001 -> ...
//    111 -> 1011 -> 1101 -> 1110 -> 10011 -> 10101 -> 10110 -> ...
//
//  special cases:
//    0 -> 0
//    if the number is too large then 0 is returned.
//
//  The bit pattern to change at the end of the word is:
//    0 1   0   ->    1 0   1
//       a   b           b+1 a-1
//
//  Example: (the bits marked by B remain fixed)
//    w       = BBBB01111100000000
//    z       = 000000000100000000   (the lowest set bit)
//    w+z     = BBBB10000000000000
//    (w+z)^w = 000011111100000000
//    next    = BBBB10000000001111
//
//  this was shamelessly stolen from Torsten Sillke's bitman.h
//  (the algorithm can be found in hakmem)
{
    if ( w==0 )  return  0;

    ulong z = ( ((w-1)^w) >> 1 )+1; // lowest set bit
    ulong v = w + z;
    if ( v )  v += ((v^w)/z >> 2);

    return  v;  // ==0 if input was the biggest combo with that number of bits
}
//------------------------

#endif  // !defined __AUXBIT_H
