
#include <math.h>

#include "sincos.h"
#include "fxtdefs.h"  // SUMDIFF, CSQR, CMULT
#include "permute.h"

// tuning parameter:
//#define USE_SINCOS3  // default = off
// whether sincos is used for 3*angle
// else: use algebraic relation


void
split_radix_fft_dif_core(double *fr, double *fi, ulong ldn)
//
// split-radix decimation in frequency fft
// output data is in revbin_permuted order
//.
// cf: Duhamel-Hollman, Electronics Letters, Jan. 5, 1984
// orig by: Dave Edelblute, edelblut@cod.nosc.mil, 05 Jan 1993
// modified: R. Mayer
// heavily modified: Joerg Arndt, arndt@jjj.de
//
{
    const ulong n = (1<<ldn);
    if ( n<=1 )  return;

    ulong n2 = 2*n;
    for (ulong k=1; k<=ldn; k++)
    {
        n2 >>= 1;
        ulong n4 = n2 >> 2;
        double e = 2.0*M_PI/n2;
        for (ulong j=0; j<n4; j++)
        {
            double a = j * e;
            double cc1,ss1, cc3,ss3;
            sincos(a, &ss1, &cc1);

#if defined USE_SINCOS3
            sincos(3.0*a, &ss3, &cc3);
#else
            SINCOS3ALG(cc1, ss1, cc3, ss3);
#endif

            ulong ix = j;
            ulong id = (n2<<1);
            while ( ix<n-1 )
            {
                for (ulong i0=ix; i0<n; i0+=id)
                {
                    ulong i1 = i0 + n4;
                    ulong i2 = i1 + n4;
                    ulong i3 = i2 + n4;

                    double tr0,tr1;
                    SUMDIFF3(fr[i0], fr[i2], tr0);
                    SUMDIFF3(fr[i1], fr[i3], tr1);

                    double ti0,ti1;
                    SUMDIFF3(fi[i0], fi[i2], ti0);
                    SUMDIFF3(fi[i1], fi[i3], ti1);

                    double t3, t4;
                    SUMDIFF3(tr0, ti1, t3);
                    tr0 = -tr0;
                    // ===
//                    t3  =  tr0 - ti1;
//                    tr0 = -ti1 - tr0;

//                    SUMDIFF3(tr1, ti0, t4);
//                    t4 = -t4;
                    // ===
                    SUMDIFF3R(tr1, ti0, t4);

                    CMULT6(ss1, cc1, t4, tr0, fr[i2], fi[i2]);

                    CMULT6(cc3, ss3, tr1, t3, fi[i3], fr[i3]);
               }

               ix = (id<<1) - n2 + j;
               id <<= 2;
            }
        }
    }

    for (ulong ix=0, id=4;  ix<n;  id*=4)
    {
        for (ulong i0=ix; i0<n; i0+=id)
            SUMDIFF2(fr[i0], fr[i0+1]);

        ix = 2*(id-1);
    }

    for (ulong ix=0, id=4;  ix<n;  id*=4)
    {
        for (ulong i0=ix; i0<n; i0+=id)
            SUMDIFF2(fi[i0], fi[i0+1]);

        ix = 2*(id-1);
    }

}
//========================== end ===========================


void
split_radix_fft(double *fr, double *fi, ulong ldn, int is)
// fast fourier transform
// split radix algorithm
{
    if ( is<0 )  split_radix_fft_dif_core(fr, fi, ldn);
    else         split_radix_fft_dif_core(fi, fr, ldn);

    revbin_permute(fr, 1<<ldn);
    revbin_permute(fi, 1<<ldn);
}
//========================== end ===========================
