#if !defined __FXT_H
#define      __FXT_H

#include "fxttypes.h"      // ulong
#include "complextype.h"   // Complex

// for modfft() (number theoretic transform) see mod/mfft.h

// for super simple (plain-C) fft & realfft see simplfft/

//----------------------------------------------------

//<<
// General format of arguments: (double *f, ulong ldn)
// f := pointer to data array,
// ldn := base 2 log of array length (length n=2**n)
//
// A function some_func0(...) (note the '0')
// expects input data with higher half zeros.
//
// Data in the arrays without const modifier is generally modified.
//
// declarations with a comment 'aux' at the line end
// are called by other routines and might not be of
// direct use for users of the library
//>>

// jjmagictag  (needed for automatic testing)
//: ------------- HARTLEY transforms --------------------

// fht/fhtsplitradixdit.cc:
void split_radix_dit_fht_core(double *f, ulong ldn); // aux
void split_radix_dit_fht(double *f, ulong ldn);

// fht/fhtsplitradixdif.cc:
void split_radix_dif_fht_core(double *f, ulong ldn); // aux
void split_radix_dif_fht(double *f, ulong ldn);

// fht/cfhtsplitradixdit.cc:
void split_radix_dit_fht_core(Complex *f, ulong ldn); // aux
void split_radix_dit_fht(Complex *f, ulong ldn);

// fht/cfhtsplitradixdif.cc:
void split_radix_dif_fht_core(Complex *f, ulong ldn); // aux
void split_radix_dif_fht(Complex *f, ulong ldn);

// fht/fht0.cc:
void fht0(double *f, ulong ldn);

// fht/cfht0.cc:
void fht0(Complex *f, ulong ldn);

// fht/hartleyshift.cc:
void hartley_shift_05(double *a, ulong n); // aux

// fht/skipfht.cc:
void skip_fht(double *f, ulong n, ulong d, double *w);
void skip_fht0(double *f, ulong n, ulong d, double *w);

// fht/twodimfht.cc:
void row_column_fht(double *f, ulong r, ulong c); // aux
void y_transform(double *f, ulong r, ulong c); // aux
void twodim_fht(double *f, ulong r, ulong c);

// fht/twodimfhtcnvl.cc:
void twodim_fht_convolution_core(const double *f, double *g, ulong r, ulong c);

//: --------------- FOURIER transforms ------------------

//<<
// typical format of arguments: (double *fr, double *fi, ulong ldn, int is)
// fr := pointer to data array (real part),
// fi := pointer to data array (imag part),
// ldn := base 2 log of array length
// is := sign of exponent in fourier kernel

// naming (semi-) conventions:
// blabla_fft() := fft implementation with blabla-algorithm
// blabla_fft0() := fft for zero paddad data
//             (i.e. second half of the data is zero)
//>>

// fht/fhtfft.cc:
void fht_fft(double *fr, double *fi, ulong ldn, int is);
void fht_fft0(double *fr, double *fi, ulong ldn, int is);
void fht_fft_pre_processing(double *fr, double *fi, ulong ldn, int is); // aux
void fht_fft_post_processing(double *fr, double *fi, ulong ldn, int is); // aux

// fht/fhtcfft.cc:
void fht_fft(Complex *f, ulong ldn, int is);
void fht_fft0(Complex *f, ulong ldn, int is);
void fht_fft_pre_processing(Complex *f, ulong ldn, int is); // aux
void fht_fft_post_processing(Complex *f, ulong ldn, int is); // aux

// fft/fftdif4.cc:
void dif4_fft_core(double *fr, double *fi, ulong ldn); // aux
void dif4_fft_core(Complex *f, ulong ldn); // aux
void dif4_fft(double *fr, double *fi, ulong ldn, int is);

// fft/fftdit4.cc:
void dit4_fft_core(double *fr, double *fi, ulong ldn); // aux
void dit4_fft_core(Complex *f, ulong ldn); // aux
void dit4_fft(double *fr, double *fi, ulong ldn, int is);

// fft/cfftsplitradix.cc:
void split_radix_dif_fft_core(Complex *f, ulong ldn); // aux
void split_radix_dit_fft_core(Complex *f, ulong ldn); // aux
void split_radix_fft(Complex *f, ulong ldn, int is);

// fft/fftsplitradix.cc:
void split_radix_fft_dif_core(double *fr, double *fi, ulong ldn); // aux
void split_radix_fft(double *fr, double *fi, ulong ldn, int is);

// fft/cfftwrap.cc: (wrappers for type complex fourier)
void complex_to_real_imag(Complex *c, long ldn);  // aux
void real_imag_to_complex(double *fr, double *fi, long ldn); // aux
void complex_fft(Complex *c, ulong ldn, int is);
void real_imag_fft(double *fr, double *fi, ulong ldn, int is);

// fft/fouriershift.cc:
void fourier_shift(Complex *a, ulong n, double v); // aux
void fourier_shift(double *fr, double *fi, ulong n, double v); // aux
void fourier_shift0(double *fr, double *fi, ulong n, double v); // aux
void fourier_shift(double *fr, double *fi, ulong n, double v, ulong k0, ulong kn); // aux
void fourier_shift0(double *fr, double *fi, ulong n, double v, ulong k0, ulong kn); // aux

// fft/skipfft.cc:
void skip_fft(double *fr, double *fi, ulong n, ulong d, double *wr, double *wi, int is); // aux
void skip_fft0(double *fr, double *fi, ulong n, ulong d, double *wr, double *wi, int is); // aux

// fft/fft8difcore.cc:
void fft8_dif_core(Complex *f); // aux
void fft8_dif_core(double *fr, double *fi); // aux

// fft/fft8ditcore.cc:
void fft8_dit_core(Complex *f); // aux
void fft8_dit_core(double *fr, double *fi); // aux

// fft/fft9.cc:
void fft9(Complex *x); // aux
void fft9(double *xr, double *xi); // aux


//: ---------- MATRIX (aka four step) transforms --------------

// matrix/matrixfft.cc:
void matrix_fft(double *fr, double *fi, ulong ldn, int is);
void matrix_fft0(double *fr, double *fi, ulong ldn, int is);
void matrix_fft(Complex *f, ulong ldn, int is);
void matrix_fft0(Complex *f, ulong ldn, int is);

// matrix/rowffts.cc:
void row_ffts(double *fr, double *fi, ulong r, ulong c, int is); // aux
void row_weighted_ffts(double *fr, double *fi, ulong r, ulong c, int is); // aux
void row_ffts(Complex *f, ulong r, ulong c, int is); // aux
void row_weighted_ffts(Complex *f, ulong r, ulong c, int is); // aux

// matrix/rowcnvls.cc:
void row_weighted_auto_convolutions(double *fr, double *fi, ulong r, ulong c, double v); // aux
void row_weighted_auto_convolutions(Complex *f, ulong r, ulong c, double v); // aux

// matrix/columnffts.cc:
void column_ffts(double *fr, double *fi, ulong r, ulong c, int is, int zp, double *tr, double *ti); // aux
void column_ffts(Complex *f, ulong r, ulong c, int is, int zp, Complex *tmp); // aux
void column_real_complex_ffts(double *fr, ulong r, ulong c, int zp, double *t); // aux
void column_complex_real_ffts(double *fr, ulong r, ulong c, double *t); // aux
void column_complex_imag_ffts(const double *fr, double *fi, ulong r, ulong c, double *tmp); // aux

// matrix/matrixcnvla.cc:
void matrix_auto_convolution(double *fr, ulong ldn);
void matrix_auto_convolution0(double *fr, ulong ldn);
void matrix_auto_convolution(double *fr, ulong r, ulong c, int zp=0);

// matrix/matrixcnvl.cc:
void matrix_convolution(double *fr, double *fi, ulong ldn);
void matrix_convolution0(double *fr, double *fi, ulong ldn);
void matrix_convolution(double *fr, double *fi, ulong r, ulong c, int zp=0);

// matrix/matrixcocnvla.cc:
void matrix_auto_convolution(Complex *f, ulong ldn);
void matrix_auto_convolution0(Complex *f, ulong ldn);
void matrix_auto_convolution(Complex *f, ulong r, ulong c, int zp=0);
void matrix_complex_auto_convolution(double *fr, double *fi, ulong ldn);
void matrix_complex_auto_convolution0(double *fr, double *fi, ulong ldn);
void matrix_complex_auto_convolution(double *fr, double *fi, ulong r, ulong c, int zp=0);

// matrix/transpos2.cc:
void matrix_transpose2(double *x, ulong rw, ulong cl, double *tmp=0); // aux

// matrix/ctranspos2.cc:
void matrix_transpose2(Complex *x, ulong rw, ulong cl, Complex *tmp=0); // aux

//: --------------- REAL FFT ---------------------

// info about the ordering of the data is in each source file

// realfft/realfftbyfht.cc:
void fht_real_complex_fft(double *f, ulong ldn);
void fht_real_complex_fft0(double *f, ulong ldn);
void fht_complex_real_fft(double *f, ulong ldn);
void realisator(double *gr, const double *gi, ulong n, int is); // aux
void imaginator(const double *gr, double *gi, ulong n, int is); // aux

// realfft/realfftwrap.cc:
void complex_real_pre_processing(double *re, double *im, ulong nh); // aux
void real_complex_post_processing(double *re, double *im, ulong nh); // aux
void wrap_real_complex_fft(double *f, ulong ldn);
void wrap_real_complex_fft0(double *f, ulong ldn);
void wrap_complex_real_fft(double *f, ulong ldn);

// realfft/realffteasyord.cc:
void easy_ordering_real_complex_fft(double *f, ulong ldn);
void easy_ordering_real_complex_fft0(double *f, ulong ldn);
void easy_ordering_complex_real_fft(double *f, ulong ldn);

// realfft/realfftsplitradix.cc:
void split_radix_real_complex_fft(double *x, ulong ldn);
void split_radix_real_complex_fft0(double *x, ulong ldn);
void split_radix_complex_real_fft(double *x, ulong ldn);
void split_radix_real_complex_fft_dit_core(double *x, ulong ldn); // aux
void split_radix_complex_real_fft_dif_core(double *x, ulong ldn); // aux

// realfft/skiprealfft.cc:
void skip_real_complex_fft(double *fr, ulong n, ulong d, double *wr); // aux
void skip_real_complex_fft0(double *fr, ulong n, ulong d, double *wr); // aux
void skip_complex_real_fft(double *fr, ulong n, ulong d, double *wr); // aux


//: --------------- REAL CONVOLUTION ------------------

// fht/fhtcnvl.cc:
void fht_convolution(double *f, double *g, ulong ldn);
void fht_convolution0(double *f, double *g, ulong ldn);
void fht_convolution_core(double *x, double *y, ulong ldn, double v=0.0); // aux
void fht_convolution_revbin_permuted_core(double *x, double *y, ulong ldn, double v=0.0); // aux

// fht/fhtcnvla.cc:
void fht_auto_convolution(double *f, ulong ldn);
void fht_auto_convolution0(double *f, ulong ldn);
void fht_auto_convolution_core(double *x, ulong ldn, double v=0.0); // aux
void fht_auto_convolution_revbin_permuted_core(double *x, ulong ldn, double v=0.0); // aux

// fht/fhtnegacnvla.cc:
void fht_negacyclic_auto_convolution(double *x, ulong ldn, double v=0.0);
void fht_negacyclic_auto_convolution_core(double *x, ulong ldn, double v=0.0); // aux

// fft/fftcnvl.cc:
void fht_fft_convolution(double *f, double *g, ulong ldn);
void split_radix_fft_convolution(double *f, double *g, ulong ldn);
void fht_fft_convolution0(double *f, double *g, ulong ldn);
void split_radix_fft_convolution0(double *f, double *g, ulong ldn);
void fft_convolution_core1(double *x, double *y, ulong ldn, double v=0.0); // aux
void fft_convolution_core2(double *x, double *y, ulong ldn, double v=0.0); // aux

// fft/fftcnvla.cc:
void fht_fft_auto_convolution(double *f, ulong ldn);
void split_radix_fft_auto_convolution(double *f, ulong ldn);
void fht_fft_auto_convolution0(double *f, ulong ldn);
void split_radix_fft_auto_convolution0(double *f, ulong ldn);
void fft_auto_convolution_core1(double *x, ulong ldn, double v=0.0); // aux
void fft_auto_convolution_core2(double *x, ulong ldn, double v=0.0); // aux

// mult/diskcnvla.cc:
void disk_weighted_complex_auto_convolution(int fd1, int fd2,
                                            double *fr, ulong fn, ulong al,
                                            double v, double nx,
                                            int zq1=0, int zq3=0);

//: -------------- REAL CORRELATION ---------------
// fht/fhtcorr.cc:
void fht_correlation0(double *f, double *g, ulong ldn);
void fht_auto_correlation0(double *f, ulong ldn);

// fft/fftcorr.cc:
void fft_correlation0(double *f, double *g, ulong ldn);
void fft_auto_correlation0(double *f, ulong ldn);


//: ------------ COmplex COnvolution & COrrelation -----------------
// fht/cfhtcnvl.cc:
void fht_convolution(Complex *f, Complex *g, ulong ldn);
void fht_convolution0(Complex *f, Complex *g, ulong ldn);
void fht_convolution_core(Complex *x, Complex *y, ulong ldn, double v=0.0); // aux
void fht_convolution_revbin_permuted_core(Complex *x, Complex *y, ulong ldn, double v=0.0); // aux

// fht/cfhtcnvla.cc:
void fht_auto_convolution(Complex *f, ulong ldn);
void fht_auto_convolution0(Complex *f, ulong ldn);
void fht_auto_convolution_core(Complex *x, ulong ldn, double v=0.0); // aux
void fht_auto_convolution_revbin_permuted_core(Complex *x, ulong ldn, double v=0.0); // aux

// fft/fftcocnvl.cc:
void fft_auto_convolution(Complex *f, ulong ldn, double v=0.0);
void fft_convolution(Complex *f, Complex *g, ulong ldn, double v=0.0);
void fft_complex_auto_convolution(double *fr, double *fi, ulong ldn, double v=0.0);
void fft_complex_convolution(double *fr, double *fi, double *gr, double *gi, ulong ldn, double v=0.0);

// fft/fftcocorr.cc:
void fft_complex_auto_correlation(double *fr, double *fi, ulong ldn);
void fft_complex_correlation(double *fr, double *fi, double *gr, double *gi, ulong ldn);


//: --------------- SPECTRUM ------------------
// fht/fhtspect.cc:
void fht_spectrum(double *f, ulong ldn, int phasesq=0);

// fft/fftspect.cc:
void fft_spectrum(double *f, ulong ldn, int phasesq=0);


//: -------- OTHER FOURIER STUFF ------------------

// chirp/fftarblen.cc: (arbitrary length fft)
void fft_arblen(double *fr, double *fi, ulong n, int is);

// chirp/fftfract.cc: (fractional fft)
void fft_fract(double *x, double *y, ulong n, double v);

// ndimfft/twodimfft.cc:
void twodim_fft(double *fr, double *fi, ulong r, ulong c, int is);

// ndimfft/ndimfft.cc: (1...5 dim fft)
void ndim_fft(double *fr, double *fi, ulong ndim, const ulong *ldn, int is);

// chirp/makechirp.cc:
void make_fft_chirp(double *wr, double *wi, ulong n, int is); // aux
void complete_fft_chirp(double *wr, double *wi, ulong n, ulong nn); // aux
void make_fft_fract_chirp(double *wr, double *wi, double v, ulong n, ulong nn); // aux


//: ---------- COSINE/SINE TRANSFORM ----------

// dctdst/dct.cc:
//void dct(double *x, ulong ldn, double *y=0);

// dctdst/dst.cc:
void dst(double *x, ulong ldn, double *y=0);

// dctdst/dcth.cc:
void dcth(double *x, ulong ldn, double *y=0);
void idcth(double *x, ulong ldn, double *y=0);

// dctdst/dctzapata.cc:
void dct_zapata(double *x, ulong ldn, double *y=0);

// dctdst/dsth.cc:
void dsth(double *x, ulong ldn, double *y=0);
void idsth(double *x, ulong ldn, double *y=0);


//: ---------- WALSH TRANSFORM ----------

// walsh/walshdif2.cc:
void dif2_walsh_wak(double *f, ulong ldn);
void dif2_walsh_wal(double *f, ulong ldn);
void dif2_walsh_pal(double *f, ulong ldn);

// walsh/walshdit2.cc:
void dit2_walsh_wak(double *f, ulong ldn);
void dit2_walsh_wal(double *f, ulong ldn);
void dit2_walsh_pal(double *f, ulong ldn);

// walsh/walshseq.cc:
void walsh_seq(double *f, ulong ldn);
void inverse_walsh_seq(double *f, ulong ldn);

// walsh/walshcirc.cc:
void walsh_circ(double *f, ulong ldn);

// walsh/walsh0.cc:
void walsh_wak0(double *f, ulong ldn);
void walsh_wal0(double *f, ulong ldn);
void walsh_pal0(double *f, ulong ldn);

// walsh/dyadiccnvl.cc:
void dyadic_convolution(double *f, double *g, ulong ldn);
void dyadic_auto_convolution(double *f, ulong ldn);
void dyadic_convolution0(double *f, double *g, ulong ldn);
void dyadic_auto_convolution0(double *f, ulong ldn);


//: ---------- HAAR TRANSFORM ----------

// haar/haar.cc:
void haar(double *f, ulong ldn, double *ws=0);
void inverse_haar(double *f, ulong ldn, double *ws=0);

// haar/inthaar.cc:
void int_haar(double *f, ulong ldn, double *ws=0);
void inverse_int_haar(double *f, ulong ldn, double *ws=0);

// haar/haarinplace.cc:
void inplace_haar(double *f, ulong ldn);
void inverse_inplace_haar(double *f, ulong ldn);


//: ---------- WAVELET TRANSFORM ----------

//<<
// wavelet/waveletfilter.h and
// wavelet/waveletfilter.cc:
class wavelet_filter;
//>>

// wavelet/wavelet.cc:
void wavelet(double *f, ulong ldn, const wavelet_filter &wf);

// wavelet/invwavelet.cc:
void inverse_wavelet(double *f, ulong ldn, const wavelet_filter &wf);

// wavelet/harmonicwavelet.cc:
void harmonic_wavelet(double *fr, double *fi, ulong ldn);
void inverse_harmonic_wavelet(double *fr, double *fi, ulong ldn);


//: ---------- WEIGHTED TRANSFORM ----------

// weighted/weightedfft.cc:
void weighted_fft(double *fr, double *fi, ulong ldn, int is, double w);
void weighted_inverse_fft(double *fr, double *fi, ulong ldn, int is, double w);

// weighted/weightedconv.cc:
void weighted_complex_auto_convolution(double *fr, double *fi, ulong ldn, double w, double v=0.0);
void negacyclic_complex_auto_convolution(double *fr, double *fi, ulong ldn, double v=0.0);
void right_angle_complex_auto_convolution(double *fr, double *fi, ulong ldn, double v=0.0);
void weighted_complex_auto_convolution(Complex *f, ulong ldn, double w, double v=0.0);
void negacyclic_complex_auto_convolution(Complex *f, ulong ldn, double v=0.0);
void right_angle_complex_auto_convolution(Complex *f, ulong ldn, double v=0.0);


//: ------------- LEARNERS STRAIGHT & SIMPLE VERSIONS: ------------
// even simpler (plain C) stuff is in simplfft/


// learn/recfftdit2.cc:
void recursive_dit2_fft(Complex *a, ulong ldn, int is);

// learn/recfftdif2.cc:
void recursive_dif2_fft(Complex *a, ulong ldn, int is);

// learn/fftdif2.cc:
void dif2_fft_localized(Complex *f, ulong ldn, int is);
void dif2_fft(Complex *f, ulong ldn, int is);
void dif2_fft(double *fr, double *fi, ulong ldn, int is);

// learn/fftdit2.cc:
void dit2_fft_localized(Complex *f, ulong ldn, int is);
void dit2_fft(Complex *f, ulong ldn, int is);
void dit2_fft(double *fr, double *fi, ulong ldn, int is);

// learn/fftdif4l.cc:
void dif4l_fft(Complex *f, ulong ldn, int is);

// learn/fftdit4l.cc:
void dit4l_fft(Complex *f, ulong ldn, int is);

// learn/fhtdit2.cc:
void dit2_fht_localized(double *f, ulong ldn);

// learn/fhtdif2.cc:
void dif2_fht_localized(double *f, ulong ldn);

// learn/recfhtdit2.cc:
void recursive_dit2_fht(double *a, ulong ldn);

// learn/recfhtdif2.cc:
void recursive_dif2_fht(double *a, ulong ldn);


//: ---------- SLOW TRANSFORMS (mostly for testing) ----------

// slow/slowht.cc:
void slow_ht(double *f, ulong n);

// slow/slowtwodimht.cc:
void slow_twodim_ht(double *f, ulong r, ulong c);

// slow/slowrowcolht.cc:
void slow_row_column_ht(double *f, ulong r, ulong c);

// slow/slowtwodimft.cc:
void slow_twodim_ft(double *fr, double *fi, ulong r, ulong c, int is);

// slow/slowcnvl.cc:
void slow_convolution(const double *f, const double *g, double *r, ulong nu);
void slow_convolution0(const double *f, const double *g, double *r, ulong nu);
void slow_auto_convolution(const double *f, double *r, ulong nu);
void slow_auto_convolution0(const double *f, double *r, ulong nu);

// slow/slowtwodimcnvl.cc:
void slow_twodim_convolution(const double *f, double *g, ulong ru, ulong cu);

// slow/slowcorr.cc:
void slow_correlation(const double *w, double *f, ulong nu);
void slow_correlation0(const double *w, double *f, ulong nu);
void slow_auto_correlation(double *f, ulong nu);
void slow_auto_correlation0(double *f, ulong nu);

// slow/slowcocorr.cc:
void slow_complex_correlation(const double *wr, const double *wi,
                              double *fr, double *fi, ulong n);
void slow_complex_auto_correlation(double *fr, double *fi, ulong n);

// slow/slowcocnvl.cc:
void slow_convolution(const Complex *w , Complex *f, ulong nu);
void slow_complex_convolution(const double *wr, const double *wi,
                              double *fr, double *fi, ulong n);
void slow_complex_auto_convolution(double *fr, double *fi, ulong n);
void slow_auto_convolution(Complex *f, ulong nu);


// slow/slowft.cc:
void slow_ft(double *fr, double *fi, ulong n, int is);
void slow_rotated_ft(double *fr, double *fi, ulong n, int is, int r);
void slow_ft(Complex *f, long n, int is);

// slow/slowfracft.cc:
void slow_fract_ft(double *fr, double *fi, ulong n, double v);
void slow_fract_ft(Complex *f, ulong n, double v);

// slow/slowwalsh.cc:
void dit2_walsh_wak_slow(double *f, ulong ldn);
void dif2_walsh_wak_slow(double *f, ulong ldn);

// slow/slowzt.cc:
void slow_zt(double *f, ulong n, double z);

//-------------------------------------------------
// jjmagictag  (needed for automatic testing)

//: ------- inlines give default implementations -------
inline void fft(double *fr, double *fi, ulong ldn, int is)
{ fht_fft(fr,fi,ldn,is); }

inline void fft(Complex *f, ulong ldn, int is)
{ fht_fft(f,ldn,is); }

inline void fht(double *f, ulong ldn)
{ split_radix_dit_fht(f, ldn); }

inline void fht(Complex *f, ulong ldn)
{ split_radix_dit_fht(f, ldn); }

inline void walsh_wal(double *f, ulong ldn)
{ dif2_walsh_wal(f, ldn); }

inline void walsh_pal(double *f, ulong ldn)
{ dif2_walsh_pal(f, ldn); }

inline void walsh_wak(double *f, ulong ldn)
{ dif2_walsh_wak(f, ldn); }

//-------------------------------------------------


#endif // !defined __FXT_H
