
#include "fxttypes.h"
#include "fxtmult.h"  // diskio
#include "jjassert.h"
#include "diskaux.h"

#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <unistd.h>  // EBADF etc
#include <stdio.h>
#include <errno.h>

#include <iostream.h>

#include <sys/time.h>


#define  PRINTRW  0

#if  ( PRINTRW==1 )
#warning  'FYI: verbose disk read/write'
#endif

static timeval t0, t1;
static double timediff(timeval &x, timeval &y);


static void
tell_lseek_err()
//
// check errno and print info if set
//
{
    if ( errno )  cerr << "\n errno=" << errno << " == ";
    {
        switch ( errno )
        {
        case EBADF:  cerr << "EBADF" << endl;  break;
        case ESPIPE:  cerr << "ESPIPE" << endl;  break;
        case EINVAL:  cerr << "EINVAL" << endl;  break;
        default:  cerr << "UNKNOWN ERROR" << endl;
        }
        cerr << "see 'man 2 read' for explanation." << endl;
    }
}
//--------------------

static void
tell_read_err()
//
// check errno and print info if set
//
{
    if ( errno )
    {
        cerr << "\n errno=" << errno << " == ";
        switch ( errno )
        {
        case EINTR:  cerr << "EINTR" << endl;  break;
        case EAGAIN:  cerr << "EAGAIN" << endl;  break;
        case EIO:  cerr << "EIO" << endl;  break;
        case EISDIR:  cerr << "EISDIR" << endl;  break;
        case EBADF:  cerr << "EBADF" << endl;  break;
        case EINVAL:  cerr << "EINVAL" << endl;  break;
        case EFAULT:  cerr << "EFAULT" << endl;  break;
        default:  cerr << "UNKNOWN ERROR" << endl;
        }
        cerr << "see 'man 2 read' for explanation." << endl;
    }
}
//--------------------


void
seek_read(int fd, ulong off, double *f, ulong fn)
//
// read doubles f[0..fn-1]
// from file with filedescriptor fd
// start in file after (double-)offset off
//
{
#if  ( PRINTRW==1 )
    cerr << "seek_read():"
         << "  d_off=" << off
         << "  d_len="<< fn
         << "  d_end="<< off+fn-1
         << "  c_end="<< (off+fn)*sizeof(double)-1
         << endl;
#endif

    v_seek_read(fd, off*sizeof(double), (void *)f, fn*sizeof(double));
}
// ================== end ===================


void
v_seek_read(int fd, ulong off, void *f, ulong fn)
//
// from file with filedescriptor fd
// read fn bytes into f[]
// start in file after (byte-)offset off
// called by seek_read()
//
{
    gettimeofday(&t0,0);
    ulong w;

    w = lseek(fd,off,SEEK_SET);
    fxtmult::ndiskseeks += 1;
    if ( errno )
    {
        tell_lseek_err();
        jjassert2( 0, "lseek for file failed");
    }

    w = read(fd,f,fn);
    if ( errno )
    {
        tell_read_err();
        jjassert2( 0, "read from file failed" );
    }

    gettimeofday(&t1,0);
    fxtmult::diskiotime += timediff(t0,t1);
    fxtmult::diskio += fn;
}
// ================== end ===================


static void
tell_write_err()
//
// check errno and print info if set
//
{
    if ( errno )
    {
        cerr << "\n errno=" << errno << " == ";
        switch ( errno )
        {
        case EBADF:   cerr << "EBADF" << endl;   break;
        case EINVAL:  cerr << "EINVAL" << endl;  break;
        case EFAULT:  cerr << "EFAULT" << endl;  break;
        case EPIPE:   cerr << "EPIPE" << endl;   break;
        case EAGAIN:  cerr << "EAGAIN" << endl;  break;
        case EINTR:   cerr << "EINTR" << endl;   break;
        case ENOSPC:  cerr << "ENOSPC" << endl;  break;
        case EIO:     cerr << "EIO" << endl;     break;
        default:      cerr << "UNKNOWN ERROR" << endl;
        }
        cerr << "see man 2 write for explanation." << endl;
    }
}
//--------------------

void
seek_write(int fd, ulong off, double *f, ulong fn)
//
// write doubles from f[0..fn-1]
// to file with filedescriptor fd
// start in file after (double-)offset off
//
{
#if  ( PRINTRW==1 )
    cerr << "seek_write():"
         << "  doff=" << off
         << "  dlen="<< fn
         << "  d_end="<< off+fn-1
         << "  c_end="<< (off+fn)*sizeof(double)-1
         << endl;
#endif

    v_seek_write(fd, off*sizeof(double), (void *)f, fn*sizeof(double));
}
// ================== end ===================


void
v_seek_write(int fd, ulong off, void *f, ulong fn)
//
// to file with filedescriptor fd
// write fn bytes from f[]
// start in file after (byte-)offset off
// called by seek_write()
//
{
    gettimeofday(&t0,0);
    ulong w;

    w = lseek(fd,off,SEEK_SET);
    fxtmult::ndiskseeks += 1;
    if ( errno )
    {
        tell_lseek_err();
        jjassert2( 0, "lseek for file failed");
    }

    w = write(fd,f,fn);
//    w = TEMP_FAILURE_RETRY (write (fd, f, fn));
    if ( errno )
    {
        tell_write_err();
        jjassert2( 0, "write to file failed" );
    }

    gettimeofday(&t1,0);
    fxtmult::diskiotime += timediff(t0,t1);
    fxtmult::diskio += fn;
}
// ================== end ===================



void
open_err(const char *fn, int *fd)
{
    int flag = O_RDWR | O_CREAT;
//    int flag = O_RDWR | O_CREAT | O_TRUNC;  // debug
    
    *fd = open(fn, flag, 0600);
    jjassert2( -1!=*fd, "error opening file" );
}
// ================== end ===================

void
close_err(int fd)
{
    ulong w;
    w = close(fd);
    jjassert2( 0==w, "close of file failed");
}
// ================== end ===================



static double
timediff(timeval &x, timeval &y)
{
    static const double mio1 = 1.0/1000000.0;
    double tx = x.tv_sec + x.tv_usec * mio1;
    double ty = y.tv_sec + y.tv_usec * mio1;
    return  ty-tx;
}
// ================== end ===================
