
#include "hfdatafu.h"
#include "shift.h"  // shift_right()
#include "scan.h"  // first_ne_idx()

#include "jjassert.h"
#include "dtparanoia.h"


int
dt_div_sh(hfdata &a, ulong u, ulong *rem)
//
// c = a/b
// rem = a%b
// return = amount of left shift that occured
//
{
#ifdef  DT_PARANOIA
    jjassert( a.OK() );
#endif

    jjassert2( u, "division by zero" );

    int ret;
    if ( u <= MAX_LIMB )
    {
	ret = i_div_sh(a.dig(), a.prec(), u, hfdata::rx, rem); 
    }
    else
    {
#if defined HAVE_IDIVSEMI
        ret = i_div_semi(a.dig(), a.prec(), u, hfdata::rx, rem);
#else
        jjassert2( 0, "not implemented: division by int>MAX_LIMB" );  // jjnote: do it
#endif // defined HAVE_IDIVSEMI
    }

#ifdef  DT_PARANOIA
    jjassert( a.OK() );
#endif

    return ret;
}
//============== end ====================



int
i_div_sh(LIMB *cd, ulong cn, ulong b, const uint rx, ulong *r)
//
// return= amount of left shift that occured
//
{
    jjassert( b<= MAX_LIMB );

    ulong k;
    ulong zz, rem = 0;
    for (k=0; k<cn; ++k)
    {
        zz  = cd[k];
	zz += rx * rem;
        cd[k] = zz / b;
        rem   = zz % b;

        // is not faster:
        /* 
        zz  =  cd[k];
	zz +=  rem * rx;
        cd[k] = (rem=(zz/b));
        rem  = zz - b*rem;
        */
    }

    int f = first_ne_idx(cd,cn,(LIMB)0);

    if ( f )   // normalize 
    {
        shift_left(cd,cn,f);
	
	k = f;
	do
	{
	    zz = rx*rem;
	    cd[cn-k] = zz / b;
	    rem      = zz % b;
	}
	while ( --k );
    }

    if ( r )  *r = rem;

    return f;
}
//================== end ========================



#if defined HAVE_IDIVSEMI

#undef  UINTLL
#define UINTLL unsigned long long

int
i_div_semi(LIMB *cd, ulong cn, ulong b, const uint rx, ulong *r) // =0
//
// return= amount of left shift that occured
//
{
    ulong k;
    UINTLL zz, rem = 0;
    for (k=0; k<cn; ++k)
    {
        /*
        zz  =  (UINTLL)cd[k];
	zz +=  rx*rem;
        cd[k] = (LIMB)(zz/b);
        rem  = zz%b;
        */

        // faster:
        zz  =  (UINTLL)cd[k];
	zz +=  (rem*rx);
        cd[k] = (LIMB)(rem=(zz/b));
        rem  = zz-b*rem;
    }


    int f = first_ne_idx(cd,cn,(LIMB)0);

    if ( f )   // normalize 
    {
        shift_left(cd,cn,f);
	
	k = f;
	do
	{
	    zz = rx*rem;
	    cd[cn-k] = (LIMB)(zz/b);
	    rem = zz%b;
	}
	while ( --k );
    }

    if ( r )  *r = (ulong)rem;

    return f;
}
//================== end ========================
#endif // defined HAVE_IDIVSEMI

