
#include <assert.h>

#include "fxt.h"
#include "fxtdefs.h"


#define  FFT(fr,fi,ldn,is)   fht_fft(fr,fi,ldn,is)

// tuning parameter:
#define  USE_SPLRX  0  // whether split radix dif/dit is used (default=0)
#if ( USE_SPLRX==1 )
#warning 'FYI: complex_(auto)_convolution() use split radix ffts'
#define  DIT_FFT_CORE  split_radix_fft_dit_core
#define  DIF_FFT_CORE  split_radix_fft_dif_core
#else
#warning 'FYI: complex_(auto)_convolution() use radix 4 ffts'
#define  DIT_FFT_CORE  dit4_fft_core
#define  DIF_FFT_CORE  dif4_fft_core
#endif


void
fft_auto_convolution(Complex *f, ulong ldn,
                     double v/*=0.0*/)
{
    const ulong n = (1<<ldn);

    DIF_FFT_CORE(f, ldn);
    if ( v==0.0 )  v = 1.0/n;
    for (ulong i=0; i<n; ++i)
    {
        Complex t = f[i];
        t *= t;
        f[i] = t * v;
    }
    DIT_FFT_CORE(f, ldn);
}
//=========================== end =================


void
fft_convolution(Complex *f, Complex *g,
                ulong ldn, double v/*=0.0*/)
//
// _cyclic_ convolution
// (use zero padded data for usual conv.)
//
// f, g  must not overlap
//
// result in g
//
{
    const ulong n = (1<<ldn);

    assert( f!=g );
//    assert( f+n<=g || g+n<=f );

    DIF_FFT_CORE(f,ldn);
    DIF_FFT_CORE(g,ldn);
    if ( v==0.0 )  v = 1.0/n;
    for (ulong i=0; i<n; ++i)
    {
        Complex t = g[i] * f[i];
        g[i] = t * v;
    }
    DIT_FFT_CORE(g, ldn);
}
//=========================== end =================


void
fft_complex_auto_convolution(double *fr, double *fi,
                             ulong ldn, double v/*=0.0*/)
//
// _cyclic_ (self-)convolution
// (use zero padded data for usual conv.)
//
// fr,fi must not overlap
//
{
    const int is = 1;
    const ulong n = (1<<ldn);

    FFT(fr,fi,ldn,is);

    if ( v==0.0 )  v = 1.0/n;
    for (ulong k=0; k<n; ++k)
    {
        double tr = fr[k];
        double ti = fi[k];
        CSQR2(tr,ti);
        fr[k] = tr * v;
        fi[k] = ti * v;
    }

    FFT(fr,fi,ldn,-is);
}
//=========================== end =================


void
fft_complex_convolution(double *fr, double *fi,
                        double *gr, double *gi,
                        ulong ldn, double v/*=0.0*/)
//
// _cyclic_ convolution
// (use zero padded data for usual conv.)
//
// fr,fi,gr,gi must be pairwise non-overlapping
//
// result in gr,gi
//
{
    assert( (fr!=gr) && (fr!=gi) );
    assert( (fi!=gr) && (fi!=gi) );

    const int is = 1;
    const ulong n = (1<<ldn);

    FFT(fr,fi,ldn,is);
    FFT(gr,gi,ldn,is);

    if ( v==0.0 )  v = 1.0/n;
    for (ulong k=0; k<n; ++k)
    {
        double tr = fr[k];
        double ti = fi[k];
        CMULT(gr[k], gi[k], tr, ti);
        gr[k] = tr * v;
        gi[k] = ti * v;
        
        CMULT(fr[k],fi[k],gr[k],gi[k]);
    }

    FFT(gr,gi,ldn,-is);
}
//=========================== end =================
