
#include <math.h>

#include "fxt.h"
#include "fxtdefs.h"  // SUMDIFF, CSQR, CMULT
#include "evenoddpermute.h"
#include "sincos.h"


#define FFT(re,im,ldn,is)    fht_fft(re,im,ldn,is)
#define FFT0(re,im,ldn,is)   fht_fft0(re,im,ldn,is)


void
complex_real_pre_processing(double *re, double *im, ulong nh)
// preprocessing to use a complex length nh fft 
// as complex-to-real fft of length 2*nh
{
    const ulong n4 = (nh>>1);
    const double ph0 = -1.0*M_PI/nh;

    for (ulong i=1,j=nh-1; i<=n4; ++i,--j)
    {
        double evr, evi, odr, odi;
        SUMDIFF4(re[i], re[j], evr, odr);
        SUMDIFF4(im[i], im[j], odi, evi);

        double c, s;
        sincos(ph0*i, &s, &c);
        c = -c;
        CMULT(s, c, odr, odi);
        odi = -odi;
        
        SUMDIFF4(evr, odr, re[j], re[i]);
        SUMDIFF4(odi, evi, im[i], im[j]);
    }

    SUMDIFF2(re[0], im[0]);
}
// =============== end ===========


void
real_complex_post_processing(double *re, double *im, ulong nh)
// postprocessing to use a complex length nh fft 
// as real-to-complex fft of length 2*nh
{
    const ulong n4 = (nh>>1);
    const double ph0 = +1.0*M_PI/nh;

    for (ulong i=1,j=nh-1; i<=n4; ++i,--j)
    {
        double evr, evi, odr, odi;
        SUMDIFF4(re[i], re[j], evr, odr);
        SUMDIFF4(im[i], im[j], odi, evi);

        double c, s;
        sincos(ph0*i, &s, &c);
        c = -c;
        CMULT(s, c, odr, odi);

        SUMDIFF4_05(evr, odr, re[i], re[j]);
        SUMDIFF4_05(odi, evi, im[i], im[j]);
    }

    SUMDIFF2(re[0], im[0]);
}
// =============== end ===========


void
wrap_real_complex_fft(double *f, ulong ldn)
//
// isign = +1
//
// ordering on output:
// f[0]     = re[0]
// f[1]     = re[1]
//         ...
// f[n/2-1] = re[n/2-1]
// f[n/2]   = re[n/2](==nyquist freq)
//
// f[n/2+1] = im[1]  (wrt. compl fft with isign=+1)
// f[n/2+2] = im[2]
//         ...
// f[n-1]   = im[n/2-1]
//
// corresponding real and imag parts (with the exception of
// zero and nyquist freq) are found in f[i] and f[n/2+i]
//
// note that the order of imaginary parts
// is reversed wrt. fht_real_complex_fft()
//
{
    const ulong n = (1<<ldn);

    if ( n<=1 )  return;

    const ulong nh = (n>>1);

    double *re = f;
    double *im = f+nh;

    // re gets even indexed elements, im gets odd indexed:
    inverse_evenodd_permute(f,n);

    FFT(re,im,ldn-1,+1);

    real_complex_post_processing(re,im,nh);
}
// =============== end ===========



void
wrap_real_complex_fft0(double *f, ulong ldn)
//
// isign = +1
// version for zero padded data
//
{
    const ulong n = (1<<ldn);

    if ( n<=1 )  return;

    const ulong nh = (n>>1);
    const ulong n4 = (n>>2);

    double *re = f;
    double *im = f+nh;

    // re gets even indexed elements, im gets odd indexed:
    for (ulong i=0,j=1; i<n4; i++,j+=2)
    {
        im[i] = re[j];
        re[i] = re[j-1];
    }

    for (ulong i=n4; i<nh; i++)  re[i] = 0.0;  // im[i] already zero

    FFT0(re,im,ldn-1,+1);

    real_complex_post_processing(re,im,nh);
}
// =============== end ===========



void
wrap_complex_real_fft(double *f, ulong ldn)
//
// isign = +1
//
// ordering on input:
// like the output of wrap_real_complex_fft()
//
{
    const ulong n = (1<<ldn);

    if ( n<=1 )  return;

    const ulong nh = (n>>1);

    double *re = f;
    double *im = f+nh;

    complex_real_pre_processing(re,im,nh);

    FFT(re,im,ldn-1,-1);

    evenodd_permute(f,n);
}
//=========== end ===========

