
#include "hfdatafu.h"
#include "auxtempl.h"  // copy(), shift_right(), ...
#include "workspace.h"

#include "jjassert.h"
#include "dtparanoia.h"

//#define  USE_WORKSPACE


int
dt_add(const hfdata &a, const hfdata &b, ulong dx, hfdata &c, ulong cp)
// c[k] = a[k+dx]+b[k]
// returns last carry (increment for exponent)  (0 or 1)
{
#ifdef  DT_PARANOIA
    jjassert( a.OK() );
    jjassert( b.OK() );
#endif

    const uint rx = hfdata::rx;
    const LIMB * const ad = a.dig();
    const LIMB * const bd = b.dig();
    LIMB * const cd = c.dig();
    const ulong ap = a.prec();
    const ulong bp = b.prec();
//    const ulong cp = c.prec();

    if ( dx>=cp )
    {
        copy(ad,ap,cd,cp);
        return 0;
    }

#if defined USE_WORKSPACE
#warning "FYI: dt_add() uses workspace"
    LIMB *rd;
    ulong rp = max(max(ap,bp),cp);
    workspace::get_ws(rd,rp);
    copy(ad,ap,rd,rp);
    int i = i_add_to(rd,rp,dx,bd,bp,rx);
    copy(rd,rp,cd,cp);
    workspace::let_ws();

#else   // USE_WORKSPACE
    int   i;
    if ( ad==bd )  // a,a,c
    {
        copy(ad,ap,cd,cp);  // c = a
        i = i_add_to(cd, cp, dx, bd, bp, rx);  // c+=b
        // ... or: c= 2*a
    }
    else // ad!=bd
    {
        if ( ad==cd )  // a,b,a
        {
            i = i_add_to(cd, cp, dx, bd, bp, rx);  // c+=b
        }
        else // ad!=cd
        {
            if ( bd!=cd )  // a,b,c
            {
                copy(ad,ap,cd,cp);  // c=a
                i = i_add_to(cd, cp, dx, bd, bp, rx);  // c+=b
            }
            else  // a,b,b
            {
                shift_right(cd,cp,dx);
                i = i_add_to(cd,min(cp,ap),ad,rx);
            }
        }
    }
#endif   // USE_WORKSPACE

    if ( i )
    {
        shift_right(cd,cp,1);
        cd[0] = i;
    }

#ifdef  DT_PARANOIA
    jjassert( c.OK() );
#endif

    return i;
}
//===================== end =====================



uint
i_add_to(LIMB *ad, ulong an, ulong dx, const LIMB *bd, ulong bn, const uint rx)
// c[k] = a[k+dx]+b[k]
// returns last carry (0 or 1)
{
    if ( dx>=an )  return 0;  // no overlap at all

    uint cy = i_add_to(ad+dx, min(an-dx,bn), bd, rx);  // add overlapping region
    if ( (dx>0) && cy )  cy = i_carry(ad, dx, cy, rx);  // carry thru nonoverlapping region
    return cy;
}
//===================== end =====================


uint
i_carry(LIMB *d, ulong n, uint cy, const uint rx)
// d[k] += cy(k)  k = n-1...0
{
    for (long k=n-1; (k>=0) && (cy!=0) ; --k)
    {
        uint rk = d[k]+cy;

        if ( rk>=rx )
        {
	    rk -= rx;
            cy = 1;
        }
        else
        {
            cy = 0;
        }

	d[k] = (LIMB)rk;
    }

    return cy;
}
//===================== end =====================


uint
i_add_to(LIMB *ad, ulong n, const LIMB *bd, const uint rx)
// ad[k] += bd[k]   k = n-1...0
// returns last carry (0 or 1)
{
    uint cy = 0;
    for (long k=n-1; k>=0; --k)
    {
        uint rk = ad[k]+bd[k]+cy;

        if ( rk>=rx )
        {
	    rk -= rx;
            cy = 1;
        }
        else
        {
            cy = 0;
        }

	ad[k] = (LIMB)rk;
    }

    return cy;
}
//===================== end =====================
