
#include <math.h>
#include <iostream.h>
#include <stdlib.h>  // RAND_MAX, rand()

#include "hfdata.h"
#include "hfverbosity.h"
#include "auxtempl.h"  // copy(), shift_right(), ...
#include "auxbit.h"  // ld()

#include "jjassert.h"


ulong   hfdata::nbytes = 0;
ulong   hfdata::nbytesmax = 0;
long    hfdata::ndata = 0;

// meaningless values to make sure initialisation works:
ulong  hfdata::rx = 0;       // 10000;
ulong  hfdata::ldrx = 0;     // 12
ulong  hfdata::rxbits = 0;   // 13
double hfdata::lg10rx = -1.0; // 4
int    hfdata::rx2pw = 0;     // 0
//ulong  hfdata::rxdig = 0;     // 4
char*  hfdata::rxfmt = "";    // "%04d";
ulong  hfdata::rxbase = 0;    // 10
ulong  hfdata::rxbspw = 0;    // 4
void (* hfdata::mulcnvl)(double *, double *, ulong) = NULL;
void (* hfdata::sqrcnvl)(double *, ulong) = NULL;

const char * const hfdata::classname = "hfdata";

#define  OVERWRITE_MEM  0  // 0 (default) or 1 (paranoia)
#if  ( OVERWRITE_MEM!=0 )
#warning  'FYI: OVERWRITE_MEM activated'
#endif

hfdata::hfdata(ulong n)
    : limbs_(0),
      size_(0),
      prec_(0)
{
    if ( hfverbosity::printdtcd )
        cout << "\n   hfdata(size_=" << n << ") ";

    size(n);

    if ( hfverbosity::printdtcd )
        cout << "  @ " << (void *)limbs_ << endl;

    ndata++;
    jjassert( ndata>0 );
}
//--------------


hfdata::~hfdata()
{
    if ( hfverbosity::printdtcd )
        cout << "\n   ~hfdata() " << endl;
    
    delete [] limbs_;
    ndata--;
    jjassert( ndata>=0 );
}
//--------------


void
hfdata::size(ulong n)
{
    LIMB *oldd = limbs_;
    LIMB *newd = 0;
    ulong osz = size_;

    if ( n!=0 )
    {
//        newd = new LIMB [n];
        newd = (LIMB *)operator new(n * sizeof(LIMB));
        nbytes += (n * sizeof(LIMB));

        if ( oldd )  ::copy(oldd, osz, newd, n);
#if  ( OVERWRITE_MEM!=0 )
        else         null(newd, n);  // debug-oid
#endif
    }
    if ( oldd )
    {
#if  ( OVERWRITE_MEM!=0 )
        null(oldd,osz);  // debug
#endif
//        delete [] oldd;
        operator delete(oldd);

        if ( hfverbosity::printresize )
            cout << "  hfdata::" << __FUNCTION__ << "(): "
                 << (void *)oldd << " [" << osz << "]  --> "
                 << (void *)newd << " [" << n << "]" << endl;

        nbytes -= (osz * sizeof(LIMB));
    }
    

    limbs_ = newd;
    size_ = n;
    prec_ = n;

    if ( nbytes>nbytesmax )  nbytesmax = nbytes;
}
//--------------


void
hfdata::prec(ulong p)
{
    if ( p<=size_ )  prec_ = p;
    else             size(p);
}
//--------------


void
hfdata::copy(const hfdata &h)
{
    if ( hfverbosity::printcopy )
        cout << "\n   hfdata::copy(const hfdata &) " << endl;
    
    if ( this==&h )  return; // nothing to do

    if ( hfverbosity::printcopy )  
        cout << "\n   hfdata::copy(const hfdata &) "
             << "calling copy()" << endl;

    ::copy(h.dig(), h.prec(), limbs_, prec_);
}
//--------------


static void
i_rand(LIMB *a, ulong n, int nine)
{
//    srand(time(0));

    int rn = (RAND_MAX/nine);

    while ( n-- )  a[n] = (LIMB)((int)rand()/rn);
}
//--------------
// used for:

void
hfdata::rand(ulong n1, ulong n2)
{
    ulong mi = min(n1,n2);
    mi = min(size_,mi);

    ulong ma = max(n1,n2);
    ma = min(size_,ma);

    i_rand(limbs_+mi, ma-mi, radix()-1);
}
//--------------


void
hfdata::dump(char *bla/*=0*/, ulong n/*=0*/) const
{
    if ( n==0 )  n = prec_;
    if ( n>size_ )  n = size_;

    if ( bla )  cout << bla << "   @"<< this << endl;

    cout << "   size_="<< size_
         << "   prec_=" << prec_
         << "   radix()="<< radix()
         << endl;

    cout << "   DIGIT[]= "<< (void *)limbs_
         << " ... " << (void *)(limbs_ + size_ - 1)
         << ":  " << endl;

    for (ulong k=0; k<n; ++k)  cout << limbs_[k] << "," << ((k+1)&15?"":"\n");
    cout << endl;
    cout << endl;
}
//--------------



char *
get_radix_format(int r)
{
    static char fmt[22];
    strcpy(fmt,"%05d,");
    //          01234

    fmt[2] = '0' + (char)ceil(log(r)/log(10));

    if ( 10 == hfdata::rxbase )
    {
        fmt[2] = '0'+ (char)(hfdata::rxbspw);
        fmt[4] = 0;
    }

    if ( 16 == hfdata::rxbase )  
    {
        fmt[2] = '0'+ (char)(hfdata::rxbspw);
        fmt[3] = 'x';
        fmt[4] = 0;
    }

    return fmt;
}
//--------------


inline int
ispowof(long x, int p)
// returns 0 or k so that p^k==x
{
    int k=0;
    while( x>1 && x==(x/p)*p )  { x/=p; k++; }
    return (x==1?k:0);
}
//--------------
// used for ...

void // static
hfdata::radix(int r, char *rfmt/*=0*/)
{
    jjassert2(r>=2, " hfdata::setradix(): radix must be >=2 " );

    jjassert2( r<=(1<<(8*sizeof(LIMB))),
               " hfdata::setradix(): radix doesn't fit into LIMB " );

    rx = r;

    ldrx = ld(rx);
    lg10rx = log(rx) * 1.0/log(10.0);

    rxbase = rx;
    rxbspw = 1;

    int p;
    p = ispowof(rx,10);
    if ( p )
    {
        rxbase = 10;
        rxbspw = p;
    }

    p = ispowof(rx,16);
    if ( p )
    {
        rxbase = 16;
        rxbspw = p;
    }

    if ( !rfmt ) rxfmt = get_radix_format(rx);
    else         rxfmt = rfmt;

    if ( rx==((uint)1<<ldrx) )
    {
        rx2pw = 1;
	rxbits = ldrx;
    }
    else
    {
        rx2pw = 0;
	rxbits = ldrx+1;
    }
}
//--------------

#include <unistd.h> // pause()

#define  WARNIF(x)  if ( x ) { cerr << "\nWARNING: " << #x << endl;  errq=1; } 
int
hfdata::check() const
{
    int errq = 0;

    WARNIF( !limbs_ );
    WARNIF( !size_ );

    WARNIF( ndata<0 );

    WARNIF( rx<0 );
    WARNIF( rx>65536 );

    for (ulong k=0; k<prec_; ++k)
    {
        if ( limbs_[k] >= rx )
        {
            cerr << "\nWARNING: limbs_[" << k << "] == " << limbs_[k] << " > rx"<< endl;
            errq = 1;
            pause();
            jjassert( limbs_[k]<rx );
        }
    }

    return errq;
}
//--------------



void // static
hfdata::print_statistics(ulong n)
{
    cout << " hfdata: bytes currently allocated: "
         << hfdata::nbytes << endl;

    cout << " hfdata: max bytes allocated: "
         << hfdata::nbytesmax << " (plus workspace)" << endl;
}
//--------------
