
#include "fxttypes.h"
#include "complextype.h"
#include "sincos.h"

#include <math.h>


void
fourier_shift(Complex *a, ulong n, double v)
//
// a[k] *= exp(k*v*sqrt(-1)*2*pi/n)
// k = 0...n-1
{
    if ( v==0.0 )  return;  // nothing to do

    const double phi0 = v*2.0*M_PI/n;
    for (ulong k=1; k<n; ++k) // k==0 skipped
    {
        double s,c;
        sincos(phi0*k, &s, &c);
        a[k] *= Complex(c,s);
    }
}
// =========== end  ===========


void
fourier_shift(double *fr, double *fi, ulong n, double v)
//
// (fr[k],fi[k]) *= exp(k*v*sqrt(-1)*2*pi/n)
// k = 0...n-1
{
    if ( v==0.0 )  return;  // nothing to do

    const double phi0 = v*2.0*M_PI/n;
    for (ulong k=1; k<n; ++k) // k==0 skipped
    {
        double s,c;
        sincos(phi0*k, &s, &c);
        Complex t =  Complex(fr[k],fi[k])*Complex(c,s);
        fr[k] = t.real();
        fi[k] = t.imag();
    }
}
// =========== end  ===========


void
fourier_shift0(double *fr, double *fi, ulong n, double v)
//
// (fr[k],fi[k]) *= exp(k*v*sqrt(-1)*2*pi/n)
// k = 0...n-1
// assume fi[] is zero
{
    if ( v==0.0 )  return;  // nothing to do

    const double phi0 = v*2.0*M_PI/n;
    for (ulong k=1; k<n; ++k) // k==0 skipped
    {
        double s,c;
        sincos(phi0*k, &s, &c);
        double t =  fr[k];
        fr[k] = c * t;
        fi[k] = s * t;
    }
}
// =========== end  ===========


void
fourier_shift(double *fr, double *fi, ulong n, double v, ulong k0, ulong kn)
//
// (fr[k],fi[k]) *= exp((k0+k)*v*sqrt(-1)*2*pi/n)
// k = 0...kn-1
{
    if ( v==0.0 )  return;  // nothing to do

    const double phi0 = v*2.0*M_PI/n;
    for (ulong k=0; k<kn; ++k)
    {
        double s,c;
        sincos(phi0*(k0+k), &s, &c);
        Complex t =  Complex(fr[k],fi[k])*Complex(c,s);
        fr[k] = t.real();
        fi[k] = t.imag();
    }
}
// =========== end  ===========


void
fourier_shift0(double *fr, double *fi, ulong n, double v, ulong k0, ulong kn)
//
// (fr[k],fi[k]) *= exp((k0+k)*v*sqrt(-1)*2*pi/n)
// k = 0...kn-1
// assume fi[] is zero
{
    if ( v==0.0 )  return;  // nothing to do

    const double phi0 = v*2.0*M_PI/n;
    for (ulong k=0; k<kn; ++k)
    {
        double s,c;
        sincos(phi0*(k0+k), &s, &c);
        double t =  fr[k];
        fr[k] = c * t;
        fi[k] = s * t;
    }
}
// =========== end  ===========
