#if !defined __MOD_H
#define      __MOD_H

class ostream;
class istream;

#include "mtypes.h"
#include "factor.h"


#define  USE_MODM  0  // 0 or 1, set to 0


class mod_init
// the class 'mod_init' has the sole purpose
// of initialising the class 'mod'
{
public:
    mod_init(umod_t m, umod_t *primes=0);
    ~mod_init();
};
//--------------------------


class mod
// class to implement a modular type
// and arithmetic operations on it
{
    friend class mod_init;

public:

    mod() : x(0)  { ; }
    mod(const mod &m) : x(m.x)  { ; }
    mod(const umod_t i) : x(i) { if ( i>=modulus )  x -= modulus; }
    ~mod()  { ; }

    static void reinit(umod_t m, factorization *mf=0);

    inline mod & operator = (const mod &h)  { x = h.x;  return *this; }
    inline mod & operator = (umod_t i)  { (*this) = mod(i);  return *this; }
    inline mod & operator = (uint i)  { (*this) = mod(i);  return *this; }
#if  ( USE_64BIT_MOD_T )
    inline mod & operator = (ulong i)  { (*this) = mod(i);  return *this; }
#endif

    friend inline mod & operator ++ (mod &z);
    friend inline mod & operator -- (mod &z);

    friend inline mod & operator += (mod &z, const mod &h);
    friend inline mod & operator -= (mod &z, const mod &h);
    friend inline mod & operator *= (mod &z, const mod &h);
    friend inline mod & operator /= (mod &z, const mod &h);

    friend inline const mod operator + (const mod &h1, const mod &h2);
    friend inline const mod operator - (const mod &h1, const mod &h2);
    friend inline const mod operator * (const mod &h1, const mod &h2);
    friend inline const mod operator / (const mod &h1, const mod &h2);

    friend inline const int operator == (const mod &h1, const mod &h2);
    friend inline const int operator != (const mod &h1, const mod &h2);
    friend inline const int operator < (const mod &h1, const mod &h2);
    friend inline const int operator <= (const mod &h1, const mod &h2);
    friend inline const int operator > (const mod &h1, const mod &h2);
    friend inline const int operator >= (const mod &h1, const mod &h2);

    static mod root2pow(int ldorder);  // root of order 2^ldorder
    static int modulus_prime();
    static int modulus_cyclic();

    static void print_info();


    // --- IMPLEMENTATION:
    umod_t x;


    // --- STATIC:
    static umod_t modulus;        // 0 <= x < modulus 
    static umod_t mm1;            // modulus - 1 
    static factorization modfact; // factorization of modulus

    static double  mbitsd;        // log_2(modulus)
    static uint    mbits;         // ceil(mbitsd) = how many bits modulus uses
    static ldouble m1dd;          // 1.0/modulus (needed for multiplication)

    static umod_t maxorder;       // maximal order of elements
    static factorization xfact;   // factorization of maxorder

    static uint    max2pow;       // root of order 2**ldn exists for ldn<=max2pow 
    // ==> fft length max 2^(max2pow) if restricted to powers of 2

    static umod_t phi;            // phi(modulus)
    // == modulus-1 for prime modulus
    static factorization phifact;   // factorization of phi

    static mod  zero;             // = 0
    static mod  one;              // = 1
    static mod  two;              // = 2
    static mod  minus_one;        // = -1 == mod(modulus-1)
    static mod  maxordelem;       // element of maximal order
    static mod  invmaxordelem;    // inverse of above

private:
    static mod * root_2pow; //[max2pow+1];      // element[k] is of order 2^k
    static mod * invroot_2pow; //[max2pow+1];   // element[k] is of order 2^-k


    // magic initialiser:
    // (MUST be initialized after all other static stuff!)
    // cf. modinit.cc
    static mod_init *mod_initializer;
};
//---------------------------------------------

#if  ( USE_MODM==1 )
#warning "FYI: mod.h uses slow funcs from arithmodm.h"
#include "modm.h"
#define mod_m(x)    mod_mod(x,mod::modulus)
#define incr_m(x)   incr_mod(x,mod::modulus)
#define decr_m(x)   decr_mod(x,mod::modulus)
#define add_m(x,y,z)  z=add_mod(x,y,mod::modulus)
#define sub_m(x,y,z)  z=sub_mod(x,y,mod::modulus)
#define mul_m(x,y,z)  z=mul_mod(x,y,mod::modulus)
#else
#include "modarith.h"
#endif // ( USE_MODM==1 )


//inline mod::mod & operator = (const mod &h)  { x = h.x;  return *this; }
//inline mod::mod & operator = (umod_t i)  { (*this) = mod(i);  return *this; }
//inline mod::mod & operator = (uint i)  { (*this) = mod(i);  return *this; }
//#if  ( USE_64BIT_MOD_T )
//inline mod::mod & operator = (ulong i)  { (*this) = mod(i);  return *this; }
//#endif


inline mod & operator ++ (mod &z)
{ if ( z.x==mod::mm1 ) z.x=0;  else z.x++;  return z; }

inline mod & operator -- (mod &z)
{ if ( z.x==0 ) z.x=mod::mm1;  else z.x--;  return z; };


inline mod & operator += (mod &z, const mod &h)
{ z.x = add_m(z.x,h.x);  return z; }

inline mod & operator -= (mod &z, const mod &h)
{ z.x = sub_m(z.x,h.x);  return z; }

inline mod & operator *= (mod &z, const mod &h)
{ z.x = mul_m(z.x,h.x);  return z; }

inline mod operator + (const mod &h)  { return h; }

inline mod operator - (const mod &h)
{ mod n;  n.x = (mod::modulus)-h.x;  return n; }

inline mod & operator /= (mod &z, const mod &h)
{ mod inv(const mod&);  z = inv(h);  return z; }


inline const mod operator + (const mod &h1, const mod &h2)
{ mod z(h1); z += h2; return z; }

inline const mod operator - (const mod &h1, const mod &h2)
{ mod z(h1); z -= h2; return z; }

inline const mod operator * (const mod &h1, const mod &h2)
{ mod z(h1); z *= h2; return z; }

inline const mod operator / (const mod &h1, const mod &h2)
{ mod inv(const mod&); mod h3 = inv(h2); h3 *= h1; return h3; }


inline const int operator == (const mod &h1, const mod &h2)
{ return  h1.x == h2.x; }

inline const int operator != (const mod &h1, const mod &h2)
{ return  h1.x != h2.x; }

inline const int operator < (const mod &h1, const mod &h2)
{ return  h1.x < h2.x; }

inline const int operator <= (const mod &h1, const mod &h2)
{ return  h1.x <= h2.x; }

inline const int operator > (const mod &h1, const mod &h2)
{ return  h1.x > h2.x; }

inline const int operator >= (const mod &h1, const mod &h2)
{ return  h1.x >= h2.x; }


istream&  operator >> (istream& is, mod& h);
ostream&  operator << (ostream& os, const mod& h);


// mod/modfuncs.cc:
umod_t order(mod x);
mod div(const mod& m1, const mod& m2);
mod inv(const mod& m);
mod pow(const mod& a, umod_t ex);
mod invpow(const mod& a, umod_t ex);
mod root(umod_t);
mod invroot(umod_t);


// mod/order.cc:
umod_t order_mod(umod_t x, umod_t m, const factorization &phifact);


// mod/maxorder.cc:
umod_t maxorder_mod(const factorization &mf);
umod_t maxorder_element_mod(const factorization &mf, const factorization &pf);


// mod/sqrtmod.cc:
//umod_t sqrt(umod_t a);
int is_quadratic_residue(umod_t a, const factorization &mf);
umod_t sqrt_modp(umod_t a, umod_t p);
umod_t sqrt_modpp(umod_t a, umod_t p, long ex);
umod_t sqrt_modf(umod_t a, const factorization &mf);


inline void add(const mod& m1, const mod& m2, mod& r)
{ r.x = add_m(m1.x,m2.x); }

inline void sub(const mod& m1, const mod& m2, mod& r)
{ r.x = sub_m(m1.x,m2.x); }

inline void mul(const mod& m1, const mod& m2, mod& r)
{ r.x = mul_m(m1.x,m2.x); }

inline void sqr(const mod& m, mod& r)
{ r.x = mul_m(m.x,m.x); }


// mod/modaux.cc:
void copy(const mod *src, mod *dst, ulong n);
//void fill(mod *dst, ulong n, mod src);
void multiply(mod *f, ulong n, mod m);
void print(char *bla, const mod *f, ulong n);
ulong diff_print(mod *f, mod *g, ulong n, int supreq=1);
ulong diff_print(double *f, double *g, ulong n);
ulong diff(mod *src, mod *dst, ulong n);
void rand(mod *f, ulong n, umod_t m);
void rand(double *f, ulong n, umod_t m);

// jj_end_autodoc

// mod/modinfo.cc:
void mod_info0();
void mod_info1();
void mod_info2();
void mod_info3();
void mod_info4();
void mod_info99();



#endif  // !defined __MOD_H
